/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QMetaObject>
#include <QMetaClassInfo>

#ifdef __SYMBIAN32__
#include <string.h>
#endif // __SYMBIAN32__

#include "signalforwarder.h"
#include "bindingutility.h"
#include "servicebase.h"
#include "servicefactory.h"
#include "sfwerrorhandler.h"
#include "serviceobjectbinding.h"
#include "serviceframeworkdefs.h"

// CONSTANTS
const int KMaxFunctionSig = 255;

namespace WRT
{
static const char KAddEventListener[] = "addEventListener";
static const char KRemoveEventListenser[] = "removeEventListener";
static const char KGetInterface[] = "getInterface";
static const char KGetLastErrCode[] = "getLastErrCode";
static const char KGetLastErrDescription[] = "getLastErrDescription";

/*!
    \class ServiceObjectBinding
    \brief The ServiceObjectBinding class is used to make a Service Object scriptable.

   It enables script language like JavaScript to call public slots in a Service Object
   dymamically like Qt applcation does. It also allows script language to connect callback
   function to the service object signal.

   The object of ServiceObjectBinding is a scriptable object of NetScape plguin.
*/


/*!
    Constructs an object of the ServiceObjectBinding.

    @param npp represents a single instance of a plugin
*/
ServiceObjectBinding::ServiceObjectBinding(NPP npp) :
    ObjectBinding(npp),
    m_connectedCounter(0),
    m_signals(NULL),
    m_base(NULL),
    m_content(NULL)
{
    m_addEventListener = NPN_GetStringIdentifier(KAddEventListener);
    m_removeEventListener = NPN_GetStringIdentifier(KRemoveEventListenser);
    m_getInterface = NPN_GetStringIdentifier(KGetInterface);
    m_getLastErrCode = NPN_GetStringIdentifier(KGetLastErrCode);
    m_getLastErrDescription = NPN_GetStringIdentifier(KGetLastErrDescription);
}

/*!
    Destructor
*/
ServiceObjectBinding::~ServiceObjectBinding()
{
    qDeleteAll(m_hash);
    m_hash.clear();

    NPN_MemFree(m_signals);
    m_signalMethodOffset.clear();
#if !defined(QT_MOBILITY_SFW)
    if (m_base) {
        m_base->release();
    }
#endif
}

/*!
    Check if a method exists in the ServiceObjectBinding object given \a methodId

    A ServiceObjectBinding object has the following methods:
    (1) addEventListener
    (2) removeEventListener
    (3) all of public slots in the service object

    @param methodId method identifier
    @return true if the method exists, otherwise false

    @see HasProperty
*/
bool ServiceObjectBinding::HasMethod(NPIdentifier methodId)
{
    bool ret = false;
    if (m_object) {
        if (methodId == m_addEventListener || methodId == m_removeEventListener ||
           methodId == m_getInterface || methodId == m_getLastErrCode ||
           methodId == m_getLastErrDescription) {
            ret = true;
        }
        else {
            CLEAR_ERRORS(this);
            ret = ObjectBinding::HasMethod(methodId);
        }
    }
    return ret;
}

/*!
    Check if a property exists in the ServiceObjectBinding object given \a propertyId

    The list of properties contains all of proerties and signals in the service object

    @param propertyId property identifier
    @return true if the property exists, otherwise false

    @see GetProperty SetProperty
*/
bool ServiceObjectBinding::HasProperty(NPIdentifier propertyId)
{
    bool ret(false);
    for ( int i=0; i<m_signalCount; ++i ) {
        if ( propertyId == m_signals[i] ) {
            ret = true;
            break;
        }
    }
    if (!ret) {
        ret = ObjectBinding::HasProperty(propertyId);
    }
    return ret;
}

/*!
    Get the value of a property of the this object given \a propertyId

    @param propertyId property identifier
    @param[out] result  property value
    @return true if succeed in getting value of the property, otherwise false

    @see HasProperty SetProperty
*/
bool ServiceObjectBinding::GetProperty(NPIdentifier propertyId,
                                       NPVariant* result)
{
    bool ret(false);
    ret = ObjectBinding::GetProperty(propertyId, result);
    return ret;
}

/*!
    Set a property of this object given \a propertyId and \a value

    It is used to connect script callback function to a signal in the service object

    @param propertyId property identifier
    @param value property value
    @return true if succeed in setting the property, otherwise false

    @see HasProperty GetProperty
*/
bool ServiceObjectBinding::SetProperty(NPIdentifier propertyId,
                                      const NPVariant* value)
{
    bool ret(false);
    for ( int i=0; i<m_signalCount; ++i ) {
        if ( propertyId == m_signals[i] ) {
            if ( connectCallback(value,m_signalMethodOffset[i]) > 0 ) {
                ret = true;
            }
            break;
        }
    }
    if (!ret) {
        ret = ObjectBinding::SetProperty(propertyId, value);
    }
    return ret;
}

/*!
    Dispatch a script call

    The method can be:
    (1) addEventListener
    (2) removeEventListener
    (3) all of public slots in the service object

    @param methodId the identifier of the method
    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call success, otherwise false

    @see ObjectBinding::Invoke HasMethod
*/
bool ServiceObjectBinding::Invoke(NPIdentifier methodId, const NPVariant *args,
                                  uint32_t argCount, NPVariant *result)
{
    bool bRet(true);
    //clear error code and error message before next method invoke

    if (!m_object) {
        bRet = false;
        NULL_TO_NPVARIANT(*result);
    }
    else {
        if ( methodId == m_addEventListener ) {
            CLEAR_ERRORS(this);
            if (argCount == 2) {
                if (NPVARIANT_IS_STRING(args[0]) && NPVARIANT_IS_OBJECT(args[1])) {
                    const NPString* signal = &NPVARIANT_TO_STRING( args[0] );
                    int connected = connect( signal, &args[1] );
                    INT32_TO_NPVARIANT( connected, *result );
                } else {
                    NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                       ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
                }
            } else {
                NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS,
                        ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS]);
            }
        }
        else if ( methodId == m_removeEventListener ) {
            CLEAR_ERRORS(this);
            if ( argCount == 1 ) {
                int id(0);
                if ( NPVARIANT_IS_INT32(args[0]) ) {
                    id = NPVARIANT_TO_INT32( args[0] );
                }
                else if ( NPVARIANT_IS_DOUBLE( args[0]) ) {
                    id = NPVARIANT_TO_DOUBLE( args[0] );
                }
                if ( id ) {
                    bool dc = disconnect( id );
                    BOOLEAN_TO_NPVARIANT(dc, *result );
                } else {
                    NULL_TO_NPVARIANT(*result);
                    NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                       ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
                }
            } else {
                NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS,
                        ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS]);
            }

        }
        else if (methodId == m_getInterface ) {
            CLEAR_ERRORS(this);
            if (2 == argCount) {
                if (NPVARIANT_IS_STRING(args[0]) && NPVARIANT_IS_STRING(args[1])) {
                    const NPString *interfaceName = &NPVARIANT_TO_STRING(args[0]);
                    const NPString *interfaceVersion = &NPVARIANT_TO_STRING(args[1]);
                    ServiceInterfaceInfo interfaceInfo;
                    const char* namePtr(NULL);
                    const char* versionPtr(NULL);
                    QByteArray name = BindingUtility::NPStringToQByteArray(interfaceName);
                    namePtr = name.constData();
                    QByteArray version = BindingUtility::NPStringToQByteArray(interfaceVersion);
                    versionPtr = version.constData();
                    QByteArray name_ver(namePtr);
                    name_ver.append(KSlash);
                    name_ver.append(versionPtr);
                    interfaceInfo.m_iid = name_ver.constData();
#if defined(QT_MOBILITY_SFW)
                    QObject* base(NULL);
                    base = m_base;
#else
                    IServiceBase* base(NULL);
                    m_base->getInterface(interfaceInfo, &base);
#endif
                    if (base) {
                        NPObject *pObj = m_factory->createBindingObject(
                                         GET_NPOBJECT_CLASS(ServiceObjectBinding));
                        if (pObj) {
                            static_cast<ServiceObjectBinding*>(pObj)->setInterface(
                                    base,
                                    *m_bindingUtility);
                            OBJECT_TO_NPVARIANT(pObj, *result);
#if !defined(QT_MOBILITY_SFW)
                            base->release(); // ownership passed to service object binding
#endif
                        }
                        else {
                            bRet = false;
                            NULL_TO_NPVARIANT(*result);
                        }
                    } else {
                        NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_INTERFACE_NOT_FOUND,
                                         ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_INTERFACE_NOT_FOUND]);
                        NULL_TO_NPVARIANT(*result);
                    }
                } else {
                    NULL_TO_NPVARIANT(*result);
                    NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                       ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
                }
            } else {
                NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS,
                        ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS]);
            }
        }
        else if (methodId == m_getLastErrCode) {
            int errCode = this->getLastErrCode();
            INT32_TO_NPVARIANT(errCode, *result );
        }
        else if (methodId == m_getLastErrDescription) {
            QByteArray ba = this->getLastErrDescription().toUtf8();
            int size = ba.size();
            m_content = (char*) NPN_MemAlloc(size + 1);
            strcpy(m_content, ba.data());
            m_content[size] = '\0';
            STRINGZ_TO_NPVARIANT(m_content, *result);
        }
        else {
            CLEAR_ERRORS(this);
            if (!ObjectBinding::Invoke(methodId, args, argCount, result)) {
                NULL_TO_NPVARIANT(*result);
            }
        }
    }
    return bRet;
}

/*!
    Initialize interface pointer

    It creates identifiers for all of signals in the service object

    @param aInterface interface pointer, ownership transferred
    @param bindingUtility binding utility object
*/
#if defined(QT_MOBILITY_SFW)
void ServiceObjectBinding::setInterface(QObject* aInterface,
                                        BindingUtility& bindingUtility)
#else
void ServiceObjectBinding::setInterface(IServiceBase* aInterface,
                                        BindingUtility& bindingUtility)
#endif
{
#if !defined(QT_MOBILITY_SFW)
    if (m_base) {
        m_base->release();
    }
#endif
    m_base = aInterface;
#if !defined(QT_MOBILITY_SFW)
    m_base->addRef();
#endif
#if defined(QT_MOBILITY_SFW)
    setObject(m_base, bindingUtility, false);
#else
    setObject(m_base->getServiceObject(), bindingUtility, false);
#endif
    if ( m_object ) {
        constructSignalIdentifiers();
    }
}

/*!
    Creates identifiers for all of signals in the service objects
*/
void ServiceObjectBinding::constructSignalIdentifiers()
{
    // Iterate through the list of meta-methods and create identifiers for all signals
    //
    m_signalCount=0;

    const QMetaObject *metaObject = m_object->metaObject();

    // Look for number of signals
    int c( metaObject->methodCount());
    for (int i=0; i<c; ++i) {
        QMetaMethod method = metaObject->method(i);
        if ( method.methodType() == QMetaMethod::Signal )  {
            m_signalMethodOffset.append(i);
            m_signalCount++;
        }
    }
    // Allocate String identifiers
    m_signals = (NPIdentifier*) NPN_MemAlloc(sizeof(NPIdentifier)*m_signalCount);

    // Create the string array consisting of all signals
    // No method name should be more than 255 chars?
    int item(0);
    char* name = (NPUTF8*) NPN_MemAlloc( sizeof(NPUTF8)*KMaxFunctionSig );
    for (int i=0; i<m_signalCount; ++i) {
        QMetaMethod method = metaObject->method(m_signalMethodOffset[i]);
        // Strip signal arguments
        const char* sig = method.signature();
        strncpy( name, sig, KMaxFunctionSig );
        char* token = strtok(name,"(");
        strcpy( name, token );

        m_signals[i] = NPN_GetStringIdentifier(name);
        item++;
    }
    NPN_MemFree(name);
}

/*!
Connect callback to a signal

@param callback callback function
@param signalIndex index of the signal
@return id of the connection, or -1 if failed
*/
int ServiceObjectBinding::connectCallback(const NPVariant* callback, int signalIndex)
{
    int returnId(-1);
    // Connect the signal here
    // all writeable properties are names of callback functions
    if (NPVARIANT_IS_OBJECT(*callback)) {
        // Get the metamethod
        const QMetaObject *metaObject = m_object->metaObject();
        QMetaMethod mmeth = metaObject->method(signalIndex);
        QList<QByteArray> params = mmeth.parameterTypes();

        // Save javascript object, JS callback function and parameters' types of
        // service object into forwarder object
        NPObject* callbackNPObj = NPVARIANT_TO_OBJECT(*callback);
        SignalForwarder* sfobj = new SignalForwarder(m_Npp,callbackNPObj,NULL,
                                                     params,*m_bindingUtility,
                                                     ++m_connectedCounter);

        //our object will ignore index, we just need to make sure it passes QObject check
        //only one slot in the forwarder object, slot index is not needed
        int dummyindex = metaObject->methodCount();

        // connect service object signal by index to forwarder slot (only one defined)
        bool ret = QMetaObject::connect(m_object, signalIndex, sfobj, dummyindex);

        if (ret) {
            // Insert into connected signals list
            m_hash.insertMulti(QString(mmeth.signature()),sfobj);
            returnId = m_connectedCounter;
        }
        else  {
            delete sfobj;
        }
    }
    return returnId;
}

/*!
Connect callback to a signal

@param signalSignature signal signature
@param callback callback function
@return id of the connection , or -1 if failed
*/
int ServiceObjectBinding::connect(const NPString* signalSignature,
                                  const NPVariant* callback)
{
    int ret( -1 );
    // Look for the callback from the list of meta methods
    //
    int index = indexOfSignal( signalSignature );

    // Do the rest of the connection
    if ( index != -1 ) {
        ret = connectCallback( callback, index );
    } else {
        NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_SIGNAL_NOT_FOUND,
                         ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_SIGNAL_NOT_FOUND]);
    }
    return ret;
}

/*!
disconnect callback from the signal

@param id identifier of the connection
@return true succeeded, otherwise false
*/
bool ServiceObjectBinding::disconnect(int id)
{
    bool ret( false );

    QHash<QString, SignalForwarder*>::iterator it = m_hash.begin();
    while ( it != m_hash.end() ) {
        SignalForwarder* forward = it.value();
        if ( forward->id() == id ) {
            m_object->disconnect(forward);
            it = m_hash.erase(it);
            if (forward->callbackState()==SignalForwarder::InProgress) {
                // callback in progress, set state, the object will be
                // deleted later after callback completed.
                forward->setCallbackState(SignalForwarder::DisConnected);
            } else {
                delete forward;
            }
            ret = true;
            break;
        } else {
            ++it;
        }
    }
    if (!ret) {
        QString idStr;
        NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_CONNECTIONID_NOT_FOUND,
                         ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_CONNECTIONID_NOT_FOUND]);
    }
    return ret;
}

}
// END OF FILE
