/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QMetaObject>
#include <QMetaClassInfo>
#include <QVector>

#ifdef __SYMBIAN32__
#include <string.h>
#endif // __SYMBIAN32__

#include "sfwlog.h"
#include "bindingutility.h"
#include "servicefactory.h"
#include "sfwexception.h"
#include "objectbinding.h"
#include "serviceobjectbinding.h"

namespace WRT
{
static const char KQObject[] = "QObject*";
static const char KQVariant[] = "QVariant";
static const int MIN_INT32_VAL = -2147483647-1;//-1 * ((MAXDWORD - 1 ) / 2 + 1);
static const int MAX_INT32_VAL = 2147483647;//(MAXDWORD - 1 ) / 2 ;

//-------------------------------------------------------------------------------------------------
// found the offset to the slots in object. excluding slots in the base class
static int metaOffset(const QMetaObject *metaObject, MetaOffset offsetType)
{
    // Qt magic from Qt browser solution.
    // qtbrowserplugin\src\qtbrowserplugin.cpp. ask Qt team
    int offset(0);
    int classInfoIndex = metaObject->indexOfClassInfo("ToSuperClass");
    if (-1 != classInfoIndex) {
        // there are slots in the base class,
        // we need to skip the slots and return the offset to slots in top-level class
        QByteArray ToSuperClass = metaObject->classInfo(classInfoIndex).value();
        int offset = offsetType == MetaProperty ? metaObject->propertyOffset()
                : metaObject->methodOffset();

        while (ToSuperClass != metaObject->className()) {
            metaObject = metaObject->superClass();
            if (!metaObject)
                break;
            offset -= offsetType == MetaProperty ? metaObject->propertyCount()
                    : metaObject->methodCount();
        }
    }
    return offset;
}

/*!
    \class ObjectBinding
    \brief The ObjectBinding class is used to make a QObject scriptable.

   It enables script language like JavaScript to call public slots in a QObject
   dymamically like Qt applcation does. The object of ObjectBinding is a scriptable
   object of NetScape plguin.
*/

/*!
    Constructs an object of the ObjectBinding.

    @param npp represents a single instance of a plugin
*/
ObjectBinding::ObjectBinding(NPP npp) :
    NPScriptableObjectBase(npp),
    m_object(NULL),
    m_factory(NULL),
    m_owned(false)
{
    SFW_FUNC_EX("ObjectBinding::ObjectBinding");
}

/*!
    Destroys embedded QObject
*/
ObjectBinding::~ObjectBinding()
{
    SFW_FUNC_EX("ObjectBinding::~ObjectBinding");
    if (m_owned) {
        delete m_object;
    }
    if (m_factory) {
        m_factory->removeBindingObject(this);
        NPN_ReleaseObject(m_factory);
    }
}

/*!
    Check if a method exists in the QObject given \a methodId

    @param methodId method identifier
    @return true if the method exists, otherwise false

    @see HasProperty
*/
bool ObjectBinding::HasMethod(NPIdentifier methodId)
{
    NPUTF8 *slotName=NPN_UTF8FromIdentifier(methodId);
    int index = indexOfSlot(slotName);
    NPN_MemFree(slotName);
    return (index != -1);
}

/*!
    Check if a property exists in the QObject given \a propertyId

    @param propertyId property identifier
    @return true if the property exists, otherwise false

    @see GetProperty  SetProperty
*/
bool ObjectBinding::HasProperty(NPIdentifier propertyId)
{
    //check the property of QObject
    Q_ASSERT(m_object);
    bool ret(false);
    NPUTF8* propName = NPN_UTF8FromIdentifier(propertyId);
    QVariant value = m_object->property(propName);
    if (value.type() != QVariant::Invalid) {
        m_metaOffsetType = MetaProperty;
        ret = true;
        //clear previous errors
        CLEAR_ERRORS(this);
    } else {
        m_metaOffsetType = MetaMethod;
    }
    NPN_MemFree(propName);
    return ret;
}

/*!
    Get the value of a property of the QObject given \a propertyId

    @param propertyId property identifier
    @param[out] result  property value
    @return true if succeed in getting value of the property, otherwise false

    @see HasProperty SetProperty
*/
bool ObjectBinding::GetProperty(NPIdentifier propertyId, NPVariant* result)
{
    Q_ASSERT(m_object);
    bool ret(false);
    QVariant propertyValue;
    NPUTF8* propName = NPN_UTF8FromIdentifier(propertyId);
    propertyValue = m_object->property(propName);
    *result = m_bindingUtility->fromQVariant(propertyValue);
    if (!NPVARIANT_IS_VOID(*result)) {
        ret = true;
    }
    NPN_MemFree(propName);
    return ret;
}

/*!
    Set a property of the QObject given \a propertyId and \a value

    @param propertyId property identifier
    @param value property value
    @return true if succeed in setting the property, otherwise false

    @see HasProperty GetProperty
*/
bool ObjectBinding::SetProperty(NPIdentifier propertyId, const NPVariant* value)
{
    //set the property of QObject
    Q_ASSERT(m_object);
    bool ret(false);
    NPUTF8* propName = NPN_UTF8FromIdentifier(propertyId);
    try {
        ret = m_object->setProperty(propName,
                                    m_bindingUtility->toQVariant(*value));
    } catch (SFWException *e) {
         //catch exceptions raised by service provider in setting a property
        NPN_SETEXCEPTION(this, e->getErrCode(), e->getErrDescription().toLatin1());
        delete e;
        e = NULL;
    }
    NPN_MemFree(propName);
    return ret;
}

/*!
    Dispatch a script call to the public method in the the QObject

    @param methodId the identifier of the method
    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call success, otherwise false
*/
bool ObjectBinding::Invoke(NPIdentifier methodId,
                           const NPVariant *args,
                           uint32_t argCount,
                           NPVariant *result)
{
    bool bRet(true);
    if (!m_object) { // underlying object is not ready
        NULL_TO_NPVARIANT(*result);
        bRet = false;
    } else {
        //prepare the arguments for qt_metacall later
        QVector<QVariant> variants(argCount); // keep data alive from JS
        QVector<const void*> metacallArgs(argCount+1); // arguments for qt_metacall
        bool bPrepareParams = false;
        try {
            bPrepareParams = prepareParameters(args, static_cast<int>(argCount),
                                                    variants, metacallArgs);
        } catch (SFWException *e) {
            bRet = false;
            NPN_SETEXCEPTION(this, e->getErrCode(), e->getErrDescription().toLatin1());
            delete e;
            e = NULL;
        }

        if (bPrepareParams) {
            // get slot/method name from id
            NPUTF8 *slotName=NPN_UTF8FromIdentifier(methodId);
            QMetaMethod slot;

            // Search public slots in service object
            int slotIndex = indexOfSlot(slotName, &variants);
            if (slotIndex == -1) {
                bRet = false;
            }
            else {
                // Get slot
                slot = m_object->metaObject()->method(slotIndex);
            }
            if (bRet) {
                //Prepare variables back to web app
                //keep the return type of the slot, leave return value as 0.
                QVariant returnVariant(QVariant::nameToType(slot.typeName()), (void*)0);
                //qt_metacall require the first item to be return variant
                metacallArgs[0] = returnVariant.data(); // args[0] == return value
                NPN_MemFree(slotName);

                // qt_metacall will invoke the actual slot in the service object by index
                try {
                    m_object->qt_metacall(QMetaObject::InvokeMetaMethod, slotIndex,
                                          const_cast<void**>(metacallArgs.data()));
                } catch (SFWException *e) {
                    bRet = false;
                    NPN_SETEXCEPTION(this, e->getErrCode(), e->getErrDescription().toLatin1());
                    delete e;
                    e = NULL;
                }

                if (bRet && returnVariant.isValid() && result) {
                    *result = m_bindingUtility->fromQVariant(returnVariant);
                    if (result->type != NPVariantType_Null) {
                        bRet = true;
                    } else if (QString(KQObject) == slot.typeName()) { // QObject*
                        int* objaddr = reinterpret_cast<int*>(returnVariant.data());
                        QObject* object = reinterpret_cast<QObject*>(*objaddr);
                        // check for null pointer, if null, set return npvariant to null
                        if (object) {
                            NPObject *pObj = m_factory->createBindingObject(GET_NPOBJECT_CLASS(ServiceObjectBinding));
                            if (!pObj) {
                                delete object;
                                NULL_TO_NPVARIANT(*result);
                                bRet = false;
                            } else {
                                static_cast<ServiceObjectBinding*>(pObj)->setObject(object,*m_bindingUtility, true);
                                static_cast<ServiceObjectBinding*>(pObj)->constructSignalIdentifiers();
                                OBJECT_TO_NPVARIANT(pObj, *result);
                                bRet = true;
                            }
                        } else {
                            NULL_TO_NPVARIANT(*result);
                        }
                    } else {
                         bRet = false;
                    }
                }
                // Check for condition that the return value is void
                else if (slot.typeName() == QString("")) {
                    bRet = true;
                }
            } // end if (bRet)
        } else { // if (bPrepareParams)
            bRet = false;
            NULL_TO_NPVARIANT(*result);
        }
    }
    return bRet;
}

/*!
    Initialize for the QObject \a aObj

    @param aObj pointer to a QObject
    @param bindingUtility binding utility object
    @param takeQObject true take the ownership of the \a aObj
                       false not take the ownership
*/
void ObjectBinding::setObject(QObject* aObj,
                              BindingUtility& bindingUtility,
                              bool takeQObject)
{
    Q_ASSERT(!m_object);
    m_object = aObj;
    m_owned = takeQObject;
    m_bindingUtility = &bindingUtility;
}

/*!
    Set or reset the factory object

    @param factory pointer to factory object or NULL
*/
void ObjectBinding::setFactory(ServiceFactory* factory)
{
    m_factory = factory;
    //factory object should be the last one to be deleted.
    if (m_factory) {
        NPN_RetainObject(m_factory);
    }
}

/*!
    Search the signal in the QObject given \a signalSignature

    @param signalSignature signature of the signal
    @return the index of the signal in the Qt meta object, -1 returned if not found

    @see indexOfSlot
*/
int ObjectBinding::indexOfSignal(const NPString* signalSignature )
    {
#ifdef __SYMBIAN32__
    // TRIM string to be the correct length, add null character to the end
    char* s = (char*)NPN_MemAlloc( sizeof(char) * signalSignature->UTF8Length+1 );
    strncpy(s,signalSignature->UTF8Characters,signalSignature->UTF8Length);
    s[signalSignature->UTF8Length] = '\0';
    QByteArray theSignal = QMetaObject::normalizedSignature(s);
    NPN_MemFree(s);
#else
    // remove trivial staff e.g. const, ...
    QByteArray theSignal = QMetaObject::normalizedSignature(
                                                QByteArray(NPSTRING_UTF8(*signalSignature),
                                                NPSTRING_LENGTH(*signalSignature)));
#endif // __SYMBIAN32__

        // Found index to the signal
        return m_object->metaObject()->indexOfSignal(theSignal.data());
}

/*!
    Search the slot in the QObject given \a slotName

    @param slotName the name of slot
    @param vaiants variant value from JS script, if not Null, check the type
    @return the index of the slot in the Qt meta object, -1 returned if not found

    @see indexOfSignal
*/
int ObjectBinding::indexOfSlot(const QByteArray& slotName,
                               QVector<QVariant>* variants/*= NULL*/)
{
    Q_ASSERT(m_object);
    int index(-1);
    const QMetaObject *metaObject = m_object->metaObject();
    //Walk through all of slots/signals in top level class
    for (int slotIndex = metaOffset(metaObject, m_metaOffsetType);
         slotIndex < metaObject->methodCount(); ++slotIndex) {
        const QMetaMethod slot = metaObject->method(slotIndex);
        // excluding non-public methods and signal
        if (slot.access() != QMetaMethod::Public
            || slot.methodType() == QMetaMethod::Signal)
            continue;
        // signature of the slot name and parameters. e.g. "function(int, string)"
        QByteArray signature = slot.signature();
        QList<QByteArray> parameterTypes = slot.parameterTypes();
        // extract the slot name and compare with input name
        if (signature.left(signature.indexOf('(')) == slotName) {
            bool match(true);
            if (variants) {
                match = matchParameters(parameterTypes, *variants);
            } // ignore check type, match is true
            if (match) {
                index = slotIndex;
                break;
            }
        }
    }
    //clear errors if found method with requested signature or slot is a property
    if (index != -1 || m_metaOffsetType == MetaProperty){
        CLEAR_ERRORS(this);
    } else {
        if (this->getLastErrCode() == 0) {
                NPN_SETEXCEPTION(this, ObjectBinding::SFW_ERROR_SLOT_NOT_FOUND,
                                 ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_SLOT_NOT_FOUND]);
        } else {
            NPN_SETEXCEPTION(this, this->getLastErrCode(), this->getLastErrDescription().toLatin1());
        }
    }
    return index;
}

/*!
    Prepare parameters for meta_call

    @param args the list of NP arguments to be converted
    @param argCount number of the arguments
    @param variants the list of QVariant converted from \a args
    @param metacallArgs the argument list for metadata call
    @return true if all NP arguments can be converted to QVariant vars, false otherwise

*/
bool ObjectBinding::prepareParameters(const NPVariant *args,
                                      int argCount,
                                      QVector<QVariant>& variants,
                                      QVector<const void*>& metacallArgs)
{
    bool bRet(true);
    for (int i = 0; i < argCount; ++i) {
        // convert a argument value of JS into Qvariant
        QVariant qvar;
        qvar = m_bindingUtility->toQVariant(args[i]);
        variants[i] = qvar;
        if (qvar.type() == QVariant::Invalid) {
            metacallArgs[i+1] = &variants.at(i); // QVariant with no type (yet)
        }
        else {
            // assign pointer of qvariant to qt_metacall  arguments
            metacallArgs[i+1] = variants.at(i).constData(); // must not detach!
        }
    } // end for loop for arguments
    return bRet;
}

/*!
    Check parameters are match or not

    @param parameterTypes parameter types of the slot
    @param variants variants passed from JS
    @return true if types are compatible, false not compatible
*/
bool ObjectBinding::matchParameters(QList<QByteArray>& parameterTypes,
                                    QVector<QVariant>& variants)
{
    bool match(true);
    int argCount = variants.count();
    if (parameterTypes.count() == argCount) {
        // check type of individual argument
        for (int i=0; i<argCount && match; ++i) {
            const QByteArray& typeName=parameterTypes[i];
            QVariant::Type parameterType = QVariant::nameToType(typeName);
            if (variants[i].type() != parameterType) {
                // Special case handling, when types don't match
                match = false;
                if (typeName == KQVariant) {
                    // Provider expects any type. Work around for
                    // QVariant::Map, QVariant::List, QVariant::DateTime,
                    // Number Object, etc
                    match = true;
                }
                else if (parameterType == QVariant::Double &&
                         variants[i].type() == QVariant::Int) {
                    // Special handling for double input.
                    // Convert variant from int to double value
                    variants[i].convert(parameterType);
                    match = true;
                }
                else if (parameterType == QVariant::Int &&
                         variants[i].type() == QVariant::Double) {
                    // Special handling for integer input
                    // S60 browser converts integer into NPAPI double value
                    int intTemp = variants[i].toDouble();
                    double dblTemp = variants[i].toDouble();
                    // Check if variants[i] is really an int number
                    if (dblTemp - intTemp != 0.0) {
                        SET_ERROR(this, ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                                  ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
                    }
                    // Check if invalid int number
                    else if (variants[i].toDouble() > MAX_INT32_VAL  ||
                        variants[i].toDouble() < MIN_INT32_VAL) {
                        SET_ERROR(this, BindingUtility::SFW_ERROR_INVALID_INT,
                                  BindingUtilityErrDesc[BindingUtility::SFW_ERROR_INVALID_INT]);
                    }
                    else {
                        // Seems QT does not convert big integer numbers
                        // properly from QVariant::Double to QVariant::Int.
                        // NOTE: variants[i].convert(type) does not work in
                        // this case, so the workaround: replace variants[i]
                        // with a new QVariant with correct int value
                        variants[i] = QVariant(intTemp);
                        match = true;
                    }
                }
                else if (variants[i].type() == QVariant::Invalid) {
                    // NOTE: variants[i].convert(type) does not work for
                    // invalid type, so workaround: replace variants[i] with a
                    // new QVariant with type needed
                    variants[i] = QVariant(parameterType);
                    match = true;
                }
            } // bool, int, double, QString should match
        }

        if (!match && this->getLastErrCode() == 0) {
            //should not use NPN_SETEXCEPTION here since sfw supports overloaded methods
            //if the current slot signature does not match input parameter types it is possible
            //method has been overloaded with another signature. should throw an npn exception
            //at the end of 'indexOfSlot' method if no slot if found
            SET_ERROR(this, ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH,
                             ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_PARAMETER_TYPE_MISMATCH]);
        }
    }
    else {
        SET_ERROR(this, ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS,
                               ObjectBindingErrDesc[ObjectBinding::SFW_ERROR_NUMBER_OF_PARAMETERS]);
        match = false;
    }

    return match;
}

} // namespace
// END OF FILE
