/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



// INCLUDE FILES
#include <QtDebug>
#include "secpolicy.h"
#include "policyparser.h"
#include "xmlpolicyserializer.h"
#include "usercondition.h" // For debug only
namespace WRT
{

/*!
 * \class SecPolicy
 * This class represents an access control policy
 * Operations of this class include evaluating (i.e. parsing) the policy,
 * retrieving the results of parsing, as well as serializing the results
 * to be reused later.
 */

/*!
 * Constructor
 * \param[in] policy - Policy file name
 */
SecPolicy::SecPolicy(const QString &policy) :
    m_parser(new PolicyParser(policy)),
    m_serializer(new XMLPolicySerializer())
{
    m_parser->setSerializer(static_cast<XMLPolicySerializer*>(m_serializer));
}

/*!
 * Constructor
 * \param[in] policy - Policy file in memory
 */
SecPolicy::SecPolicy(const QByteArray &policy) :
    m_parser(new PolicyParser(policy)),
    m_serializer(new XMLPolicySerializer())
{
    m_parser->setSerializer(static_cast<XMLPolicySerializer*>(m_serializer));
}

/*!
 * Destructor
 */
SecPolicy::~SecPolicy()
{
    delete m_parser;
    delete m_serializer;
}

/*!
 * Evaluate the policy for the specified domain
 * \param[in] domain - Domain to evaluate the policy file for
 * \return List of capabilities
 */
const CapabilitySet * SecPolicy::evaluateOn(const QString &domain) const
{
    m_parser->parse(domain);
    return m_parser->capabilities(domain);
}

/*!
 * Evaluate the policy for the default domain
 * \return List of capabilities
 */
const CapabilitySet * SecPolicy::evaluate() const
{
    m_parser->parse();
    return m_parser->capabilities(m_parser->defaultDomain());
}

/*!
 * Checks whether a capability has an alias
 * \pre The policy file must be specified and parsed
 * \param[in] capability - Capability to check for aliases
 * \return TRUE if aliase exists, else FALSE
 */
bool SecPolicy::hasAlias(const QString &capability) const
{
    return m_parser->hasAlias(capability);
}

/*!
 * Gets the aliases for a specified capability
 * \pre The policy file must be specified and parsed
 * \param[in] capability - Capability to check for aliases
 * \return The list of aliases (which are capabilities themselves)
 */
CapabilityList SecPolicy::aliasesFor(const QString &capability) const
{
    return m_parser->aliasesFor(capability);
}

/*!
 * Persists the policy to a storage location
 * \pre The policy file must be specified and parsed
 * \param[in] key - Key for the policy
 * \param[in] store - Storage location
 * \param[in] domain - Domain of the policy to persist
 * \return TRUE if succeeds, else FALSE
 */
bool SecPolicy::persist(const QString &key,
                        const QString &store,
                        const QString &domain) const
{

    return m_serializer->persist(key,
                                store,
                                domain,
                                m_parser->capabilities(domain),
                                m_parser->aliases());

}

bool SecPolicy::persist(QByteArray& blob,
                        const QString &domain) const
{

    return m_serializer->persist(blob,
                                domain,
                                m_parser->capabilities(domain),
                                m_parser->aliases());

}


//TBD: not quite right that this is a method of this class since
//policy should represent the static state of a policy. This just
//gets passed through to the policy parser. This suggests that the
//policy parser should be allocated by the session rather than here!!!!

/*!
 * Sets the user condition handler for this policy
 * \param[in] handler - User Condition Handler
 * \return void
 */
void SecPolicy::setUserHandler(const UserConditionHandler *handler) const
{
    m_parser->setUserHandler(handler);
}

/*!
 * Gets the domains for this policy
 * \pre The policy file must be specified and parsed
 * \return List of domains
 */
QStringList SecPolicy::domains() const
{
    return m_parser->domains();
}

/*!
 * Gets the default domain for this policy
 * \pre The policy file must be specified and parsed
 * \return The default domain
 */
QString SecPolicy::defaultDomain() const
{
    return m_parser->defaultDomain();
}

} // end namespace WRT
