/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "webtouchnavigation.h"
#include "wrtpage.h"

#include <QApplication>
#include <QGraphicsSceneMouseEvent>
#include <QMouseEvent>
#include <QGraphicsScene>
#include <QGraphicsSceneMouseEvent>

#include <qwebframe.h>
#include <qwebpage.h>
#include <qgraphicswebview.h>

const int KTouchDownStartTime = 200;
const int KHoverTimeoutThreshold = 100;

const int KJitterBufferThreshold = 200;
const int KFlickScrollThreshold = 400;
const int KCumlativeDistanceThreshold = 40;

const int KNodeSearchThreshold = 500;
const int KNodeSearchQuickClickThreshold = 900;

const int KDecelerationCount = 40;

int xInRect(const QRect& r, int x)
{
    int x1 = qMin(x, r.right()-2);
    return qMax(x1, r.left()+2);
}

int yInRect(const QRect& r, int y)
{
    int y1 = qMin(y, r.bottom()-2);
    return qMax(y1, r.top()+2);
}

int fastDistance(const QPoint& p, const QPoint& q)
{
    return (p.x() - q.x()) * (p.x() - q.x()) + (p.y() - q.y()) * (p.y() - q.y());
}

QPoint closestElement (QWebFrame* frame, WRT::WebTouchEvent& touchEvent, const int searchThreshold)
{
    QPoint adjustedPoint(touchEvent.m_pos);

    QWebHitTestResult htr = frame->hitTestContent(adjustedPoint);

    touchEvent.m_editable = htr.isContentEditable();

    if (!htr.element().tagName().toLower().compare("select")  && htr.element().hasAttribute("multiple"))
        touchEvent.m_modifier = Qt::ControlModifier;

    if (!htr.linkElement().isNull())
        return adjustedPoint;

    QWebElementCollection elementList = frame->findAllElements(QString("a"));

    // find all the link element
    QWebElementCollection::iterator it(elementList.begin());

    QPoint fp = adjustedPoint + frame->scrollPosition();
    int maxDist = searchThreshold;
    while (it != elementList.end()) {
        QRect nRect((*it).geometry());
        if (nRect.isValid()) {
                QPoint pt(xInRect(nRect, fp.x()), yInRect(nRect, fp.y()));
                int dist = fastDistance(pt, fp);
                if (dist < maxDist) {
                    adjustedPoint = pt - frame->scrollPosition();
                    maxDist = dist;
                }
        }
        ++it;
    }

    return adjustedPoint;
}

#if defined FIXED_35873  // FIXME: remove the #else once 35873 gets in the product branch http://trac.webkit.org/wiki/QtWebKitBackportingFixes
void QWEBKIT_EXPORT qtwebkit_webframe_scrollRecursively(QWebFrame* qFrame, int dx, int dy, const QPoint& pos);
#else
void qtwebkit_webframe_scrollRecursively(QWebFrame* qFrame, int dx, int dy, const QPoint& pos)
{
    if (!qFrame || !qFrame->page())
        return;

    QWebFrame* frame = qFrame->page()->frameAt(pos);
    while (frame) {
        if (dx > 0) {
            if (frame->scrollBarValue(Qt::Horizontal) < frame->scrollBarMaximum(Qt::Horizontal))
                break;
        }
        else if (dx < 0) {
            if (frame->scrollBarValue(Qt::Horizontal) > frame->scrollBarMinimum(Qt::Horizontal))
                break;
        }

        if (dy > 0) {
            if (frame->scrollBarValue(Qt::Vertical) < frame->scrollBarMaximum(Qt::Vertical))
                break;
        }
        else if (dy < 0) {
            if (frame->scrollBarValue(Qt::Vertical) > frame->scrollBarMinimum(Qt::Vertical))
                break;
        }
        frame = frame->parentFrame();
    }

    frame = (frame) ? frame : qFrame->page()->mainFrame();
    frame->scroll(dx, dy);
}
#endif

namespace WRT {

/*!
    \class WebTouchNavigation
    \since cwrt 1.0
    \brief cwrt navigation.

    \sa WebNavigation, WebHtmlTabIndexedNavigation, WebCursorNavigation, WebDirectionalNavigation
*/
WebTouchNavigation::WebTouchNavigation(QObject* object, QWebPage* webPage) :
m_viewObject(object)
, m_webPage(webPage)
{
    Q_ASSERT(m_viewObject != 0);
    m_viewObject->installEventFilter(this);

    connect(&m_downTimer, SIGNAL(timeout()), this, SLOT(downTimerFired()));
    connect(&m_hoverTimer, SIGNAL(timeout()), this, SLOT(hoverTimerFired()));
    connect(&m_quickDownTimer, SIGNAL(timeout()), this, SLOT(quickDownTimerFired()));
    connect(&m_quickUpTimer, SIGNAL(timeout()), this, SLOT(quickUpTimerFired()));
    connect(&m_decelerationTimer, SIGNAL(frameChanged(int)), this, SLOT(decelerationTimerEvent(int)));
}

WebTouchNavigation::~WebTouchNavigation()
{
    m_viewObject->removeEventFilter(this);
}

bool WebTouchNavigation::eventFilter(QObject *object, QEvent *event)
{
    if (object != m_viewObject)
        return false;

    bool eventHandled = false;
    switch (event->type()) {
    case QEvent::MouseButtonPress:
        if (static_cast<QMouseEvent*>(event)->buttons() & Qt::LeftButton) {
            WebTouchEvent e(static_cast<QMouseEvent*>(event));
            handleDownEvent(e);
        }
        eventHandled = true;
    break;
    case QEvent::MouseMove:
        if (static_cast<QMouseEvent*>(event)->buttons() & Qt::LeftButton) {
            WebTouchEvent e(static_cast<QMouseEvent*>(event));
            handleMoveEvent(e);
        }
        eventHandled = true;
    break;
    case QEvent::MouseButtonRelease: {
        WebTouchEvent e(static_cast<QMouseEvent*>(event));
        handleReleaseEvent(e);
        eventHandled = true;
    }
    break;
    case QEvent::GraphicsSceneMousePress:
        if (static_cast<QGraphicsSceneMouseEvent*>(event)->buttons() & Qt::LeftButton) {
            WebTouchEvent e(static_cast<QGraphicsSceneMouseEvent*>(event));
            handleDownEvent(e);
        }
        eventHandled = true;
    break;
    case QEvent::GraphicsSceneMouseMove:
        if (static_cast<QGraphicsSceneMouseEvent *>(event)->buttons() & Qt::LeftButton) {
            WebTouchEvent e(static_cast<QGraphicsSceneMouseEvent*>(event));
            handleMoveEvent(e);
        }
        eventHandled = true;
    break;
    case QEvent::GraphicsSceneMouseRelease: {
        WebTouchEvent e(static_cast<QGraphicsSceneMouseEvent*>(event));
        handleReleaseEvent(e);
        eventHandled = true;
    }
    break;
    case QEvent::MouseButtonDblClick:
    case QEvent::ContextMenu:
    case QEvent::CursorChange:
    case QEvent::DragEnter:
    case QEvent::DragLeave:
    case QEvent::DragMove:
    case QEvent::Drop:
    case QEvent::GrabMouse:
#ifndef CWRT_BUILDING_TENONE
    case QEvent::GraphicsSceneContextMenu:
#endif
    case QEvent::GraphicsSceneDragEnter:
    case QEvent::GraphicsSceneDragLeave:
    case QEvent::GraphicsSceneDragMove:
    case QEvent::GraphicsSceneDrop:
    case QEvent::GraphicsSceneHelp:
    case QEvent::GraphicsSceneHoverEnter:
    case QEvent::GraphicsSceneHoverLeave:
    case QEvent::GraphicsSceneHoverMove:
    case QEvent::HoverEnter:
    case QEvent::HoverLeave:
    case QEvent::HoverMove:
    case QEvent::TouchBegin:
    case QEvent::TouchUpdate:
    case QEvent::TouchEnd:
    case QEvent::Gesture:
    case QEvent::GestureOverride:
        eventHandled = true;
    break;
    default:
        break;
    }

    return eventHandled;
}

void WebTouchNavigation::quickDownTimerFired()
{
    m_touchDown.m_pos = closestElement(m_webFrame, m_touchDown, KNodeSearchQuickClickThreshold);
    m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMousePress : QEvent::MouseButtonPress;
    m_touchDown.m_button = Qt::LeftButton;
    m_touchDown.m_buttons = Qt::NoButton;
    generateMouseEvent(m_touchDown);
    m_quickUpTimer.setSingleShot(true);
    m_quickUpTimer.start(0);
}

void WebTouchNavigation::quickUpTimerFired()
{
    m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMouseRelease : QEvent::MouseButtonRelease;
    m_touchDown.m_button = Qt::LeftButton;
    m_touchDown.m_buttons = Qt::NoButton;
    generateMouseEvent(m_touchDown);
}

void WebTouchNavigation::downTimerFired()
{
    m_touchDown.m_pos = closestElement(m_webFrame, m_touchDown, KNodeSearchThreshold);
    m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMousePress : QEvent::MouseButtonPress;
    m_touchDown.m_button = Qt::LeftButton;
    m_touchDown.m_buttons = Qt::NoButton;
    generateMouseEvent(m_touchDown);
    m_touchDown.m_fired = true;
}

void WebTouchNavigation::hoverTimerFired()
{
    m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMouseMove : QEvent::MouseMove;
    m_touchDown.m_button = Qt::NoButton;
    m_touchDown.m_buttons = Qt::NoButton;
    generateMouseEvent(m_touchDown);
}

void WebTouchNavigation::decelerationTimerEvent(int count)
{
    if (count == m_decelerationTimer.endFrame()) {
        m_decelerationTimer.stop();
        return;
    }

    qreal decelerationFactor = 1 - m_decelerationTimer.currentValue();
    QPointF diff(decelerationFactor * m_decelerationSpeed.x(), decelerationFactor * m_decelerationSpeed.y());
#if defined CWRT_BUILDING_NINETWO && CWRT_BUILDING_QTUNSTABLE
    qtwebkit_webframe_scrollRecursively(m_webFrame, diff.x(), diff.y(), m_prevMouseMove.m_pos);
#else
    m_webFrame->scroll(diff.x(), diff.y());
#endif

}


void WebTouchNavigation::handleDownEvent(const WebTouchEvent &event)
{
    m_decelerationTimer.stop();
    m_downTimer.stop();
    m_hoverTimer.stop();
    m_quickDownTimer.stop();
    m_quickUpTimer.stop();

    m_webFrame = m_webPage->frameAt(event.m_pos);
    if (!m_webFrame)
        m_webFrame = m_webPage->currentFrame();

    m_scrolling = false;
    m_decelerationSpeed = QPoint();
    m_cumlativeDistance = QPoint();
    m_touchDown = event;
    m_prevMouseMove = m_touchDown;

    m_hoverTimer.setSingleShot(true);
    m_hoverTimer.start(KHoverTimeoutThreshold);

    m_downTimer.setSingleShot(true);
    m_downTimer.start(KTouchDownStartTime);
}

void WebTouchNavigation::handleMoveEvent(const WebTouchEvent &event)
{
    int dist = fastDistance(event.m_pos, m_touchDown.m_pos);

    if (m_downTimer.isActive()) {

        // this mouse move event is far away from the touch down position in
        // less than xx sec, the user must want to scroll quickly.
        if (!m_scrolling && dist > KFlickScrollThreshold) {
            m_scrolling = true;
            QPoint diff = m_prevMouseMove.m_pos - event.m_pos;
#if defined CWRT_BUILDING_NINETWO && CWRT_BUILDING_QTUNSTABLE
            qtwebkit_webframe_scrollRecursively(m_webFrame, diff.x(), diff.y(), event.m_pos);
#else
           m_webFrame->scroll(diff.x(), diff.y());
#endif
            m_decelerationSpeed = m_prevMouseMove.m_pos;
            m_cumlativeDistance += event.m_pos - m_prevMouseMove.m_pos;
            m_prevMouseMove = event;

            // don't send mouse down event.
            m_downTimer.stop();
            m_hoverTimer.stop();
        }
        return;
    }

    // pan scroll
    if (dist > KJitterBufferThreshold) {
        m_scrolling = true;
        QPoint diff = m_prevMouseMove.m_pos - event.m_pos;
#if defined CWRT_BUILDING_NINETWO && CWRT_BUILDING_QTUNSTABLE
            qtwebkit_webframe_scrollRecursively(m_webFrame, diff.x(), diff.y(), event.m_pos);
#else
           m_webFrame->scroll(diff.x(), diff.y());
#endif
        m_decelerationSpeed = m_prevMouseMove.m_pos;
        m_cumlativeDistance += event.m_pos - m_prevMouseMove.m_pos;
        m_prevMouseMove = event;
    }
}

void WebTouchNavigation::handleReleaseEvent(const WebTouchEvent &event)
{

    if (!m_scrolling && (m_hoverTimer.isActive() || m_downTimer.isActive())) { // Quick tap

        if (m_hoverTimer.isActive()) {
            m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMouseMove : QEvent::MouseMove;
            m_touchDown.m_button = Qt::NoButton;
            m_touchDown.m_buttons = Qt::NoButton;
            generateMouseEvent(m_touchDown);
        }

        m_hoverTimer.stop();
        m_downTimer.stop();

        m_quickDownTimer.setSingleShot(true);
        m_quickDownTimer.start(0);
        return;
    }

    m_hoverTimer.stop();
    m_downTimer.stop();

    // use the cumlative distance in this case because it seems like a positive user experience
    if (m_cumlativeDistance.manhattanLength() > KCumlativeDistanceThreshold) {

        if (m_touchDown.m_fired) {
            // send mouse up event invalidate click
            m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMouseRelease : QEvent::MouseButtonRelease;
            m_touchDown.m_pos = QPoint(-1, -1);
            m_touchDown.m_button = Qt::LeftButton;
            m_touchDown.m_buttons = Qt::NoButton;
            m_touchDown.m_editable = false;

#if defined CWRT_BUILDING_NINETWO && CWRT_BUILDING_QTUNSTABLE
            static_cast<QWidget*>(m_viewObject)->setAttribute(Qt::WA_InputMethodEnabled, false);
#endif
            generateMouseEvent(m_touchDown);
        }

        m_decelerationSpeed = m_decelerationSpeed - event.m_pos;
        m_decelerationTimer.setDuration(500);
        m_decelerationTimer.setFrameRange(0, KDecelerationCount);
        m_decelerationTimer.setCurveShape(QTimeLine::EaseOutCurve);
        m_decelerationTimer.start();
        return;
    }

    if (m_touchDown.m_fired) {
        // send mouse up event
        m_touchDown.m_type = (m_touchDown.m_graphicsSceneEvent) ? QEvent::GraphicsSceneMouseRelease : QEvent::MouseButtonRelease;
        m_touchDown.m_button = Qt::LeftButton;
        m_touchDown.m_buttons = Qt::NoButton;
        generateMouseEvent(m_touchDown);
    }
}

void WebTouchNavigation::generateMouseEvent(const WebTouchEvent& touchEvent)
{
    if (touchEvent.m_type == QEvent::GraphicsSceneMousePress
        || touchEvent.m_type == QEvent::GraphicsSceneMouseMove
        || touchEvent.m_type == QEvent::GraphicsSceneMouseRelease) {
        QGraphicsSceneMouseEvent ievmm(touchEvent.m_type);
        ievmm.setPos(touchEvent.m_pos);
        ievmm.setScenePos(touchEvent.m_scenePos);
        ievmm.setScreenPos(touchEvent.m_screenPos);
        ievmm.setButtonDownPos(touchEvent.m_button, touchEvent.m_buttonDownPos);
        ievmm.setButtonDownScenePos( touchEvent.m_button, touchEvent.m_buttonDownScenePos);
        ievmm.setButtonDownScreenPos( touchEvent.m_button, touchEvent.m_buttonDownScreenPos);
        ievmm.setLastPos(touchEvent.m_lastPos);
        ievmm.setLastScenePos(touchEvent.m_lastScenePos);
        ievmm.setLastScreenPos(touchEvent.m_lastScreenPos);
        ievmm.setButtons(touchEvent.m_buttons);
        ievmm.setButton( touchEvent.m_button);
        ievmm.setModifiers(touchEvent.m_modifier);
        m_webPage->event(&ievmm);
    } else {

#if defined CWRT_BUILDING_NINETWO && CWRT_BUILDING_QTUNSTABLE
        if (touchEvent.m_editable && touchEvent.m_type == QEvent::MouseButtonRelease
            && static_cast<QWidget*>(m_viewObject)->testAttribute(Qt::WA_InputMethodEnabled)) {
            QEvent rsipevent(QEvent::RequestSoftwareInputPanel);
            QApplication::sendEvent(static_cast<QWidget*>(m_viewObject), &rsipevent);
        }
#endif

        QMouseEvent ievmm(touchEvent.m_type, touchEvent.m_pos, touchEvent.m_button, touchEvent.m_buttons, touchEvent.m_modifier);
        m_webPage->event(&ievmm);
    }
}

}
