/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "webwidgetrunner.h"

#include "floatingview.h"
#include "homescreenview.h"
#include "jsmenu.h"
#include "jscontextmenu.h"
#include "miniview.h"
#include "standaloneview.h"
#include "viewmode.h"
#include "webwidget.h"
#include "wrtcontroller.h"
#include "wrtpage.h"
#include "wrtwidgetcontainerMaemo.h"
#include "WidgetProperties.h"
#include "w3csettingskeys.h"

#include <QTimer>
#include <QDebug>
#include <QGraphicsWebView>
#include <QWebFrame>

namespace WRT {
namespace Maemo {

WebWidgetRunner::WebWidgetRunner()
    : m_page(0)
    , m_homeScreenView(0)
    , m_standaloneView(0)
    , m_menu(0)
    , m_webWidget(0)
    , m_widgetContainer(0)
    , m_fallbackSize(QSize(312, 82))
    , m_viewMode(WidgetViewModeCount)
    , m_lastHomeScreenViewMode(WebWidgetRunner::WidgetViewModeCount)
    , m_lastStandaloneViewMode(WebWidgetRunner::WidgetViewModeCount)
    , m_webInspector(false)
{
}

WebWidgetRunner::~WebWidgetRunner()
{
    if (m_homeScreenView)
        m_homeScreenView->deleteLater();

    if (m_webWidget)
        m_webWidget->deleteLater();

    if (m_widgetContainer)
        m_widgetContainer->deleteLater();
}

bool WebWidgetRunner::load(QString desktopFilePath)
{
    m_desktopFileInfo = QFileInfo(desktopFilePath);
    if (!m_desktopFileInfo.exists()) {
        qCritical() << "desktop file doesn't exist";
        return false;
    }

    m_webWidget = new WebWidget(m_desktopFileInfo.absoluteFilePath());
    if (!m_webWidget || !m_webWidget->isValid()) {
        qCritical() << "error creating WebWidget";
        return false;
    }

    m_widgetContainer = new WidgetContainerMaemo();
    if (!m_widgetContainer) {
        qCritical() << "error creating WidgetContainerMaemo";
        return false;
    }

    QString widgetId = m_webWidget->webAppId();
    QString widgetPath = QUrl(m_webWidget->getWidgetResource()).toString(QUrl::RemoveScheme);

    m_widgetContainer->setWidgetInfo(widgetPath, widgetId);

    WidgetProperties* widgetProps = m_webWidget->widgetProperties();
    if (!widgetProps) {
        qCritical() << "error getting WidgetProperties";
        return false;
    }

    m_widgetContainer->setWidgetProperties(widgetProps);
    m_widgetContainer->loadWrtApis();

    // check is Debug feature can be found from config.xml
    m_webInspector = initInspector( widgetProps);

    connect(m_widgetContainer, SIGNAL(viewModeChangeRequested(QString)),
            this, SLOT(onViewModeChangeRequested(QString)));

    loadWidgetView();

    return true;
}

void WebWidgetRunner::loadWidgetView()
{
    WidgetViewMode mode = WidgetViewModeCount;
    getViewModeValue(m_webWidget->viewmode(), mode);

    if (mode == FloatingViewMode || mode == MinimizedViewMode) {
        QString id = homeScreenId();

        if (mode == MinimizedViewMode) {
            m_homeScreenView = new MiniView(id);
            connect(m_homeScreenView, SIGNAL(clicked()),
                    this, SLOT(onHomeViewClicked()));
        } else {
            QGraphicsWebView* webView = m_widgetContainer->wrtController()->graphicsWebView();
            m_homeScreenView = new FloatingView(webView, id);
        }

        connect(m_homeScreenView, SIGNAL(visibilityChanged(bool)),
                this, SLOT(onVisibilityChanged(bool)));
    }

    m_page = m_widgetContainer->wrtController()->wrtPage();
    QWebFrame* mainFrame = m_page->mainFrame();

    connect(mainFrame, SIGNAL(titleChanged(QString)),
            this, SLOT(onFrameTitleChanged(QString)));
    connect(mainFrame, SIGNAL(loadFinished(bool)),
            this, SLOT(onFrameLoadFinished(bool)));

    QUrl url = m_webWidget->getUrl();
    m_widgetContainer->whitelistAccess(url);
    mainFrame->load(url);

    createViewMenu();
    setInitialSize(m_webWidget->size());
    setViewMode(mode);
}

WebWidgetRunner::WidgetViewMode WebWidgetRunner::viewMode() const
{
    return m_viewMode;
}

void WebWidgetRunner::setInitialSize(const QSize& size)
{
    m_initialSize = size;
}

QSize WebWidgetRunner::initialSize() const
{
    return !m_initialSize.isNull() ? m_initialSize : m_fallbackSize;
}

QSize WebWidgetRunner::fallbackSize() const
{
    return m_fallbackSize;
}

QString WebWidgetRunner::homeScreenId() const
{
    return m_desktopFileInfo.fileName() + "-0";
}

bool WebWidgetRunner::prepareHomeScreenView(const QSize& size, bool deleteStandaloneView)
{
    if (!m_homeScreenView)
        return false;

    if (m_standaloneView && deleteStandaloneView)
        delete m_standaloneView;

    m_homeScreenView->setGeometry(QRect(m_homeScreenView->geometry().topLeft(), size));
    m_homeScreenView->setPage(m_page);
    m_homeScreenView->restore();

    m_homeScreenView->show();

    return true;
}

void WebWidgetRunner::prepareStandaloneView()
{
    if (m_viewMode == WindowedViewMode || m_viewMode == FullscreenViewMode)
        return;

    if (!m_standaloneView) {
        QGraphicsWebView* webView = m_widgetContainer->wrtController()->graphicsWebView();
        m_standaloneView = new StandaloneView(webView, NULL, m_webInspector);

        connect(m_standaloneView, SIGNAL(visibilityChanged(bool)),
                this, SLOT(onVisibilityChanged(bool)));
        connect(m_standaloneView, SIGNAL(exitRequested()),
                this, SLOT(onStandaloneViewExitRequested()));

        m_standaloneView->setMenu(m_menu);

        WRT::JSContextMenu* jsContextMenu =
                     static_cast<WRT::JSContextMenu*>(m_widgetContainer->apiObject("JSContextMenu"));
        m_standaloneView->setContextMenu(jsContextMenu);
    }
    else {
        // NOTE:
        // This postpones the standalone view activation until the next
        // main loop iteration, in order to avoid a flick when changing
        // view modes.
        QTimer::singleShot(0, this, SLOT(onStandaloneViewActivated()));

        if (m_lastHomeScreenViewMode == FloatingViewMode)
            m_standaloneView->restore();
    }

    m_standaloneView->setPage(m_page);
}

void WebWidgetRunner::onStandaloneViewActivated()
{
    m_standaloneView->activateWindow();
    onFrameTitleChanged(m_page->mainFrame()->title());
}

void WebWidgetRunner::onStandaloneViewExitRequested()
{
    QStringList supportedViewModes = m_webWidget->supportedViewModes();
    if (supportedViewModes.contains(getViewModeName(WindowedViewMode)))
        setViewMode(WindowedViewMode);
    else
        delete m_standaloneView;
}

bool WebWidgetRunner::setViewMode(WidgetViewMode mode, bool deleteCurrentView)
{
    switch (mode) {

    case MinimizedViewMode:
        if (!prepareHomeScreenView(fallbackSize(), deleteCurrentView))
            return false;
        m_lastHomeScreenViewMode = mode;
        break;

    case FloatingViewMode:
        if (!prepareHomeScreenView(initialSize(), deleteCurrentView))
            return false;
        m_lastHomeScreenViewMode = mode;
        break;

    case FullscreenViewMode:
        prepareStandaloneView();
        m_lastStandaloneViewMode = mode;
        if (preferredStandaloneViewMode() == WindowedViewMode)
            m_standaloneView->showFullScreen(true);
        else
            m_standaloneView->showFullScreen(false);
        break;

    case WindowedViewMode:
        prepareStandaloneView();
        if (m_lastGeometry.isValid())
            m_standaloneView->setGeometry(m_lastGeometry);
        m_lastStandaloneViewMode = mode;
        m_standaloneView->showNormal();
        break;

    default:
        return false;
    }

    if (m_viewMode < WidgetViewModeCount && m_viewMode != mode)
        m_widgetContainer->viewModeChanged(getViewModeName(mode));

    m_viewMode = mode;

    return true;
}

void WebWidgetRunner::createViewMenu()
{
    QStringList supportedViewModes = m_webWidget->supportedViewModes();
    QString windowedMode = getViewModeName(WindowedViewMode);
    if (!supportedViewModes.contains(windowedMode))
        return;

    m_menu = new QMenu;
    WRT::JSMenu* jsMenu = static_cast<WRT::JSMenu*>(m_widgetContainer->apiObject("JSMenu"));
    connect(jsMenu, SIGNAL(titleChanged(QString)),
            this, SLOT(onFrameTitleChanged(QString)));
    jsMenu->setMenuWidget(m_menu);
}

void WebWidgetRunner::onHomeViewClicked()
{
    setViewMode(preferredStandaloneViewMode());
}

WebWidgetRunner::WidgetViewMode WebWidgetRunner::preferredStandaloneViewMode()
{
    QStringList supportedViewModes = m_webWidget->supportedViewModes();

    if (supportedViewModes.contains(getViewModeName(WindowedViewMode)))
        return WindowedViewMode;
    else if (supportedViewModes.contains(getViewModeName(FullscreenViewMode)))
        return FullscreenViewMode;

    return WidgetViewModeCount;
}

bool WebWidgetRunner::initInspector( WidgetProperties* widgetProps)
{
    if( !widgetProps){
        return false;
    }

    int i = 1;
    QVariant val;
    do {
        val = widgetProps->plistValue( W3CSettingsKey::WIDGET_FEATURE.arg(i++) + ":name");
        if( !val.isNull() && !val.toString().compare( "Debug", Qt::CaseInsensitive)){
            // Initialize web inspector, id Debug flag is on
            return true;
        }
    } while( !val.isNull());

    return false;
}

void WebWidgetRunner::onFrameTitleChanged(const QString &title)
{
    if (!m_standaloneView)
        return;

    if (!title.isEmpty())
        m_standaloneView->setWindowTitle(title);
    else
        m_standaloneView->setWindowTitle(m_webWidget->getName());
}

void WebWidgetRunner::onFrameLoadFinished(bool ok)
{
    Q_UNUSED(ok);

    // NOTE:
    // This sends the view mode change to the widget for the first
    // time. Maybe this should be emitted on QWebPage::loadFinished
    // instead of QWebFrame::loadFinished.
    m_widgetContainer->viewModeChanged(getViewModeName(m_viewMode));

    QWebFrame* mainFrame = m_page->mainFrame();
    disconnect(mainFrame, SIGNAL(loadFinished(bool)),
              this, SLOT(onFrameLoadFinished(bool)));
}

void WebWidgetRunner::onViewModeChangeRequested(QString viewMode)
{
    WidgetViewMode viewModeValue;

    if (!getViewModeValue(viewMode, viewModeValue)) {
        qWarning() << viewMode << "is not a valid view mode";
        return;
    }

    if (viewModeValue == m_viewMode)
        return;

    // NOTE:
    // In Maemo5 we have no means to go back to home screen in order to show
    // the home view. Hence, we should ignore the requests to set any of the
    // home modes using the JavaScript API, at least for Beta release.
    if (viewModeValue == MinimizedViewMode || viewModeValue == FloatingViewMode)
        return;

    QStringList supportedViewModes = m_webWidget->supportedViewModes();
    if (!supportedViewModes.contains(viewMode)) {
        qWarning() << viewMode << "not supported";
        return;
    }

    if (!setViewMode(viewModeValue)) {
        qWarning() << "Error setting view mode" << viewMode;
        return;
    }
}

void WebWidgetRunner::onVisibilityChanged(bool visible)
{
    m_widgetContainer->visibilityChanged(visible);

    if (m_viewMode >= WidgetViewModeCount)
        return;

    if (QObject::sender() == m_homeScreenView) {
        if (visible)
            setViewMode(m_lastHomeScreenViewMode, false);
        else
            m_homeScreenView->freeze();

        return;
    }

    if (QObject::sender() == m_standaloneView) {
        if (visible)
            setViewMode(m_lastStandaloneViewMode);
    }
}

}
}


