/*
  This file is part of "WhoAmI" - A "Guess whats on the photo" riddle game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "DownloadImage.h"

#include <cerrno>
#include <cstdio>
#include <cstring>
#include <sys/stat.h>
#include <fcntl.h>

#include <iostream>
#include <list>

#include <curl/curl.h>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Images/Loader/JPEG.h>

static size_t WriteFunction(void* ptr, size_t size, size_t nmemb, void* stream)
{
  DownloadImageTask* task=static_cast<DownloadImageTask*>(stream);

  task->AppendToBuffer((const char*)ptr,size*nmemb);

  return size*nmemb;
}

DownloadImageTask::DownloadImageTask()
{
  // no code
}

void DownloadImageTask::SetImage(Image* image)
{
  this->image=image;
}

void DownloadImageTask::Run()
{
  CURL         *curl;
  CURLcode     result;
  std::string  url;

  error.clear();

  curl=curl_easy_init();

  if (curl==NULL) {
    error="Cannot initiale curl handle!";
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_NOSIGNAL,1);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_COOKIEFILE,"");
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_WRITEFUNCTION,WriteFunction);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  result=curl_easy_setopt(curl,CURLOPT_WRITEDATA,this);
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  SetProgress(0);
  SetCaption(L"Downloading image from flickr...");

  std::string imageUrl=image->GetImageLink();

  std::cout << imageUrl << std::endl;

  std::string::size_type offset=imageUrl.find(".static.flickr.com");

  if (offset!=std::string::npos) {
    imageUrl.erase(offset,18);
    imageUrl.insert(offset,"[...]");
  }

  SetAction(std::wstring(L"Downloading ")+Lum::Base::StringToWString(imageUrl)+L"...");

  result=curl_easy_setopt(curl,CURLOPT_URL,image->GetImageLink().c_str());
  if (result!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  buffer.clear();

  if (curl_easy_perform(curl)!=CURLE_OK) {
    error=curl_easy_strerror(result);
    curl_easy_cleanup(curl);
    return;
  }

  Lum::Base::Path path;
  FILE            *tmpfile;

  path.SetNativeDir(Lum::Base::Path::GetTemporary());
  path.SetBaseName(L"WhoAmI-tmp.jpg");

  tmpfile=fopen(Lum::Base::WStringToString(path.GetPath()).c_str(),"w+b");
  if (tmpfile==NULL) {
    error=std::string("Cannot open file: ")+strerror(errno);
    curl_easy_cleanup(curl);
    return;
  }

  if (fwrite(buffer.data(),1,buffer.length(),tmpfile)!=buffer.length()) {
    error=std::string("Cannot write into file: ")+strerror(errno);
    fclose(tmpfile);
    remove(Lum::Base::WStringToString(path.GetPath()).c_str());
    curl_easy_cleanup(curl);
    return;
  }

  fclose(tmpfile);

  Lum::Images::ImageRef   image=Lum::Images::Factory::factory->CreateImage();
  Lum::Images::LoaderJPEG loader;

  if (loader.Load(path.GetPath(),image)) {
    this->image->SetImage(image);
  }
  else {
    error="Cannot load image as jpg";
    std::cout << "Cannot load image " << this->image->GetImageLink() << " " << Lum::Base::WStringToString(path.GetPath()) << std::endl;
  }

  remove(Lum::Base::WStringToString(path.GetPath()).c_str());

  SetAction(L"Done.");
  SetProgress(100);

  curl_easy_cleanup(curl);
}

void DownloadImageTask::AppendToBuffer(const char* text, size_t size)
{
  buffer.append(text,size);
}

bool DownloadImageTask::HasError() const
{
  return !error.empty();
}

std::wstring DownloadImageTask::GetError() const
{
  return Lum::Base::StringToWString(error);
}

