/* $Id: dialog.hpp 31858 2009-01-01 10:27:41Z mordante $ */
/*
   copyright (c) 2008 - 2009 by mark de wever <koraq@xs4all.nl>
   part of the battle for wesnoth project http://www.wesnoth.org/

   this program is free software; you can redistribute it and/or modify
   it under the terms of the gnu general public license version 2
   or at your option any later version.
   this program is distributed in the hope that it will be useful,
   but without any warranty.

   see the copying file for more details.
*/

#ifndef GUI_DIALOGS_DIALOG_HPP_INCLUDED
#define GUI_DIALOGS_DIALOG_HPP_INCLUDED

#include "gui/dialogs/field-fwd.hpp"

#include <string>
#include <vector>

class CVideo;

namespace gui2 {

/**
 * Abstract base class for all dialogs.
 *
 * A dialog shows a certain window instance to the user. The subclasses of this
 * class will hold the parameters used for a certain window, eg a server
 * connection dialog will hold the name of the selected server as parameter that
 * way the caller doesn't need to know about the 'contents' of the window.
 */
class tdialog
{
public:
	tdialog() :
		retval_(0),
		fields_(),
		restore_(true)
	{}

	virtual ~tdialog();

	/** Shows the window */
	void show(CVideo& video);

	/***** ***** ***** setters / getters for members ***** ****** *****/

	int get_retval() const { return retval_; }

	void set_restore(const bool restore) { restore_ = restore; }

protected:

	/**
	 * Creates a new boolean field.
	 *
	 * The field created is owned by tdialog, the returned pointer can be used
	 * in the child classes as access to a field.
	 *
	 * @param id                  Id of the widget, same value as in WML.
	 * @param optional            Is the widget mandatory or optional.
	 * @param callback_load_value The callback function to set the initial value
	 *                            of the widget.
	 * @param callback_save_value The callback function to write the resulting
	 *                            value of the widget. Saving will only happen
	 *                            if the widget is enabled and the window closed
	 *                            with ok.
	 * @param callback_change     When the value of the widget changes this
	 *                            callback is called.
	 *
	 * @returns                   Pointer to the created widget.
	 */
	tfield_bool* register_bool(const std::string& id,
		const bool optional = false,
		bool (*callback_load_value) () = NULL,
		void (*callback_save_value) (const bool value) = NULL,
		void (*callback_change) (twidget* widget) = NULL);

	/**
	 * Creates a new integer field.
	 *
	 * See register_bool for more info.
	 */
	tfield_integer* register_integer(const std::string& id,
		const bool optional = false,
		int (*callback_load_value) () = NULL,
		void (*callback_save_value) (const int value) = NULL);

	/**
	 * Creates a new text field.
	 *
	 * See register_bool for more info.
	 */
	tfield_text* register_text(const std::string& id,
		const bool optional = false,
		std::string (*callback_load_value) () = NULL,
		void (*callback_save_value) (const std::string& value) = NULL);
private:
	/** Returns the window exit status, 0 means not shown. */
	int retval_;

	/**
	 * Contains the automatically managed fields.
	 *
	 * Since the fields are automatically managed and there are no search
	 * functions defined we don't offer access to the vector. If access is
	 * needed the creator should store a copy of the pointer.
	 */
	std::vector<tfield_*> fields_;

	/**
	 * Restore the screen after showing?
	 *
	 * Most windows should restore the display after showing so this value
	 * defaults to true. Toplevel windows (like the titlescreen don't want this
	 * behaviour so they can change it in pre_show().
	 */
	bool restore_;

	/**
	 * Builds the window.
	 *
	 * Every dialog shows it's own kind of window, this function should return
	 * the window to show.
	 *
	 * @returns                   The window to show.
	 */
	virtual twindow* build_window(CVideo& video) = 0;

	/**
	 * Actions to be taken before showing the window.
	 *
	 * @param video               The video which contains the surface to draw
	 *                            upon.
	 * @param window              The window to be shown.
	 */
	virtual void pre_show(CVideo& /*video*/, twindow& /*window*/) {}

	/**
	 * Actions to be taken after the window has been shown.
	 *
	 * @param window              The window which has been shown.
	 */
	virtual void post_show(twindow& /*window*/) {}

	/**
	 * Initializes all fields in the dialog.
	 *
	 * @param window              The window which has been shown.
	 */
	virtual void init_fields(twindow& window);

	/**
	 * When the dialog is closed with the OK status saves all fields.
	 *
	 * Saving only happens if a callback handler is installed.
	 *
	 * @param window              The window which has been shown.
	 */
	virtual void finalize_fields(twindow& window);
};

} // namespace gui2

#endif

