#include "digits.h"

/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the Graphics Dojo project on Qt Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 or 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

Digits::Digits(QWidget *parent)
        : QWidget(parent)
        , m_number(0)
        , m_transition(Slide)
{
    setAttribute(Qt::WA_OpaquePaintEvent, true);
    setAttribute(Qt::WA_NoSystemBackground, true);
    connect(&m_animator, SIGNAL(frameChanged(int)), SLOT(update()));
    m_animator.setFrameRange(0, 100);
    m_animator.setDuration(600);
    m_animator.setCurveShape(QTimeLine::EaseInOutCurve);
}

void Digits::setTransition(int tr) {
    m_transition = tr;
}

int Digits::transition() const {
    return m_transition;
}

void Digits::setNumber(int n) {
    if (m_number != n) {
        m_number = qBound(0, n, 99);
        preparePixmap();
        update();
    }
}

void Digits::flipTo(int n) {
    if (m_number != n) {
        m_number = qBound(0, n, 99);
        m_lastPixmap = m_pixmap;
        preparePixmap();
        m_animator.stop();
        m_animator.start();
    }
}


void Digits::drawFrame(QPainter *p, const QRect &rect) {
    p->setPen(Qt::NoPen);
    QLinearGradient gradient(rect.topLeft(), rect.bottomLeft());
    gradient.setColorAt(0.00, QColor(245, 245, 245));
    gradient.setColorAt(0.49, QColor(192, 192, 192));
    gradient.setColorAt(0.51, QColor(245, 245, 245));
    gradient.setColorAt(1.00, QColor(192, 192, 192));
    p->setBrush(gradient);
    QRect r = rect;
    p->drawRoundedRect(r, 15, 15, Qt::RelativeSize);
    r.adjust(1, 4, -1, -4);
    p->setPen(QColor(181, 181, 181));
    p->setBrush(Qt::NoBrush);
    p->drawRoundedRect(r, 15, 15, Qt::RelativeSize);
    p->setPen(QColor(159, 159, 159));
    int y = rect.top() + rect.height() / 2 - 1;
    p->drawLine(rect.left(), y, rect.right(), y);
}

QPixmap Digits::drawDigits(int n, const QRect &rect) {

    int scaleFactor = 2;
#if defined(Q_OS_SYMBIAN)
    if (rect.height() > 240)
        scaleFactor = 1;
#endif

    QString str = QString::number(n);
    if (str.length() == 1)
        str.prepend("0");

    QFont font;
    font.setFamily("Helvetica");
    int fontHeight = scaleFactor * 0.55 * rect.height();
    font.setPixelSize(fontHeight);
    font.setBold(true);

    QPixmap pixmap(rect.size() * scaleFactor);
    pixmap.fill(Qt::transparent);

    QLinearGradient gradient(QPoint(0, 0), QPoint(0, pixmap.height()));
    gradient.setColorAt(0.00, QColor(128, 128, 128));
    gradient.setColorAt(0.49, QColor(64, 64, 64));
    gradient.setColorAt(0.51, QColor(128, 128, 128));
    gradient.setColorAt(1.00, QColor(16, 16, 16));

    QPainter p;
    p.begin(&pixmap);
    p.setFont(font);
    QPen pen;
    pen.setBrush(QBrush(gradient));
    p.setPen(pen);
    p.drawText(pixmap.rect(), Qt::AlignCenter, str);
    p.end();

    return pixmap.scaledToWidth(width(), Qt::SmoothTransformation);
}

void Digits::preparePixmap() {
    m_pixmap = QPixmap(size());
    m_pixmap.fill(Qt::transparent);
    QPainter p;
    p.begin(&m_pixmap);
    p.drawPixmap(0, 0, drawDigits(m_number, rect()));
    p.end();
}

void Digits::resizeEvent(QResizeEvent*) {
    preparePixmap();
    update();
}

void Digits::paintStatic() {
    QPainter p(this);
    p.fillRect(rect(), Qt::black);

    int pad = width() / 10;
    drawFrame(&p, rect().adjusted(pad, pad, -pad, -pad));
    p.drawPixmap(0, 0, m_pixmap);
}

void Digits::paintSlide() {
    QPainter p(this);
    p.fillRect(rect(), Qt::black);

    int pad = width() / 10;
    QRect fr = rect().adjusted(pad, pad, -pad, -pad);
    drawFrame(&p, fr);
    p.setClipRect(fr);

    int y = height() * m_animator.currentFrame() / 100;
    p.drawPixmap(0, y, m_lastPixmap);
    p.drawPixmap(0, y - height(), m_pixmap);
}

void Digits::paintFlip() {
    QPainter p(this);
#if !defined(Q_OS_SYMBIAN)
    p.setRenderHint(QPainter::SmoothPixmapTransform, true);
    p.setRenderHint(QPainter::Antialiasing, true);
#endif
    p.fillRect(rect(), Qt::black);

    int hw = width() / 2;
    int hh = height() / 2;

    // behind is the new pixmap
    int pad = width() / 10;
    QRect fr = rect().adjusted(pad, pad, -pad, -pad);
    drawFrame(&p, fr);
    p.drawPixmap(0, 0, m_pixmap);

    int index = m_animator.currentFrame();

    if (index <= 50) {

        // the top part of the old pixmap is flipping
        int angle = -180 * index / 100;
        QTransform transform;
        transform.translate(hw, hh);
        transform.rotate(angle, Qt::XAxis);
        p.setTransform(transform);
        drawFrame(&p, fr.adjusted(-hw, -hh, -hw, -hh));
        p.drawPixmap(-hw, -hh, m_lastPixmap);

        // the bottom part is still the old pixmap
        p.resetTransform();
        p.setClipRect(0, hh, width(), hh);
        drawFrame(&p, fr);
        p.drawPixmap(0, 0, m_lastPixmap);
    } else {

        p.setClipRect(0, hh, width(), hh);

        // the bottom part is still the old pixmap
        drawFrame(&p, fr);
        p.drawPixmap(0, 0, m_lastPixmap);

        // the bottom part of the new pixmap is flipping
        int angle = 180 - 180 * m_animator.currentFrame() / 100;
        QTransform transform;
        transform.translate(hw, hh);
        transform.rotate(angle, Qt::XAxis);
        p.setTransform(transform);
        drawFrame(&p, fr.adjusted(-hw, -hh, -hw, -hh));
        p.drawPixmap(-hw, -hh, m_pixmap);

    }

}

void Digits::paintRotate() {
    QPainter p(this);

    int pad = width() / 10;
    QRect fr = rect().adjusted(pad, pad, -pad, -pad);
    drawFrame(&p, fr);
    p.setClipRect(fr);

    int angle1 = -180 * m_animator.currentFrame() / 100;
    int angle2 = 180 - 180 * m_animator.currentFrame() / 100;
    int angle = (m_animator.currentFrame() <= 50) ? angle1 : angle2;
    QPixmap pix = (m_animator.currentFrame() <= 50) ? m_lastPixmap : m_pixmap;

    QTransform transform;
    transform.translate(width() / 2, height() / 2);
    transform.rotate(angle, Qt::XAxis);

    p.setTransform(transform);
    p.setRenderHint(QPainter::SmoothPixmapTransform, true);
    p.drawPixmap(-width() / 2, -height() / 2, pix);
}

void Digits::paintEvent(QPaintEvent *event) {
    Q_UNUSED(event);
    if (m_animator.state() == QTimeLine::Running) {
        if (m_transition == Slide)
            paintSlide();
        if (m_transition == Flip)
            paintFlip();
        if (m_transition == Rotate)
            paintRotate();
    } else {
        paintStatic();
    }
}
