/****************************************************************************
**
** Copyright (C) 2009 Alvaro Manera.
** Contact: alblurrow@yute.com
**
** This file is part of weiight program
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 or 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "balanceboard.h"
#include <QDebug>

#define toggle_bit(bf,b)	\
        (bf) = ((bf) & b)		\
               ? ((bf) & ~(b))	\
               : ((bf) | (b))

#define NUMBER_ITERATIONS  100

BalanceBoard::BalanceBoard() :
            mState(BalanceBoard::STOP)
{
}



void BalanceBoard::run()
{

    switch (mState) {

        case BalanceBoard::STOP:
            break;

        case BalanceBoard::INIT:
            init_run();
            break;

        case BalanceBoard::STATE:
            weight_run();
            break;
        case BalanceBoard::CLOSE:
            close_run();
            break;
        default:
            qDebug() << "Default in the select taken!!!";
            break;
        }


}


void BalanceBoard::init_run()
{

    unsigned char led_state = 0;
    unsigned char rpt_mode = 0;


    /* Connect to the wiimote */
    qDebug("Put Wiimote in discoverable mode now (press 1+2)...");
    if (!(wiimote = cwiid_open(&bdaddr, 0))) {
        qDebug("Unable to connect to wiimote\n");
        emit error(ERR_NO_CONNECTION);
        return;
    }

    toggle_bit(rpt_mode, CWIID_RPT_EXT);
    if (cwiid_set_rpt_mode(wiimote, rpt_mode)) {
        qDebug("Error setting report mode");
        emit error(ERR_SET_RPT_MODE);
        return;
    }

    toggle_bit(led_state, CWIID_LED1_ON);
    if (cwiid_set_led(wiimote, led_state)) {
        qDebug("Error setting LED");
        emit error(ERR_SET_LED);
        return;
    }

    if (cwiid_get_balance_cal(wiimote, &bal)) {
        qDebug("Error getting balance calibration");
        emit error(ERR_GET_CALIBRATION);
        return;
    }

    emit  initDone();
    mState = BalanceBoard::STOP;
}


bool BalanceBoard::init(QString addr)
{


    if (isRunning()) {
        return false;
    }

    if (addr.isEmpty()) {
        bdaddr = *BDADDR_ANY;
    } else {
        str2ba(addr.toLatin1().data(), &bdaddr);
    }

    mState = BalanceBoard::INIT;
    start();

    return true;
}

void BalanceBoard::weight_run()
{
    struct cwiid_state state;
    float weight=0.0;
    int i;

    for ( i = 0; i< NUMBER_ITERATIONS; i++) {

        if (cwiid_get_state(wiimote, &state)) {
            qDebug("Error getting state\n");
            emit error(ERR_GET_STATE);
        }
        if (state.ext_type == CWIID_EXT_BALANCE) {
            weight += calculate_weight(&state, &bal);

        } else {
            qDebug("Not a balance board\n");
            emit error(ERR_NOT_BALANCE_BOARD);
        }
        msleep(30);
    }

    emit weightDone(weight / NUMBER_ITERATIONS);    
    mState = BalanceBoard::STOP;
}

bool BalanceBoard::weight()
{

    if (isRunning()) {
        return false;
    }

    mState = BalanceBoard::STATE;
    start();
    return true;
}

bool BalanceBoard::close()
{

    if (isRunning()) {
        return false;
    }

    mState = BalanceBoard::CLOSE;
    start();
    return true;

}

void BalanceBoard::close_run()
{

    if (cwiid_close(wiimote)) {
        qDebug("Error on wiimote disconnect");
        emit error(ERR_DISCONNECT);
    }

    emit closeDone();
    mState = BalanceBoard::STOP;

}
float BalanceBoard::weight_interpolate(uint16_t value, uint16_t *cal) {

    if (value < cal[1]) {
        return  17.0 * (value - cal[0]) / (cal[1] - cal[0]);
    } else {
        return 17.0 * (value - cal[1]) / (cal[2] - cal[1]) + 17.0;
    }
}

float BalanceBoard::calculate_weight(struct cwiid_state *state, struct balance_cal *calibration) {


    float weight;
    weight = weight_interpolate(state->ext.balance.right_top, calibration->right_top);
    weight += weight_interpolate(state->ext.balance.right_bottom, calibration->right_bottom);
    weight += weight_interpolate(state->ext.balance.left_top, calibration->left_top);
    weight += weight_interpolate(state->ext.balance.left_bottom, calibration->left_bottom);
    return weight;

}
