/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "ImportExport.h"

#include <iostream>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>

#include <Lum/Config/Config.h>

#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Msg.h>

#include "Configuration.h"
#include "Util.h"

#include <Lum/Base/String.h>

#include <Lum/Manager/FileSystem.h>

void Import(Lum::Dialog *dialog)
{
  Lum::Base::Path        path;
  std::wstring           dir;
  std::wstring           file;
  Lum::Config::ErrorList errors;
  Lum::Config::Node      *top;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDocumentDir,
                                                      dir)) {
    Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                   dir);
  }

  path.SetNativeDirAndFile(dir,L"WeightJinni-Export.xml");

  file=path.GetPath();
  if (!Lum::Dlg::File::GetFile(dialog->GetWindow(),
                               _(L"IMPORT_FILE_TITLE",L"Select import file..."),
                               file)) {
    return;
  }

  top=Lum::Config::LoadConfigFromXMLFile(file,errors);

  if (top==NULL || top->GetName()!=L"WeightJinni-ImportExport") {
    for (std::list<Lum::Config::Error>::const_iterator iter=errors.begin();
         iter!=errors.end();
         ++iter) {
      std::cerr << iter->GetRow() << "," << iter->GetColumn() << ": " << Lum::Base::WStringToString(iter->GetError()) << std::endl;
    }
    Lum::Dlg::Msg::ShowOk(dialog,
                          _(L"IMPORT_FILE_ERROR_TITLE",
                            L"Cannot load import/export file!"),
                          _(L"IMPORT_FILE_ERROR_TEXT",
                            L"Either the file does not exists,\n"
                            L"is not a valid XML file or is not\n"
                            L"a valid WeightJinni import/export file!\n"
                            L"\n"
                            L"Please check the file!"));
    delete top;
    return;
  }

  for (std::list<Lum::Config::Node*>::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    if ((*iter)->GetName()==L"food") {
      Food food;

      if ((*iter)->GetAttribute(L"name",food.name) &&
          (*iter)->GetAttribute(L"unit",food.unit) &&
          (*iter)->GetAttribute(L"value",food.value)) {
        std::set<Food>::const_iterator hit;

        hit=data.foods.find(food);
        if (hit==data.foods.end()) {
          data.foods.insert(food);
        }
        else if (hit->IsIdentical(food)) {
          // ignore already existing entry...
        }
        else {
          size_t       result;
          std::wstring text=_(L"IMPORT_CONFLICT_TEXT",
                              L"An entry in the import file also exists in the local database!\n"
                              L"\n"
                              L"Which version to you want to keep?\n"
                              L"\n"
                              L"Original: Name '%on', unit '%ou', value '%ov'\n"
                              L"Imported: Name '%in', unit '%iu', value '%iv'");

          Lum::Base::L10N::Substitute(text,L"%on",hit->name);
          Lum::Base::L10N::Substitute(text,L"%ou",hit->unit);
          Lum::Base::L10N::Substitute(text,L"%ov",AmountToString(hit->value));

          Lum::Base::L10N::Substitute(text,L"%in",food.name);
          Lum::Base::L10N::Substitute(text,L"%iu",food.unit);
          Lum::Base::L10N::Substitute(text,L"%iv",AmountToString(food.value));

          result=Lum::Dlg::Msg::Ask(dialog,
                                    _(L"IMPORT_CONFLICT_TITLE",
                                      L"Conflicting food entries!"),
                                    text,
                                    _(L"IMPORT_CONFLICT_BUTTONS",
                                      L"Keep _original*|Keep _imported|_Stop import^"));

          switch (result) {
          case 0:
            break;
          case 1:
            data.foods.erase(hit);
            data.foods.insert(food);
            break;
          case 2:
          default:
            delete top;
            return;
          }
        }
      }
    }
  }

  delete top;
}

void Export(Lum::Dialog *dialog)
{
  Lum::Config::Node *top;
  Lum::Base::Path   path;
  std::wstring      dir;
  std::wstring      file;
  bool              res;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDocumentDir,
                                                      dir)) {
    Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::userDir,
                                                   dir);
  }

  path.SetNativeDirAndFile(dir,L"WeightJinni-Export.xml");

  file=path.GetPath();
  if (!Lum::Dlg::File::SaveFile(dialog->GetWindow(),
                                _(L"EXPORT_FILE_TITLE",L"Select export file..."),
                                file)) {
    return;
  }

  top=new Lum::Config::Node();
  top->SetName(L"WeightJinni-ImportExport");

  for (std::set<Food>::const_iterator iter=data.foods.begin(); iter!=data.foods.end(); ++iter) {
    Lum::Config::Node *entry;

    entry=new Lum::Config::Node(L"food");
    entry->SetAttribute(L"name",iter->name);
    entry->SetAttribute(L"unit",iter->unit);
    entry->SetAttribute(L"value",iter->value);

    top->Add(entry);
  }

  res=Lum::Config::SaveConfigToXMLFile(file,top);

  delete top;
}

