/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Diagram.h"

#include <cmath>
#include <cstdlib>

#include <Lum/Base/String.h>

#include "Configuration.h"
#include "Util.h"

// We resync our self with the date on the first draw
static bool adjusted=false;

double exponent(double x)
{
  if (x>=0) {
    return floor(log10(x));
  }
  else {
    return floor(log10(-x));
  }
}

Diagram::Diagram()
 : backgroundColor(1.0,1.0,1.0,Lum::OS::display->whiteColor),
   warnColor(1.0,0.0,0.0,Lum::OS::display->blackColor),
   limitColor(1.0,0.0,0.0,Lum::OS::display->blackColor),
   valueGoodColor(0.95,0.95,0.95,Lum::OS::display->blackColor),
   valueGoodFrameColor(0.8,0.8,0.8,Lum::OS::display->blackColor),
   valueBadColor(1.0,0.82,0.82,Lum::OS::display->blackColor),
   valueBadFrameColor(0.8,0.58,0.58,Lum::OS::display->blackColor),
   labelColor(0.3,0.3,0.3,Lum::OS::display->blackColor),
   scaleColor(0.9,0.9,0.9,Lum::OS::display->blackColor),
   font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote)),
   mode(modeFullDay)
{
  SetBackground(Lum::OS::display->GetFill(Lum::OS::Display::boxedBackgroundFillIndex));

  Observe(topWeight);
  Observe(bottomWeight);
  Observe(maxWeight);
  Observe(minWeight);

  Observe(topValue);
  Observe(bottomValue);
  Observe(maxValue);
  Observe(minValue);

  // For updating the visible area to the given date
  Observe(date);
}

bool Diagram::RequiresKnobs() const
{
  return false;
}

void Diagram::UpdateDimensions()
{
  size_t dayWidth=GetDayWidth();

  hAdjustment->SetDimension(width,std::max(width,data.GetDaysCovered()*dayWidth));
  hAdjustment->SetStepSize(dayWidth);
}

void Diagram::CalcSize()
{
  size_t digitWidth=0;

  fullDayWidth=0;
  smallDayWidth=0;

  for (size_t i=0; i<10; i++) {
    digitWidth=std::max(digitWidth,font->StringWidth(Lum::Base::NumberToWString(i)));
  }

  for (size_t i=0; i<7; i++) {
    fullDayWidth=std::max(fullDayWidth,font->StringWidth(Lum::Base::Calendar::GetWeekDayNameShort(i)));
  }

  for (size_t i=0; i<7; i++) {
    fullDayWidth=std::max(fullDayWidth,digitWidth+
                      font->StringWidth(L". ")+
                      font->StringWidth(Lum::Base::Calendar::GetMonthNameShort(i)));
  }

  fullDayWidth=std::max(fullDayWidth,4*digitWidth)+2;
  smallDayWidth=2*digitWidth+font->StringWidth(L".")+2;
  weightUnitWidth=4*digitWidth+font->StringWidth(L".");
  valueUnitWidth=3*digitWidth+font->StringWidth(L".");

  width=100;
  height=100;
  minWidth=width;
  minHeight=height;

  Scrollable::CalcSize();
}

int Diagram::TransformWeight(double y)
{
  return (int)floor(height-(y-bottomWeight->Get())*(height-1)/(topWeight->Get()-bottomWeight->Get()));
}

int Diagram::TransformValue(double y)
{
  return (int)floor(height-(y-bottomValue->Get())*(height-1)/(topValue->Get()-bottomValue->Get()));
}

size_t Diagram::GetDayWidth() const
{
  switch (mode) {
  case modeFullDay:
    return fullDayWidth;
  case modeSmallDay:
    return smallDayWidth;
  case modeOverview:
    return 6;
  default:
    assert(false);
  }
}

void Diagram::PrintData(Lum::OS::DrawInfo* draw,
                        size_t dayWidth,
                        const Lum::Base::Calendar& start,
                        const Lum::Base::Calendar& end)
{
  int                                                xPos=this->x+((hAdjustment->GetTop()-1)/dayWidth)*
                                                          dayWidth-(hAdjustment->GetTop()-1);
  std::map<Lum::Base::Calendar,Day*>::const_iterator startIter;
  std::map<Lum::Base::Calendar,Day*>::const_iterator iter;
  bool                                               pointsDrawn=false; // We have drawn at least one point
  bool                                               pastEnd=false;     // We have beenmoved past the visible end
  bool                                               finished=false;    // We have drawn one point pastthe visible end and can now finish
  int                                                px=0,py=0;         // Coordinates of previous point
  Lum::Base::Calendar                                day;
  size_t                                             lastWeight=0;

  startIter=data.data.lower_bound(start);
  while (startIter!=data.data.begin()) {
    bool hasPoint=false;

    --startIter;

    for (std::list<Entry>::const_iterator entry=startIter->second->entries.begin();
         entry!=startIter->second->entries.end();
         ++entry) {
      if (entry->item.empty()) {
        hasPoint=true;
        break;
      }
    }

    if (hasPoint) {
      break;
    }
  }

  for (iter=startIter; iter!=data.data.end() && !finished; ++iter) {
    int   days;
    Value value=0;

    //
    // Detect first weight points to the left and right of the
    // diagram borders
    //

    days=0;
    day=start;
    if (day<=iter->first) {
      while (day!=iter->first) {
        day.AddDays(1);
        days++;
      }
    }
    else {
      while (day!=iter->first) {
        day.AddDays(-1);
        days--;
      }
    }

    //
    // Draw value rectangles
    //

    for (std::list<Entry>::const_iterator entry=iter->second->entries.begin();
         entry!=iter->second->entries.end();
         ++entry) {
      if (!entry->item.empty()) {
        value+=entry->amount*entry->value;
      }
    }

    if (value>0) {
      value/=10;
      if ((size_t)value>maxValue->Get() || (size_t)value<minValue->Get()) {
        draw->PushForeground(valueBadColor);
      }
      else {
        draw->PushForeground(valueGoodColor);
      }
      draw->FillRectangle(xPos+days*dayWidth,this->y+TransformValue(value),
                          dayWidth,height-TransformValue(value));
      draw->PopForeground();

      if ((size_t)value>maxValue->Get() || (size_t)value<minValue->Get()) {
        draw->PushForeground(valueBadFrameColor);
      }
      else {
        draw->PushForeground(valueGoodFrameColor);
      }
      draw->DrawRectangle(xPos+days*dayWidth,this->y+TransformValue(value),
                          dayWidth,height-TransformValue(value));
      draw->PopForeground();
    }
  }

  pastEnd=false;
  for (iter=startIter; iter!=data.data.end() && !finished; ++iter) {
    int days;

    //
    // Detect first weight points to the left and right of the
    // diagram borders
    //

    if (iter->first>end) {
      pastEnd=true;
    }

    days=0;
    day=start;
    if (day<=iter->first) {
      while (day!=iter->first) {
        day.AddDays(1);
        days++;
      }
    }
    else {
      while (day!=iter->first) {
        day.AddDays(-1);
        days--;
      }
    }

    //
    // Draw weight points
    //

    for (std::list<Entry>::const_iterator entry=iter->second->entries.begin();
         entry!=iter->second->entries.end();
         ++entry) {
      if (entry->item.empty()) {
        // Draw weight points and connect points by lines
        if (entry->amount>maxWeight->Get() ||
            entry->amount<minWeight->Get()) {
          draw->PushForeground(warnColor);
        }
        else {
          draw->PushForeground(Lum::OS::Display::blackColor);
        }

        int cx,cy; // Coordinates of current point

        cx=xPos+days*dayWidth+(entry->time.GetHour()*dayWidth)/24;
        cy=this->y+TransformWeight(entry->amount);

        draw->FillArc(cx-3,
                      cy-3,
                      6,6,
                      0,360*64);

        draw->PopForeground();

        // Draw line between previous and current weight point
        if (pointsDrawn) {
          int lx1, ly1, lx2, ly2;

          // Project points outside the diagram on the axis of the diagram
          // to avoid overflow errors and similar...
          lx1=px;
          ly1=py;
          lx2=cx;
          ly2=cy;

          if (!(lx1<this->x && lx2<this->x)) {
            // Point outside diagram: Interpolate position on left diagram border
            if (lx1<this->x) {
              lx1=this->x;
              ly1=((lx1-px)*(cy-py))/(cx-px)+py;
            }

            // Point outside diagram: Interpolate position on right diagram border
            if (lx2>this->x+(int)this->width-1) {
              if (cx!=px) {
                lx2=this->x+(int)this->width-1;
                ly2=((lx2-px)*(cy-py))/(cx-px)+py;
              }
              else {
                lx2=this->x+(int)this->width-1;
                ly2=ly1;
              }
            }

            if(lastWeight<minWeight->Get() || lastWeight>maxWeight->Get() ||
               entry->amount<minWeight->Get() || entry->amount>maxWeight->Get()) {
              draw->PushForeground(warnColor);
            }
            else {
              draw->PushForeground(Lum::OS::Display::blackColor);
            }

            draw->PushPen(2,Lum::OS::DrawInfo::penRound);
            draw->DrawLine(lx1,ly1,lx2,ly2);
            draw->PopPen();
            draw->PopForeground();
          }
        }

        pointsDrawn=true;
        px=cx;
        py=cy;
        lastWeight=entry->amount;

        if (pastEnd) {
          finished=true;
        }
      }
    }
  }
}

void Diagram::PrintHorizontalLabel(Lum::OS::DrawInfo* draw,
                                   size_t dayWidth,
                                   const Lum::Base::Calendar& start)
{
  std::wstring        tmp;
  Lum::Base::Calendar day;

  draw->PushClip(this->x+weightUnitWidth,this->y,this->width-weightUnitWidth-valueUnitWidth,this->height);
  draw->PushFont(font);
  draw->PushForeground(labelColor);

  day=start;
  int xPos=this->x+((hAdjustment->GetTop()-1)/dayWidth)*dayWidth-(hAdjustment->GetTop()-1);
  while (xPos<this->x+(int)width) {
    switch (mode) {
    case modeFullDay:
      tmp=Lum::Base::Calendar::GetWeekDayNameShort(day.GetDayOfWeek());
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,this->y+height-3*font->height+font->ascent,tmp);

      tmp=Lum::Base::NumberToWString(day.GetDayOfMonth())+L". "+Lum::Base::Calendar::GetMonthNameShort(day.GetMonth());
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,this->y+height-2*font->height+font->ascent,tmp);

      tmp=Lum::Base::NumberToWString(day.GetYear());
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,this->y+height-font->height+font->ascent,tmp);
      break;
    case modeSmallDay:
      tmp=Lum::Base::NumberToWString(day.GetDayOfMonth())+L".";
      draw->DrawString(xPos+(dayWidth-font->StringWidth(tmp))/2,
                       this->y+font->ascent+height-font->height,
                       tmp);
      break;
    case modeOverview:
      if (day.GetDayOfYear()==1) {
        tmp=Lum::Base::NumberToWString(day.GetYear());
        draw->DrawString(xPos,this->y+height-2*font->height+font->ascent,tmp);
      }
      if (day.GetDayOfMonth()==1) {
        tmp=Lum::Base::Calendar::GetMonthName(day.GetMonth());
        draw->DrawString(xPos,this->y+height-font->height+font->ascent,tmp);
      }
      break;
    }

    day.AddDays(1);
    xPos+=dayWidth;
  }

  draw->PopForeground();
  draw->PopFont();
  draw->PopClip();

}

void Diagram::PrintVerticalScale(Lum::OS::DrawInfo* draw,
                                 size_t dayWidth,
                                 const Lum::Base::Calendar& start)
{
  Lum::Base::Calendar day;

  draw->PushClip(this->x+weightUnitWidth,
                 this->y,
                 this->width-weightUnitWidth-valueUnitWidth,
                 this->height);

  draw->PushForeground(scaleColor);

  day=start;
  int xPos=this->x+((hAdjustment->GetTop()-1)/dayWidth)*dayWidth-(hAdjustment->GetTop()-1);
  while (xPos<this->x+(int)width) {
    switch (mode) {
    case modeFullDay:
    case modeSmallDay:
      draw->DrawLine(xPos+dayWidth-1,this->y,xPos+dayWidth-1,this->y+height-1);
      break;
    default:
      break;
    }

    day.AddDays(1);
    xPos+=dayWidth;
  }

  draw->PopForeground();

  draw->PopClip();
}

/**
  Printing vertial value scale
 */
void Diagram::PrintVerticalLabel(Lum::OS::DrawInfo* draw)
{
  draw->PushForeground(labelColor);
  draw->PushFont(font);

  size_t step;
  int effFontHeight;

  if (font->height/4>2) {
    effFontHeight=font->height+(font->height/4);
  }
  else {
    effFontHeight=font->height+2;
  }
  assert(effFontHeight >= 0);

  // Weight on left side

  if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+1))>effFontHeight) {
    step=1;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+5))>effFontHeight) {
    step=5;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+10))>effFontHeight) {
    step=10;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+50))>effFontHeight) {
    step=50;
  }
  else if ((TransformWeight(bottomWeight->Get())-TransformWeight(bottomWeight->Get()+100))>effFontHeight) {
    step=100;
  }
  else {
    step=500;
  }

  for (size_t i=(bottomWeight->Get()/step)*step; i<=topWeight->Get(); i+=step) {
    std::wstring tmp=AmountToString(i);
    draw->DrawString(this->x+weightUnitWidth-font->StringWidth(tmp),
                     TransformWeight(i)-font->height/2+this->y+font->ascent,
                     tmp);
  }

  // Values on right side

  if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+1))>effFontHeight) {
    step=1;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+5))>effFontHeight) {
    step=5;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+10))>effFontHeight) {
    step=10;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+50))>effFontHeight) {
    step=50;
  }
  else if ((TransformValue(bottomValue->Get())-TransformValue(bottomValue->Get()+100))>effFontHeight) {
    step=100;
  }
  else {
    step=500;
  }

  for (size_t i=(bottomValue->Get()/step)*step; i<=topValue->Get(); i+=step) {
    std::wstring tmp=AmountToString(i);
    draw->DrawString(this->x+this->width-1-font->StringWidth(tmp),
                     TransformValue(i)-font->height/2+this->y+font->ascent,
                     tmp);
  }

  draw->PopFont();
  draw->PopForeground();
}

void Diagram::PrintLimits(Lum::OS::DrawInfo* draw)
{
  // Weigth limits
  draw->PushForeground(limitColor);
  draw->DrawLine(this->x,
                 this->y+TransformWeight(maxWeight->Get()),
                 this->x+weightUnitWidth-1,
                 this->y+TransformWeight(maxWeight->Get()));
  draw->DrawLine(this->x,
                 this->y+TransformWeight(minWeight->Get()),
                 this->x+weightUnitWidth-1,
                 this->y+TransformWeight(minWeight->Get()));
  draw->PopForeground();

  // Value limits
  draw->PushForeground(limitColor);
  draw->DrawLine(this->x+width-1-valueUnitWidth,
                 this->y+TransformValue(maxValue->Get()),
                 this->x+width-1,
                 this->y+TransformValue(maxValue->Get()));
  draw->DrawLine(this->x+width-1-valueUnitWidth,
                 this->y+TransformValue(minValue->Get()),
                 this->x+width-1,
                 this->y+TransformValue(minValue->Get()));
  draw->PopForeground();
}

void Diagram::SyncGraphWithDate()
{
  if (!date.Valid()) {
    return;
  }

  if (date->Get()<data.GetStartDate()) {
    if (GetHAdjustment()!=NULL && GetHAdjustment()->IsValid()) {
      GetHAdjustment()->ShowFirstPage();
    }
  }
  else if (date->Get()>data.GetEndDate()) {
    if (GetHAdjustment()!=NULL && GetHAdjustment()->IsValid()) {
      GetHAdjustment()->ShowLastPage();
    }
  }
  else {
    if (GetHAdjustment()!=NULL && GetHAdjustment()->IsValid()) {
      GetHAdjustment()->CenterOn((date->Get()-data.GetStartDate())*GetDayWidth()+GetDayWidth()/2+1);
    }
  }
}

void Diagram::GetDimension(size_t& width, size_t& height)
{
  size_t dayWidth=GetDayWidth();

  width=std::max(width,data.GetDaysCovered()*dayWidth);
}

void Diagram::Layout()
{
  UpdateDimensions();

  Scrollable::Layout();
}

void Diagram::Draw(Lum::OS::DrawInfo* draw,
                   int x, int y, size_t w, size_t h)
{
  Lum::Scrollable::Draw(draw,x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  size_t dayWidth=GetDayWidth();

  if (!adjusted) {
    SyncGraphWithDate();
    adjusted=true;
  }

  /* --- */

  Lum::Base::Calendar start=data.GetStartDate();
  Lum::Base::Calendar end;

  draw->PushForeground(backgroundColor);
  draw->FillRectangle(this->x,this->y,width,height);
  draw->PopForeground();

  start.AddDays((hAdjustment->GetTop()-1)/dayWidth);
  end=start;
  end.AddDays((width+dayWidth/2)/dayWidth);

  draw->PushClip(this->x,this->y,width,height);
  draw->PushClip(x,y,w,h);

  PrintVerticalScale(draw,dayWidth,start);
  PrintData(draw,dayWidth,start,end);
  PrintLimits(draw);
  PrintHorizontalLabel(draw,dayWidth,start);
  PrintVerticalLabel(draw);

  draw->PopClip();
  draw->PopClip();
}

bool Diagram::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (event.type==Lum::OS::MouseEvent::down &&
     PointIsIn(event) &&
     event.button==Lum::OS::MouseEvent::button1) {
    gestureStartX=event.x;
    gestureStartY=event.y;

    if (GetWindow()->IsDoubleClicked() &&
        PointIsIn(GetWindow()->GetLastButtonClickEvent())) {
      Lum::Base::Calendar day=data.GetStartDate();

      day.AddDays((event.x-x+hAdjustment->GetTop()-1)/GetDayWidth());

      date->Set(day);
      currentTab->Set(1);
    }
    return true;
  }
  else if (event.IsGrabEnd() && event.button==Lum::OS::MouseEvent::button1) {

    if (std::abs(gestureStartX-event.x)>=std::abs(gestureStartY-event.y) ||
        std::abs(gestureStartY-event.y)<int((height*10)/100)) {
      return false;
    }

    if (event.y>gestureStartY) {
      // down
      mode=(Mode)((mode+1)%(modeMax+1));
    }
    else {
      // up
      if (mode==0) {
        mode=modeMax;
      }
      else {
        mode=(Mode)((mode-1)%(modeMax+1));
      }
    }

    UpdateDimensions();
    Redraw();
  }
  else if (event.type==Lum::OS::MouseEvent::down &&
     PointIsIn(event) &&
     event.button==Lum::OS::MouseEvent::button4) {
   if (mode==0) {
     mode=modeMax;
   }
   else {
     mode=(Mode)((mode-1)%(modeMax+1));
   }

   UpdateDimensions();
   Redraw();

   return true;
  }
  else if (event.type==Lum::OS::MouseEvent::down &&
          PointIsIn(event) &&
          event.button==Lum::OS::MouseEvent::button5) {
   mode=(Mode)((mode+1)%(modeMax+1));

   UpdateDimensions();
   Redraw();

   return true;
  }

  return false;
}

void Diagram::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hAdjustment->GetTopModel()) {
    Redraw();
  }
  else if (model==topWeight ||
           model==bottomWeight ||
           model==minWeight ||
           model==maxWeight ||
           model==topValue ||
           model==bottomValue ||
           model==minValue ||
           model==maxValue) {
    Redraw();
  }
  else if (model==date) {
    SyncGraphWithDate();
  }

  Scrollable::Resync(model,msg);
}

