/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Configuration.h"

#include <iostream>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Config/Config.h>

Data                      data;

Lum::Model::CalendarRef   date=new Lum::Model::Calendar();
Lum::Model::TimeRef       timestamp=new Lum::Model::Time(Lum::Base::Time());

Lum::Model::ULongRef      currentTab=new Lum::Model::ULong(1);

Lum::Model::ULongRef      topWeight=new Lum::Model::ULong(1500);
Lum::Model::ULongRef      bottomWeight=new Lum::Model::ULong(0);
Lum::Model::ULongRef      maxWeight=new Lum::Model::ULong(1500);
Lum::Model::ULongRef      minWeight=new Lum::Model::ULong(0);

Lum::Model::ULongRef      topValue=new Lum::Model::ULong(400);
Lum::Model::ULongRef      bottomValue=new Lum::Model::ULong(0);
Lum::Model::ULongRef      maxValue=new Lum::Model::ULong(250);
Lum::Model::ULongRef      minValue=new Lum::Model::ULong(150);

Lum::Model::ULongRef      goodValueLimit=new Lum::Model::ULong(10);
Lum::Model::ULongRef      badValueLimit=new Lum::Model::ULong(30);

Lum::Model::ULongRef      refValue=new Lum::Model::ULong(1300);

Lum::Model::StringRef     valueUnit=new Lum::Model::String(L"badness");
Lum::Model::StringRef     weightUnit=new Lum::Model::String(L"kg");

Lum::Model::ULongRef      trendType=new Lum::Model::ULong(trendTwoWeeks);
Lum::Model::ULongRef      trendDirection=new Lum::Model::ULong(1);

bool Entry::operator==(const Entry& entry) const
{
  return time==entry.time && item==entry.item && amount==entry.amount;
}

Day* Data::GetDay(const Lum::Base::Calendar& date) const
{
  std::map<Lum::Base::Calendar,Day*>::const_iterator iter;

  iter=data.find(date);

  if (iter==data.end()) {
    return NULL;
  }

  return iter->second;
}

const Lum::Base::Calendar Data::GetStartDate() const
{
  if (data.begin()!=data.end()) {
    return data.begin()->first;
  }
  else {
    Lum::Base::Calendar now;

    return now;
  }
}

const Lum::Base::Calendar Data::GetEndDate() const
{
  if (data.begin()!=data.end()) {
    return data.rbegin()->first;
  }
  else {
    Lum::Base::Calendar now;

    return now;
  }
}

size_t Data::GetDaysCovered() const
{
  if (data.begin()==data.end()) {
    return 1;
  }

  Lum::Base::Calendar start=data.begin()->first;
  Lum::Base::Calendar end=data.rbegin()->first;
  size_t              daysCovered=0;

  while (start.GetYear()<end.GetYear()) {
    if (start.IsLeapYear()) {
      daysCovered=366-start.GetDayOfYear();
    }
    else {
      daysCovered=365-start.GetDayOfYear();
    }

    start.SetDate(1,1,start.GetYear()+1);
  }

  daysCovered+=end.GetDayOfYear()-start.GetDayOfYear()+1;

  //std::cout << Lum::Base::WStringToString(start.GetLocaleDate()) << "-" << Lum::Base::WStringToString(end.GetLocaleDate()) << " " << daysCovered << std::endl;

  return daysCovered;
}

Entry* Data::AddEntry(const Lum::Base::Calendar& date, const Entry& entry)
{
  std::map<Lum::Base::Calendar,Day*>::iterator iter;
  Day                                          *day;

  iter=data.find(date);
  if (iter!=data.end()) {
    day=iter->second;
  }
  else {
    day=new Day();
    data[date]=day;
  }

  day->entries.push_back(entry);

  return &day->entries.back();
}

void Data::DeleteEntry(const Lum::Base::Calendar& date, const Entry* entry)
{
  std::map<Lum::Base::Calendar,Day*>::iterator iter;
  Day                                          *day;

  iter=data.find(date);

  assert(iter!=data.end());

  day=iter->second;

  for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
    if (&(*iter)==entry) {
      day->entries.erase(iter);
      break;
    }
  }
}

const Food* Data::AddFood(const Food& food)
{
  std::pair<std::set<Food>::iterator,bool> result;

  result=foods.insert(food);

  if (result.second) {
    return &*result.first;
  }
  else {
    return NULL;
  }
}

void Data::DeleteFood(const Food& food)
{
  foods.erase(food);
}

bool Food::IsIdentical(const Food& entry) const
{
  return name==entry.name && unit==entry.unit && value==entry.value;
}

bool Food::operator==(const Food& entry) const
{
  return name==entry.name;
}

bool Food::operator<(const Food& entry) const
{
  return name<entry.name;
}

bool LoadConfig(const std::wstring& file)
{
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path(file);

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"WeightJinni") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid config file!" << std::endl;
    delete top;
    return false;
  }

  data.foods.clear();

  for (std::map<Lum::Base::Calendar,Day*>::iterator iter=data.data.begin(); iter!=data.data.end(); ++iter) {
    delete iter->second;
  }
  data.data.clear();

  for (std::list<Lum::Config::Node*>::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    if ((*iter)->GetName()==L"Food") {
      for (std::list<Lum::Config::Node*>::const_iterator iter2=(*iter)->GetChildren().begin();
           iter2!=(*iter)->GetChildren().end();
           ++iter2) {
        if ((*iter2)->GetName()==L"food") {
          Food food;

          if ((*iter2)->GetAttribute(L"name",food.name) &&
              (*iter2)->GetAttribute(L"unit",food.unit) &&
              (*iter2)->GetAttribute(L"value",food.value)) {
            data.foods.insert(food);
          }
        }
      }
    }
    else if ((*iter)->GetName()==L"Data") {
      for (std::list<Lum::Config::Node*>::const_iterator iter2=(*iter)->GetChildren().begin();
           iter2!=(*iter)->GetChildren().end();
           ++iter2) {
        if ((*iter2)->GetName()==L"day") {
          Lum::Base::Calendar date;
          std::wstring        dateString;

          if ((*iter2)->GetAttribute(L"date",dateString) && date.SetISO8601(dateString)) {
            Day *day;

            day=new Day();

            for (std::list<Lum::Config::Node*>::const_iterator iter3=(*iter2)->GetChildren().begin();
                 iter3!=(*iter2)->GetChildren().end();
                 ++iter3) {
              if ((*iter3)->GetName()==L"entry") {
                Entry        entry;
                std::wstring timeString;

                if ((*iter3)->GetAttribute(L"time",timeString) &&
                    (*iter3)->GetAttribute(L"unit",entry.unit) &&
                    (*iter3)->GetAttribute(L"value",entry.value) &&
                    (*iter3)->GetAttribute(L"amount",entry.amount) &&
                    entry.time.SetISO8601(timeString)) {
                  (*iter3)->GetAttribute(L"item",entry.item);
                }

                day->entries.push_back(entry);
              }
            }

            data.data[date]=day;
          }

        }
      }
    }
    else if ((*iter)->GetName()==L"Diagram") {
      size_t       s;
      std::wstring str;

      if ((*iter)->GetAttribute(L"topWeight",s)) {
        topWeight->Set(s);
      }
      if ((*iter)->GetAttribute(L"bottomWeight",s)) {
        bottomWeight->Set(s);
      }
      if ((*iter)->GetAttribute(L"maxWeight",s)) {
        maxWeight->Set(s);
      }
      if ((*iter)->GetAttribute(L"minWeight",s)) {
        minWeight->Set(s);
      }

      if ((*iter)->GetAttribute(L"topValue",s)) {
        topValue->Set(s);
      }
      if ((*iter)->GetAttribute(L"bottomValue",s)) {
        bottomValue->Set(s);
      }
      if ((*iter)->GetAttribute(L"maxValue",s)) {
        maxValue->Set(s);
      }
      if ((*iter)->GetAttribute(L"minValue",s)) {
        minValue->Set(s);
      }

      if ((*iter)->GetAttribute(L"goodValueLimit",s)) {
        goodValueLimit->Set(s);
      }
      if ((*iter)->GetAttribute(L"badValueLimit",s)) {
        badValueLimit->Set(s);
      }

      if ((*iter)->GetAttribute(L"refValue",s)) {
        refValue->Set(s);
      }

      if ((*iter)->GetAttribute(L"valueUnit",str)) {
        valueUnit->Set(str);
      }
      if ((*iter)->GetAttribute(L"weightUnit",str)) {
        weightUnit->Set(str);
      }

      if ((*iter)->GetAttribute(L"trendType",s)) {
        trendType->Set(s);
      }
      if ((*iter)->GetAttribute(L"trendDirection",s)) {
        trendDirection->Set(s);
      }
    }
  }

  delete top;

  return true;
}

bool SaveConfig(const std::wstring& file)
{
  Lum::Config::Node *top,*node;
  Lum::Base::Path   path(file);
  std::wstring      config;
  bool              res;

  top=new Lum::Config::Node();
  top->SetName(L"WeightJinni");

  node=new Lum::Config::Node(L"Diagram");
  node->SetAttribute(L"bottomWeight",bottomWeight->Get());
  node->SetAttribute(L"topWeight",topWeight->Get());
  node->SetAttribute(L"maxWeight",maxWeight->Get());
  node->SetAttribute(L"minWeight",minWeight->Get());

  node->SetAttribute(L"bottomValue",bottomValue->Get());
  node->SetAttribute(L"topValue",topValue->Get());
  node->SetAttribute(L"maxValue",maxValue->Get());
  node->SetAttribute(L"minValue",minValue->Get());

  node->SetAttribute(L"goodValueLimit",goodValueLimit->Get());
  node->SetAttribute(L"badValueLimit",badValueLimit->Get());

  node->SetAttribute(L"refValue",refValue->Get());

  node->SetAttribute(L"valueUnit",valueUnit->Get());
  node->SetAttribute(L"weightUnit",weightUnit->Get());

  node->SetAttribute(L"trendType",trendType->Get());
  node->SetAttribute(L"trendDirection",trendDirection->Get());

  top->Add(node);

  if (!data.foods.empty()) {
    node=new Lum::Config::Node(L"Food");
    top->Add(node);

    for (std::set<Food>::const_iterator iter=data.foods.begin(); iter!=data.foods.end(); ++iter) {
      Lum::Config::Node *entry;

      entry=new Lum::Config::Node(L"food");
      entry->SetAttribute(L"name",iter->name);
      entry->SetAttribute(L"unit",iter->unit);
      entry->SetAttribute(L"value",iter->value);

      node->Add(entry);
    }
  }

  if (!data.data.empty()) {
    node=new Lum::Config::Node(L"Data");
    top->Add(node);

    for (std::map<Lum::Base::Calendar,Day*>::const_iterator iter=data.data.begin();
         iter!=data.data.end();
         ++iter) {
      if (iter->second->entries.size()==0) {
        continue;
      }

      Lum::Config::Node *day;

      day=new Lum::Config::Node(L"day");
      day->SetAttribute(L"date",iter->first.GetISO8601());

      for (std::list<Entry>::const_iterator iter2=iter->second->entries.begin();
           iter2!=iter->second->entries.end();
           ++iter2) {
        Lum::Config::Node *entry;

        entry=new Lum::Config::Node(L"entry");
        entry->SetAttribute(L"time",iter2->time.GetISO8601());
        if (!iter2->item.empty()) {
          entry->SetAttribute(L"item",iter2->item);
        }
        entry->SetAttribute(L"unit",iter2->unit);
        entry->SetAttribute(L"value",iter2->value);
        entry->SetAttribute(L"amount",iter2->amount);

        day->Add(entry);
      }

      node->Add(day);
    }
  }

  Lum::Base::Status status;

  status=path.CreateDirRecursive();

  if (!status) {
    std::cerr << "Cannot create config directory '" << Lum::Base::WStringToString(path.GetDir()) << "': " << Lum::Base::WStringToString(status.GetDescription()) << std::endl;
    return false;
  }

  res=Lum::Config::SaveConfigToXMLFile(path.GetPath(),top);

  delete top;

  return res;
}

