/*
 * authmanager.h - authentication manager implementation.
 *
 * Copyright (C) 2010 - Aleksandr A. Lomov.
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include "authmanager.h"
#include "authdbusadaptor.h"

/**
 * @brief Constructor.
 */
AuthManager::AuthManager(QObject *parent) :
    QObject(parent)
{
    lastWindowsID = 0;
    adapter = NULL;
    isProxySet = false;
}


/**
 * @brief Destructor
 */
AuthManager::~AuthManager()
{
    qDebug() << "AuthManager destructor START.";

    delete adapter;
    adapter = NULL;

    QList<MainWindow *> values = windows.values();

    int len = values.length();

    for (int i = 0; i < len; ++i) {
        delete values[i];
        values[i] = NULL;
    }

    QList<CaptchaWindow *> captchaValues = captchaWindows.values();

    len = captchaValues.length();

    for (int i = 0; i < len; ++i) {
        delete captchaValues[i];
        captchaValues[i] = NULL;
    }

    windows.clear();
    captchaWindows.clear();

    qDebug() << "AuthManager destructor END.";
}



/********************************************/
/************ Public functions **************/
/**
 * @brief Initialize DBus connection.
 *
 * @param parent parent object.
 *
 * return 0 on success, -1 otherwise.
 */
int AuthManager::initDBus(QObject *parent)
{
    adapter = new AuthDBusAdaptor(parent);

    if (adapter == NULL || adapter->InitDBusService() == false) {
        delete adapter;
        adapter = NULL;
        return -1;
    }


    connect(adapter, SIGNAL (setProxy(const QString &, const QString &,
                                      const QString &, const QString &)),
            SLOT (setProxy(const QString &, const QString &,
                           const QString &, const QString &)));

    connect(adapter, SIGNAL (auth(const QString &)), SLOT (auth(const QString &)));
    connect(adapter, SIGNAL (auth(const QString &, const QStringList &)),
            SLOT (auth(const QString &, const QStringList &)));

    connect(adapter, SIGNAL (shutdown()), SLOT (shutdown()));
    connect(adapter, SIGNAL (login(const QString &)), SLOT (login(const QString &)));
    connect(adapter, SIGNAL (captcha(const QString &, const QString &)),
            SLOT (captcha(const QString &, const QString &)));

    return 0;
}



/********************************************/
/************** Public slots ****************/
/**
 * @brief Set proxy.
 *
 * Function for DBus call, using to setup proxy.
 *
 * @param proxyIP IP address of proxy.
 * @param proxyPort port of proxy.
 * @param username name of user (if needed).
 * @param username password for proxy (if needed).
 */
void AuthManager::setProxy(const QString &proxyIP, const QString &proxyPort,
        const QString &username, const QString &password)
{
    qDebug() << "\nSetting proxy...(host = " << proxyIP.toAscii().constData()
        << " port = " << proxyPort.toAscii().constData()
        << " username = " << username.toAscii().constData()
        << " password = " << password.toAscii().constData() << ").";
    isProxySet = true;

    QString curProxy = proxyIP;
    QString curPort = proxyPort;

    uint port = proxyPort.toUInt();

    qDebug() << "port is " << port;

    if (proxyIP.isNull() == true || proxyIP.isEmpty() == true
           || port <= 0) {

	/* try to mannualy detect proxy settings */
        QNetworkProxyQuery npq(QUrl(QLatin1String("http://maemo.org")));
        QList<QNetworkProxy> proxyList = QNetworkProxyFactory::systemProxyForQuery(npq);
        if (proxyList.length() > 0 && proxyList.at(0) != QNetworkProxy::NoProxy)
        {
    	    curProxy = proxyList.at(0).hostName();
    	    curPort = proxyList.at(0).port();
        } else {
    	    QString var(getenv("http_proxy"));
    	    if (var.isNull()) {
		var = getenv("all_proxy");
	    }
            QRegExp regex("(http://)?(.*):(\\d*)/?");
            int pos = regex.indexIn(var);

            if(pos > -1) {
        	curProxy = regex.cap(2);
		curPort = regex.cap(3);
	    } else {
    		qDebug() << "Proxy not set(incorrect host or port).";
    		return;
	    }
        }
    }

    if (username.isNull() != true
            && username.isEmpty() != true) {
        proxy.setUser(username);
        proxy.setPassword(password);
    }

    proxy.setType(QNetworkProxy::HttpProxy);
    proxy.setHostName(curProxy);
    proxy.setPort(curPort.toUInt());

    qDebug() << "\nSetting proxy...(host = " << curProxy
        << " port = " << curPort << ").";

    QNetworkProxy::setApplicationProxy(proxy);
}


/**
 * @brief Shutdown webauth service.
 */
void AuthManager::shutdown()
{
    qDebug() << "\nShudown webauth...";
    QApplication::quit();
}



/********************************************/
/************* Private slots ****************/
/**
 * @brief Setup for simple authentication.
 *
 * @param url url for user athentication.
 */
void AuthManager::auth(const QString &url)
{

    if (!isProxySet)
	this->setProxy(QString::null, QString::null, QString::null, QString::null);

    qDebug() << "\nUsing simple auth...";
    ++lastWindowsID;
    MainWindow *mw = new MainWindow(lastWindowsID);
    windows.insert(lastWindowsID, mw);
    connect(mw, SIGNAL (closed(const int)), SLOT (windowsClosed(const int)));

    mw->startAuth(url);
}


/**
 * @brief Setup for extended authentication.
 *
 * @param url url for user athentication.
 * @param params list witn regexps.
 */
void AuthManager::auth(const QString &url, const  QStringList &params)
{
    if (!isProxySet)
	this->setProxy(QString::null, QString::null, QString::null, QString::null);

    qDebug() << "\nUsing extended auth...";

    ++lastWindowsID;
    MainWindow *mw = new MainWindow(lastWindowsID);

    windows.insert(lastWindowsID, mw);
    connect(mw, SIGNAL (closed(const int)), SLOT (windowsClosed(const int)));

    mw->startAuth(url,params);
}

/**
 * @brief Setup for captcha test.
 *
 * @param strBase64 string in base64 for show image.
 * @param clientID string with clientID
 */
void AuthManager::captcha(const QString &strBase64, const QString clientID)
{

    if (!isProxySet)
        this->setProxy(QString::null, QString::null, QString::null, QString::null);

    qDebug() << "\nUsing captcha test...";

    ++lastCaptchaWindowsID;
    CaptchaWindow *mw = new CaptchaWindow(lastCaptchaWindowsID);
    captchaWindows.insert(lastCaptchaWindowsID, mw);
    connect(mw, SIGNAL (captchaClosed(const int)), SLOT (captchaWindowsClosed(const int)));

    mw->startCaptcha(strBase64, clientID);
}



void AuthManager::login(const QString &clientID)
{

    if (!isProxySet)
        this->setProxy(QString::null, QString::null, QString::null, QString::null);

    qDebug() << "\nUsing Log In...";

    ++lastLoginWindowsID;
    AuthWindow *mw = new AuthWindow(lastLoginWindowsID);
    loginWindows.insert(lastLoginWindowsID, mw);
    connect(mw, SIGNAL (loginClosed(const int)), SLOT (authWindowsClosed(const int)));

    mw->startLogin(clientID);
}
/**
 * @brief Slot for get wondow closed signal.
 * Get window identifier and remove it(window).
 * If no opened windows when shutdown webauth.
 *
 * @param windowID window identifier.
 */
void AuthManager::windowsClosed(const int windowID)
{
    qDebug() << "Window closed (ID = " << windowID << ").";

    MainWindow *mv = windows[windowID];
    windows.remove(windowID);

//    delete(mv);
    mv = NULL;

//    if (windows.isEmpty() == true && captchaWindows.isEmpty() == true) {
//        shutdown();
//    }
}



void AuthManager::captchaWindowsClosed(const int windowID)
{
    qDebug() << "CaptchaWindow closed (ID = " << windowID << ").";

    CaptchaWindow *mv = captchaWindows[windowID];
    captchaWindows.remove(windowID);

//    delete(mv);
    mv = NULL;

//    if (windows.isEmpty() == true && captchaWindows.isEmpty() == true ) {
//        shutdown();
//    }
}

void AuthManager::authWindowsClosed(const int windowID)
{
    qDebug() << "AuthWindow closed (ID = " << windowID << ").";

    AuthWindow *mv = loginWindows[windowID];
    loginWindows.remove(windowID);

//    delete(mv);
    mv = NULL;

//  if (windows.isEmpty() == true && captchaWindows.isEmpty() == true && loginWindows.isEmpty()) {
//       shutdown();
//  }
}
