/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local webtexter includes
// These have to come before Qt is included because it redefines "signals"
#include "src/settings.h"
#include "src/http_sender.h"
#include "src/vodafone_sender.h"

// Local includes
#include "webtexter.hpp"
#include "webtexteraccountsettingsdialog.hpp"
#include "../providerplugin.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtCore/QDebug>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class WebTexterProviderPrivate : public QObject
{
  Q_OBJECT

public:
  // Properties
  AppSettings2 settings;

  Providers provider;

  // Functions
};

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

WebTexterProvider::WebTexterProvider(int provider, QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new WebTexterProviderPrivate)
{
  // Set the default provider
  d->provider = static_cast<Providers>(provider);
}

//////////////////////////////////////////////////////////////////////////

WebTexterProvider::~WebTexterProvider()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void WebTexterProvider::loadSettings(const QByteArray& data)
{
  // No data? Nothing to do
  if ( data.isEmpty() )
    return;

  QDataStream stream(data);

  QByteArray userName, password, number, proxyUrl;
  int curlTimeout;
  bool useProxyScript, saveMsg, orientationEnabled, emulator, extraLogging;

  stream >> userName
         >> password
         >> number
         >> useProxyScript
         >> saveMsg
         >> orientationEnabled
         >> emulator
         >> extraLogging
         >> proxyUrl
         >> curlTimeout;

  d->settings.setUsername(userName);
  d->settings.setPassword(password);
  d->settings.setNumber(number);
  d->settings.setUseProxyScript(useProxyScript);
  d->settings.setSaveMsg(saveMsg);
  d->settings.setOrientationEnabled(orientationEnabled);
  d->settings.setEmulator(emulator);
  d->settings.setExtraLogging(extraLogging);
  d->settings.setProxyUrl(proxyUrl);
  d->settings.setCurlTimeout(curlTimeout);
}

//////////////////////////////////////////////////////////////////////////

QByteArray WebTexterProvider::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->settings.userName()
         << d->settings.password()
         << d->settings.number()
         << d->settings.useProxyScript()
         << d->settings.saveMsg()
         << d->settings.orientationEnabled()
         << d->settings.emulator()
         << d->settings.extraLogging()
         << d->settings.proxyUrl()
         << d->settings.curlTimeout();

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool WebTexterProvider::showAccountSettingsDialog(QWidget* parent)
{
  WebTexterAccountSettingsDialog dialog(parent);
  dialog.setSettings(d->settings);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->settings = dialog.settings();
    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

void WebTexterProvider::sendMessage(const Message& message)
{
  QByteArray to = message.receivers().first().number().toUtf8();
  QByteArray msg = message.text().toUtf8();

  // Construct a AppSettings instance
  AppSettings settings;
  d->settings.setProvider(d->provider); // UGLY HACK (but works)
  d->settings.fillAppSettings(&settings);

  gint sent = send_msg(&settings, to.data(), msg.data(), NULL );

  emit sendMessageReply(sent==SUCCESS, message);

  switch(sent)
  {
  case ERROR_LOGIN:
    emit error( tr("Error logging in!") );
    break;

  case ERROR_SEND:
    emit error( tr("Unable to send message!") );
    break;

  default:
    emit error( tr("Unknown error happened!") );
    break;
  }
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> WebTexterProvider::messageTypes() const
{
  return QList<MessageType>()
            << MessageType(tr("Standard"), "standard", QIcon(), QString(), 160, 160, 1); 
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class WebTexterProviderPlugin : public QObject
                              , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

public:
  WebTexterProviderPlugin()
  {
    infoList
        << ProviderInfo(tr("Vodafone Ireland"), "d52fda6e-d253-4380-b68d-4693d4012604", QIcon(":/providers/webtexter/vodafone_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("o2 Ireland"), "a8b53cbe-22b7-4823-9da0-f91dcd3a5d4c", QIcon(":/providers/webtexter/o2_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("3 Ireland"), "9bd09e1e-7e9e-4666-b516-6bfd3a23972c", QIcon(":/providers/webtexter/three_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("Meteor Ireland"), "1839f8b7-3b89-4033-9b23-314fb9b5fc44", QIcon(":/providers/webtexter/meteor_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("Blueface Ireland"), "c3bfe4b8-2ea5-4e4a-a9de-4eeb76b72f60", QIcon(":/providers/webtexter/blueface_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("WebSMS.ru"), "7d0f8d29-efa5-4ced-a0c1-a4a3b11069ee", QIcon(":/providers/webtexter/websms.ru.png"), QLocale::RussianFederation)
        << ProviderInfo(tr("Exetel Australia"), "ceb3cb26-9a73-48da-ab07-5f642d2b8c16", QIcon(), QLocale::Australia);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& name) const
  {
    int provider = -1;

    // Now check the big list ..
    if ( name == "d52fda6e-d253-4380-b68d-4693d4012604" )
      provider = VODAFONE;
    else if ( name == "a8b53cbe-22b7-4823-9da0-f91dcd3a5d4c" )
      provider = O2;
    else if ( name == "9bd09e1e-7e9e-4666-b516-6bfd3a23972c" )
      provider = THREE;
    else if ( name == "1839f8b7-3b89-4033-9b23-314fb9b5fc44" )
      provider = METEOR;
    else if ( name == "c3bfe4b8-2ea5-4e4a-a9de-4eeb76b72f60" )
      provider = BLUEFACE;
    else if ( name == "7d0f8d29-efa5-4ced-a0c1-a4a3b11069ee" )
      provider = WEBSMSRU;
    else if ( name == "ceb3cb26-9a73-48da-ab07-5f642d2b8c16" )
      provider = EXETEL;

    // Should we create a new provider?
    if ( provider != -1 )
      return new WebTexterProvider(provider);

    return NULL;
  }

protected:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(webtexter, WebTexterProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "webtexter.moc"
