/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "messagewidget.hpp"
#include "orientationhelper.hpp"
#include "providers/providerinterface.hpp"
#include "providers/contacts.h"
#include "providers/messagetypemodel.hpp"
#include "providers/qgsmcodec.h"
#include "ui_messagewidget.h"

// Global includes
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QMenuBar>
#include <QtGui/QPlainTextDocumentLayout>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5InformationBox>
#endif // Q_WS_MAEMO_5

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MessageWidgetPrivate : public QObject
{
  Q_OBJECT

public:
  MessageWidgetPrivate()
  {}

  MessageWidget* self;

  // Properties
  Ui_MessageWidget ui;

  QPointer<ProviderInterface> provider;
  MessageTypeModel messageTypeModel;

  OrientationHelper orientationHelper;

  QVBoxLayout* portraitLayout;

  QGsmCodec gsmCodec;

  // Functions
  void init();
  QLayout* setupPortraitLayout();

public slots:
  void updateUi();

  // Provider callbacks
  void onError(const QString& message);
  void onSendMessageReply(bool success, const Message& message);
  void onBalanceReply(const QString& balance);

  // Ui callbacks
  void onClearReceivers();
  void onTextChanged();
  void onChooseContactButtonClicked();
  void onSendButtonClicked();
};

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::init()
{
  // Setup the orientation helper
  orientationHelper.setWidget(self);

  // Setup the ui
  ui.setupUi(self);

  // Setup the layouts
  orientationHelper.setLandscapeLayout(ui.verticalLayout);
  orientationHelper.setPortraitLayout( setupPortraitLayout() );

  QMenuBar* menuBar = new QMenuBar(self);
  QMenu* menu = menuBar->addMenu( QString() );
  menu->addAction(tr("Clear receivers"), this, SLOT(onClearReceivers()));

#ifndef Q_WS_MAEMO_5
  menu->setTitle("menu");
#endif // Q_WS_MAEMO_5

  // Connect signals and slots
  connect(ui.chooseContactButton, SIGNAL(clicked())               , this, SLOT(onChooseContactButtonClicked()));
  connect(ui.sendButton         , SIGNAL(clicked())               , this, SLOT(onSendButtonClicked()));
  connect(ui.messageText        , SIGNAL(textChanged())           , this, SLOT(onTextChanged()));
  connect(ui.messageTypePicker  , SIGNAL(currentIndexChanged(int)), this, SLOT(updateUi()));

  // Setup the picker
  ui.messageTypePicker->setModel(&messageTypeModel);
}

//////////////////////////////////////////////////////////////////////////

QLayout* MessageWidgetPrivate::setupPortraitLayout()
{
  portraitLayout = new QVBoxLayout;

  QHBoxLayout* topBox = new QHBoxLayout;
  topBox->addWidget(ui.chooseContactButton);
  topBox->addWidget(ui.contactEdit);
  portraitLayout->addLayout(topBox);

  portraitLayout->addWidget(ui.messageText);
  portraitLayout->addWidget(ui.balanceLabel);

  QHBoxLayout* bottomBox = new QHBoxLayout;
  bottomBox->addWidget(ui.messageTypePicker);
  bottomBox->addWidget(ui.messageInfoLabel);
  bottomBox->addWidget(ui.sendButton);
  portraitLayout->addLayout(bottomBox);

  return portraitLayout;
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::updateUi()
{
  QVariant messageTypeData = messageTypeModel.index(ui.messageTypePicker->currentIndex()).data(MessageTypeModel::MessageTypeRole);
  if ( messageTypeData.userType() == qMetaTypeId<MessageType>() )
  {
    int completeCharacterCount = 0;

    // Calculate the complete character count
    QString messageText = ui.messageText->toPlainText();
    for(int i=0; i < messageText.size(); ++i)
    {
      // Get the two byte code
      unsigned short twoByte = gsmCodec.twoByteFromUnicode(messageText.at(i));

      if ( twoByte == 0x10 ) // Unsupported character
        ; // Hell, something should happen - warn the user or something else?
      else if ( twoByte > 256 )
        completeCharacterCount += 2;
      else
        completeCharacterCount++;
    }

    // Calculate message information
    MessageType messageType = qvariant_cast<MessageType>(messageTypeData);
    int messageCharacterCount = completeCharacterCount % messageType.singleMessageLength();

    // Fixup lengths
    if ( messageCharacterCount == 0 )
      messageCharacterCount = messageType.singleMessageLength();

    int remainingCharacters = messageType.singleMessageLength() - messageCharacterCount;
    int messageCount = completeCharacterCount == 0 ? 1 : ((float)completeCharacterCount / (float)messageType.singleMessageLength()) + 0.999f; // Ehem, also not the best solution

    // Update the message information
    ui.messageInfoLabel->setText( QString("%1 (%2)").arg(remainingCharacters).arg(messageCount) );
  }
  else
    ui.messageInfoLabel->setText( QString() );

  // Update the send button state
  bool enableSendButton = false;
  if ( provider && provider->isLoggedIn() )
    enableSendButton = true;

  ui.sendButton->setEnabled(enableSendButton);
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onClearReceivers()
{
  QTextDocument* document = new QTextDocument(ui.contactEdit);
  document->setDocumentLayout(new QPlainTextDocumentLayout(document));
  ui.contactEdit->setDocument( document );
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onBalanceReply(const QString& balance)
{
  ui.balanceLabel->setText( balance.isEmpty() ? QString() : tr("Balance: %1").arg(balance) );
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onError(const QString& message)
{
  QMessageBox::critical(self, self->windowTitle(), message);
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onTextChanged()
{
  // TODO: Message type calculations

  // Update the ui
  updateUi();
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onSendMessageReply(bool success, const Message& message)
{
  if ( !success )
    return;

  // Get the message text
  QString text = message.text();

  // List all contacts and add the message to sent
  foreach(const ContactInfo& contact, message.receivers())
  {
    QString name = contact.name();
    QString number = contact.number();
    QString uid = contact.uniqueId();

    // Really add the messages
    add_to_sent_sms(&name, &number, &uid, &text);
  }

  // Show a information box
#ifdef Q_WS_MAEMO_5
  QMaemo5InformationBox::information(self, tr("Message sent."));
#else // Q_WS_MAEMO_5
  QMessageBox::information(self, self->windowTitle(), tr("Message sent."));
#endif // Q_WS_MAEMO_5

  // Clear everything
  onClearReceivers();
  ui.messageText->clear();
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onChooseContactButtonClicked()
{
  QString dialogTitle = tr("Choose a contact");
  ContactInfoList contacts;

  // Let the user choose a number
  if ( get_contact_for_sms(&dialogTitle, &contacts) )
  {
    // Get the current text
    foreach(ContactInfo contact, contacts)
    {
      // Try to convert the number to an international format
      QString number = contact.number();
      if ( !make_number_international(number) )
      {
        QMessageBox::critical(self, tr("Wrong number format"),
            tr("%1's phone number \"%2\" could not be converted to an international format.").arg(contact.name()).arg(number));
      
        continue;
      }

      // Set the contacts' number back
      contact.setNumber(number);

      // Add the contact
      ui.contactEdit->addContact(contact);
    }
  }
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onSendButtonClicked()
{
  // Determine the message type
  QVariant messageTypeData = messageTypeModel.index(ui.messageTypePicker->currentIndex()).data(MessageTypeModel::MessageTypeRole);
  if ( messageTypeData.userType() != qMetaTypeId<MessageType>() )
    return;

  MessageType messageType = qvariant_cast<MessageType>(messageTypeData);

  // Ensure the message fits the settings
  QString messageText = ui.messageText->toPlainText();
  if ( messageType.maxMessageLength() != -1 && messageText.size() > messageType.maxMessageLength() )
  {
    onError( tr("The message is too long! This message type only supports %n character(s).", "", messageType.maxMessageLength()) );
    return;
  }

  // Ensure the message receivers fit the settings
  ContactInfoList receivers = ui.contactEdit->contacts();
  if ( receivers.size() < 1 )
  {
    onError( tr("This message has no receivers!") );
    return;
  }

  if ( messageType.maxReceivers() != -1 && receivers.size() > messageType.maxReceivers() )
  {
    onError( tr("Too many receivers! This message type only supports %n receiver(s).", "", messageType.maxReceivers()) );
    return;
  }

  QProgressDialog* progressDialog = new QProgressDialog(self);
  progressDialog->setAttribute(Qt::WA_DeleteOnClose);
  progressDialog->setWindowTitle( tr("Sending SMS ...") );
  progressDialog->setRange(0, 0);
  progressDialog->show();

  connect(provider.data(), SIGNAL(sendMessageReply(bool, Message)), progressDialog, SLOT(accept()));

  // Setup the message
  Message message;
  message.setMessageType( messageType );
  message.setReceivers( receivers );
  message.setText( messageText );

  // Send the message
  provider->sendMessage(message);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MessageWidget::MessageWidget(QWidget* parent /* = 0 */, Qt::WindowFlags f /* = 0 */)
: QWidget(parent, f)
, d(new MessageWidgetPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MessageWidget::~MessageWidget()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MessageWidget::setProvider(ProviderInterface* provider)
{
  // Something changed?
  if ( d->provider == provider )
    return;

  // Disconnect the old provider
  if ( d->provider )
  {
    disconnect(d->provider, SIGNAL(sendMessageReply(bool,Message)), d, SLOT(onSendMessageReply(bool,Message)));
    disconnect(d->provider, SIGNAL(balanceReply(QString)), d, SLOT(onBalanceReply(QString)));
    disconnect(d->provider, SIGNAL(loginReply(bool)), d, SLOT(updateUi()));
    disconnect(d->provider, SIGNAL(logoutReply(bool)), d, SLOT(updateUi()));
    disconnect(d->provider, SIGNAL(error(QString)), d, SLOT(onError(QString)));

    // Invalidate the balance
    d->onBalanceReply( QString() );

    // Logout
    d->provider->logout();
  }

  // Take the new provider
  d->provider = provider;

  // Connect the new provider
  if ( d->provider )
  {
    // Connect signals
    connect(d->provider, SIGNAL(sendMessageReply(bool,Message)), d, SLOT(onSendMessageReply(bool,Message)));
    connect(d->provider, SIGNAL(balanceReply(QString)), d, SLOT(onBalanceReply(QString)));
    connect(d->provider, SIGNAL(loginReply(bool)), d, SLOT(updateUi()));
    connect(d->provider, SIGNAL(logoutReply(bool)), d, SLOT(updateUi()));
    connect(d->provider, SIGNAL(error(QString)), d, SLOT(onError(QString)));

    // Setup the message types
    QList<MessageType> messageTypes = d->provider->messageTypes();
    d->messageTypeModel.setMessageTypes(messageTypes);

    // Hide the picker if we only have one message type
    //d->ui.messageTypePicker->setVisible( messageTypes.size() > 1 );

    // TODO: Remember the choice
    if ( !messageTypes.isEmpty() )
      d->ui.messageTypePicker->setCurrentIndex(0);

    // Show a dialog that we're logging out first
    if ( !d->provider->isLoggedIn() )
    {
      CallbackType cb = d->provider->login();
      if ( cb == Callback_Async )
      {
        QProgressDialog* progressDialog = new QProgressDialog(this);
        progressDialog->setAttribute(Qt::WA_DeleteOnClose);
        progressDialog->setWindowTitle( tr("Logging in ...") );
        progressDialog->setRange(0, 0);
        progressDialog->show();

        // Close the information dialog and this window
        connect(provider, SIGNAL(loginReply(bool)), progressDialog, SLOT(accept()));
      }
    }

    // Update the balance afterwards
    d->provider->updateBalance();
  }
}

//////////////////////////////////////////////////////////////////////////

ProviderInterface* MessageWidget::provider() const
{
  return d->provider;
}

//////////////////////////////////////////////////////////////////////////

void MessageWidget::setCurrentMessageType(const MessageType& messageType)
{
  // Find the index for the message type
  QModelIndex index = d->messageTypeModel.indexFromMessageType(messageType);

  // Change the index
  if ( index.isValid() )
    d->ui.messageTypePicker->setCurrentIndex(index.row());
}

//////////////////////////////////////////////////////////////////////////

MessageType MessageWidget::currentMessageType() const
{
  // Get the current index of the message type picker
  int currentIndex = d->ui.messageTypePicker->currentIndex();
  QModelIndex index = d->messageTypeModel.index(currentIndex);

  // Return the message type
  return qvariant_cast<MessageType>( index.data(MessageTypeModel::MessageTypeRole) );
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "messagewidget.moc"
