/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "sloono.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../accountsettingsdialog.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtCore/QCryptographicHash>
#include <QtXml/QDomDocument>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class SloonoPrivate : public QObject
{
  Q_OBJECT

public:
  SloonoPrivate()
  {
    // Initialize the message types
    messageTypes.append( MessageType(tr("Discount"), "0", QIcon(), QString("0,06 %1").arg(QChar(0x20AC)), 160, 1600, 250) );
    messageTypes.append( MessageType(tr("Basic"), "1", QIcon(), QString("0,034 %1").arg(QChar(0x20AC)), 160, 1600, 250) );
    messageTypes.append( MessageType(tr("Pro"), "2", QIcon(), QString("0,078 %1").arg(QChar(0x20AC)), 160, 1600, 250) );
    messageTypes.append( MessageType(tr("Flash"), "3", QIcon(), QString(), 160, 160, 250) );

    // Add us as network listener
    network.addListener(this);
  }

  Sloono* self;

  // Properties
  QString username;
  QString password;

  QHash<QByteArray, QString> errorCodes;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleBalance(QNetworkReply* reply);
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void SloonoPrivate::handleBalance(QNetworkReply* reply)
{
  // Read the reply data
  QByteArray data = reply->readAll();

  /*
    101
    Erfolgreiche Abfrage

    Kontostand: 4,578
    SMS: 1234
    Postausgang: 0
    Archiv: 20
  */

  QList<QByteArray> lines = data.split('\n');
  if ( lines.size() >= 4 )
  {
    if ( lines.at(0) == "101\r" )
    {
      QString balance = QString::fromUtf8(lines.at(3));
      balance.remove(0, 12); // Remove "Kontostand: "
      balance.remove('\r');

      emit self->balanceReply( tr("%1 %2").arg(balance).arg(QChar(0x20AC)) );
    }
    else
    {
      QString msg = QString::fromUtf8(lines.at(1));
      msg.remove('\r');
      emit self->error( msg );
    }
  }
  else
    emit self->error( tr("Error parsing balance output") );
}

//////////////////////////////////////////////////////////////////////////

void SloonoPrivate::handleSendMessage(QNetworkReply* reply)
{
  QString errorMessage;

  QString xmlErrorMsg;
  int xmlErrorLine = 0;
  int xmlErrorColumn = 0;

  // Read the xml data
  QDomDocument document;
  if ( document.setContent(reply, &xmlErrorMsg, &xmlErrorLine, &xmlErrorColumn) )
  {
    // Get the document element
    QDomElement documentElement = document.documentElement();

    QDomElement codeElement = documentElement.firstChildElement("code");
    if ( !codeElement.isNull() )
    {
      if ( !codeElement.text().startsWith("10") )
      {
        QDomElement codeTextElement = documentElement.firstChildElement("codetext");
        errorMessage = tr("Error: %1").arg(codeTextElement.text());
      }
    }
    else
      errorMessage = tr("Returned XML data has no 'code' element");
  }

  // Get the message
  Message message = qvariant_cast<Message>( reply->property("message") );

  // Emit the send reply signal
  emit self->sendMessageReply(errorMessage.isEmpty(), message);

  // Error output
  if ( !errorMessage.isEmpty() )
    emit self->error(errorMessage);
  else
    self->updateBalance();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

Sloono::Sloono(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new SloonoPrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

Sloono::~Sloono()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void Sloono::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->username;
  stream >> d->password;
}

//////////////////////////////////////////////////////////////////////////

QByteArray Sloono::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->username;
  stream << d->password;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool Sloono::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle( tr("Sloono.de") );
  dialog.setUserName(d->username);
  //dialog.setPassword(d->password);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->username = dialog.userName();
    
    QString password = dialog.password();
    if ( !password.isEmpty() )
      d->password = QString::fromUtf8(QCryptographicHash::hash(password.toUtf8(), QCryptographicHash::Md5).toHex());

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> Sloono::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void Sloono::sendMessage(const Message& message)
{
  // Merge the numbers
  QStringList numbers;
  foreach(const ContactInfo& contact, message.receivers())
    numbers.append(contact.number());

  QByteArray text = toUrlPercentEncoding( message.text().toLocal8Bit() );

  // Setup the message
  QUrl url("https://www.sloono.de/API/httpsms.php");
  url.addQueryItem("user", d->username);
  url.addQueryItem("password", d->password);
  url.addQueryItem("to", numbers.join(","));
  //url.addQueryItem("from", d->fromNumber);
  url.addEncodedQueryItem("text", text);
  url.addQueryItem("timestamp", "0");
  url.addQueryItem("typ", message.messageType().name());
  url.addQueryItem("return", "xml");
  url.addQueryItem("httphead", "0");
  url.addQueryItem("action", "send"); // for debugging: "info"

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "SendMessage");
  reply->ignoreSslErrors();
  reply->setProperty("message", QVariant::fromValue(message)); // Remember the message for later use
}

//////////////////////////////////////////////////////////////////////////

void Sloono::updateBalance()
{
  // Setup the url
  QUrl url("https://www.sloono.de/API/httpkonto.php");
  url.addQueryItem("user", d->username);
  url.addQueryItem("password", d->password);
  url.addQueryItem("httphead", "0");

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "Balance");
  reply->ignoreSslErrors();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class SloonoProviderPlugin : public QObject
                           , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

  static QUuid uuid;

public:
  SloonoProviderPlugin()
  {
    infoList << ProviderInfo(tr("Sloono.de"), uuid, QIcon(":/providers/sloono/sloono.png"), QLocale::Germany);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( providerId == uuid )
      return new Sloono;

    return NULL;
  }

private:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

QUuid SloonoProviderPlugin::uuid = QUuid("9c57f290-4023-4e11-a03e-c23a13c4b75c");

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(Sloono, SloonoProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "sloono.moc"
