/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "innosend.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../accountsettingsdialog.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QFormLayout>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class InnosendPrivate : public QObject
{
  Q_OBJECT

public:
  InnosendPrivate()
  {
    // Initialize the message types
    messageTypes.append( MessageType(tr("Typ 2"), "2", QIcon(), QString("0,03 %1").arg(QChar(0x20AC)), 160, 160, 1) );
    messageTypes.append( MessageType(tr("Typ 2 Flash"), "2_flash", QIcon(), QString("0,03 %1").arg(QChar(0x20AC)), 160, 160, 1) );
    messageTypes.append( MessageType(tr("Typ 4"), "4", QIcon(), QString("0,079 %1").arg(QChar(0x20AC)), 160, 1000, 1) );

    // Add us as network listener
    network.addListener(this);

    // Setup the error codes
    errorCodes.insert("100", "SMS verschickt"); 
    errorCodes.insert("101", "MMS verschickt"); 
    errorCodes.insert("111", "IP-Sperre aktiv");  
    errorCodes.insert("112", "Falsche Userdaten");  
    errorCodes.insert("120", "Absender fehlt"); 
    errorCodes.insert("121", "Parameter Typ fehlt"); 
    errorCodes.insert("122", "Parameter Text fehlt"); 
    errorCodes.insert("123", "Parameter Empfnger fehlt"); 
    errorCodes.insert("129", "Falscher Absender"); 
    errorCodes.insert("130", "Gateway-Problem:\nRufnummer gesperrt, Falsche Inhalte, Inhalte, die auf unlautere Werbung hinweisen oder zu viele SMS eingeliefert  schauen Sie unter 'SMS-Fehlerliste' im Kundenbereich");    
    errorCodes.insert("140", "Kein Guthaben mehr, bitte Konto aufladen"); 
    errorCodes.insert("150", "SMS-Bomber-Schutz");   
    errorCodes.insert("170", "Datum falsch");  
    errorCodes.insert("171", "Datum in der Vergangenheit");  
    errorCodes.insert("172", "Zu viele Rufnummern");  
    errorCodes.insert("173", "Nummernformat falsch");
  }

  Innosend* self;

  // Properties
  QString username;
  QString password;
  QString fromNumber;

  QHash<QByteArray, QString> errorCodes;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleBalance(QNetworkReply* reply);
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void InnosendPrivate::handleBalance(QNetworkReply* reply)
{
  // Read the reply data
  QByteArray data = reply->readAll();

  QString errorCode = errorCodes.value(data);
  if ( errorCode.isEmpty() )
    emit self->balanceReply( QString("%1 %2").arg(QString::fromUtf8(data)).arg(QChar(0x20AC)) );
  else
    emit self->error(errorCode);
}

//////////////////////////////////////////////////////////////////////////

void InnosendPrivate::handleSendMessage(QNetworkReply* reply)
{
  QByteArray data = reply->readAll();

  bool success = (data=="100");

  // Get the message
  Message message = qvariant_cast<Message>( reply->property("message") );

  // Emit the send reply signal
  emit self->sendMessageReply(success, message);

  // Error output
  if ( !success )
  {
    QString defaultError = tr("Unknown error: %1").arg(QString::fromUtf8(data));
    emit self->error( errorCodes.value(data, defaultError) );
  }
  else
    self->updateBalance();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

Innosend::Innosend(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new InnosendPrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

Innosend::~Innosend()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void Innosend::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->username;
  stream >> d->password;
  stream >> d->fromNumber;
}

//////////////////////////////////////////////////////////////////////////

QByteArray Innosend::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->username;
  stream << d->password;
  stream << d->fromNumber;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool Innosend::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle( tr("Innosend.de") );
  dialog.setUserName(d->username);
  dialog.setPassword(d->password);

  // Add additional fields
  QLabel numberLabel(tr("Absender (nur Typ 4):"), &dialog);
  QLineEdit numberEdit(d->fromNumber, &dialog);

  QFormLayout* formLayout = dialog.formLayout();
  int row = formLayout->rowCount();
  formLayout->setWidget(row, QFormLayout::LabelRole, &numberLabel);
  formLayout->setWidget(row, QFormLayout::FieldRole, &numberEdit);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->username   = dialog.userName();
    d->password   = dialog.password();
    d->fromNumber = numberEdit.text();

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> Innosend::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void Innosend::sendMessage(const Message& message)
{
  QString messageTypeName = message.messageType().name();
  QString number = message.receivers().first().number();

  // Fix for numbers. This provider doesn't like the "+" sign, so we replace it by "00"
  number.replace(QRegExp("^\\+"), "00");

  QByteArray text = toUrlPercentEncoding( message.text().toLocal8Bit() );

  // Setup the message
  QUrl url("https://www.innosend.de/gateway/sms.php");
  url.addQueryItem("id", d->username);
  url.addQueryItem("pw", d->password);
  url.addQueryItem("empfaenger", number);
  url.addQueryItem("absender", d->fromNumber);
  url.addEncodedQueryItem("text", text);

  if ( messageTypeName == "2_flash" )
  {
    url.addQueryItem("type", "2");
    url.addQueryItem("flash", "1");
  }
  else
    url.addQueryItem("type", messageTypeName);

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "SendMessage");
  reply->ignoreSslErrors();
  reply->setProperty("message", QVariant::fromValue(message)); // Remember the message for later use
}

//////////////////////////////////////////////////////////////////////////

void Innosend::updateBalance()
{
  // Setup the url
  QUrl url("https://www.innosend.de/gateway/konto.php");
  url.addQueryItem("id", d->username);
  url.addQueryItem("pw", d->password);

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "Balance");
  reply->ignoreSslErrors();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class InnosendProviderPlugin : public QObject
                           , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

  static QUuid uuid;

public:
  InnosendProviderPlugin()
  {
    infoList << ProviderInfo(tr("Innosend.de"), uuid, QIcon(":/providers/innosend/innosend.png"), QLocale::Germany);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( providerId == uuid )
      return new Innosend;

    return NULL;
  }

private:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

QUuid InnosendProviderPlugin::uuid = QUuid("74bdc21f-16c4-4201-8b52-96af6ab131d8");

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(Innosend, InnosendProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "innosend.moc"
