/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "betamax.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../accountsettingsdialog.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtCore/QUrl>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QFormLayout>
#include <QtXml/QDomDocument>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class BetamaxPrivate : public QObject
{
  Q_OBJECT

public:
  BetamaxPrivate()
  {
    // Initialize the message types
    messageTypes.append( MessageType(tr("Standard"), "standard", QIcon(), QString(), 160, 160, 250) );

    // Add us as network listener
    network.addListener(this);
  }

  Betamax* self;

  // Properties
  QString providerName;
  QUrl sendUrl;

  QString username;
  QString password;
  QString fromNumber;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void BetamaxPrivate::handleSendMessage(QNetworkReply* reply)
{
  QString errorMessage;

  QString xmlErrorMessage;
  int xmlErrorLine   = 0;
  int xmlErrorColumn = 0;

  if ( reply->error() == QNetworkReply::NoError )
  {
    // Read the reply data
    QDomDocument document;
    if ( document.setContent(reply, &xmlErrorMessage, &xmlErrorLine, &xmlErrorColumn) )
    {
      QDomElement docElement = document.documentElement();

      // Get the result element
      QDomElement resultElement = docElement.firstChildElement("result");
      if ( !resultElement.isNull() )
      {
        // The result element is either "0" (failure) or "1" (success)
        if ( resultElement.text().toInt() != 1 )
        {
          // Get the description message
          QDomElement descriptionElement = docElement.firstChildElement("description");

          // Setup a description
          QString errorDescription = descriptionElement.text();
          if ( errorDescription.isEmpty() )
            errorDescription = tr("<no description>");

          // Build a readable error message
          errorMessage = tr("Error: %1\n\nCauses could be: Wrong username or password.").arg(errorDescription);
        }
      }
      else
        errorMessage = tr("Returned XML does not contain a 'result' element");
    }
    else
    {
      errorMessage = tr("Error parsing XML output:\nMessage: %1\nLine: %2\nColumn: %3")
        .arg(xmlErrorMessage)
        .arg(xmlErrorLine)
        .arg(xmlErrorColumn);
    }
  }
  else
    errorMessage = reply->errorString();


  // Get the message
  Message message = qvariant_cast<Message>( reply->property("message") );

  // Emit the send reply signal
  emit self->sendMessageReply(errorMessage.isEmpty(), message);

  // Error output
  if ( !errorMessage.isEmpty() )
    emit self->error(errorMessage);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

Betamax::Betamax(const QString& name, const QUrl& sendUrl, QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new BetamaxPrivate)
{
  d->self         = this;
  d->providerName = name;
  d->sendUrl      = sendUrl;
}

//////////////////////////////////////////////////////////////////////////

Betamax::~Betamax()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void Betamax::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->username;
  stream >> d->password;
  stream >> d->fromNumber;
}

//////////////////////////////////////////////////////////////////////////

QByteArray Betamax::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->username;
  stream << d->password;
  stream << d->fromNumber;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool Betamax::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle(d->providerName.isEmpty() ? tr("Other Betamax") : d->providerName);
  dialog.setUserName(d->username);
  dialog.setPassword(d->password);

  // Add additional fields
  QLabel numberLabel(tr("From number:"), &dialog);
  QLineEdit numberEdit(d->fromNumber, &dialog);

  QLabel sendUrlLabel(tr("Send URL:"), &dialog);
  QLineEdit sendUrlEdit(d->sendUrl.toString(), &dialog);

  QFormLayout* formLayout = dialog.formLayout();
  int row = formLayout->rowCount();
  formLayout->setWidget(row, QFormLayout::LabelRole, &numberLabel);
  formLayout->setWidget(row, QFormLayout::FieldRole, &numberEdit);

  formLayout->setWidget(++row, QFormLayout::LabelRole, &sendUrlLabel);
  formLayout->setWidget(  row, QFormLayout::FieldRole, &sendUrlEdit);

  sendUrlLabel.setVisible( d->providerName.isEmpty() );
  sendUrlEdit.setVisible( d->providerName.isEmpty() );

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->username   = dialog.userName();
    d->password   = dialog.password();
    d->fromNumber = numberEdit.text();

    if ( d->providerName.isEmpty() )
      d->sendUrl = sendUrlEdit.text();

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> Betamax::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void Betamax::sendMessage(const Message& message)
{
  QStringList numbers;

  // List the contacts
  foreach(const ContactInfo& contact, message.receivers())
    numbers << contact.number();

  QString toField = numbers.join("\n");

  // Setup the message
  // sendsms.php?username=xxxxxxxxxx?&password=xxxxxxxxxx&from=xxxxxxxxxx&to=xxxxxxxxxx&text=xxxxxxxxxx
  QUrl url(d->sendUrl);
  url.addQueryItem("username", d->username);
  url.addQueryItem("password", d->password);
  url.addQueryItem("from", d->fromNumber);
  url.addQueryItem("to", toField);
  url.addQueryItem("text", message.text());

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "SendMessage");
  reply->setProperty("message", QVariant::fromValue(message)); // Remember the message for later use
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class BetamaxProviderPlugin : public QObject
                            , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

public:
  BetamaxProviderPlugin()
  {
    addProvider(tr("VoipCheap.com"), "fbba682a-3139-472d-934d-228de216426a", QIcon(":/providers/betamax/voipcheap.com.png"), "https://www.voipcheap.com/myaccount/sendsms.php");
    addProvider(tr("SMSdiscount.com"), "91ac866a-a013-4f84-9def-204c366b8468", QIcon(":/providers/betamax/smsdiscount.com.png"), "https://www.smsdiscount.com/myaccount/sendsms.php");
    addProvider(tr("Lowratevoip.com"), "4a8dead9-6df8-4ff9-adf5-5604bf739755", QIcon(), "https://www.lowratevoip.com/myaccount/sendsms.php");
    addProvider(QString(), "a1834119-d9f6-44d5-80b8-5550946d5df7", QIcon(), QString());
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( nameHash.contains(providerId) )
      return new Betamax(nameHash.value(providerId), sendUrlHash.value(providerId));

    return NULL;
  }

protected:
  void addProvider(const QString& name, const QUuid& uuid, const QIcon& icon, const QString& sendUrl)
  {
    infoList << ProviderInfo(name.isEmpty() ? tr("Other Betamax") : name, uuid, icon, QLocale::AnyCountry);
    
    nameHash.insert(uuid, name);
    sendUrlHash.insert(uuid, sendUrl);
  }

private:
  QList<ProviderInfo> infoList;
  QHash<QUuid, QString> nameHash;
  QHash<QUuid, QUrl> sendUrlHash;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(Betamax, BetamaxProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "betamax.moc"
