/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "orange_CH.hpp"
#include "../providerplugin.hpp"
#include "../accountsettingsdialog.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QRegExp>
#include <QtCore/QDebug>
#include <QtCore/QDataStream>
#include <QtCore/QFile>
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class orangeCHPrivate : public QObject
{
    Q_OBJECT
public:
    explicit orangeCHPrivate(QObject *parent = 0);

    enum Action {
        Stop,
        Balance,
        Sent,
        Logout,
        Error
    };

    orangeCH* self;

    struct
    {
      QUrl loginUrl;
      QUrl logoutUrl;
      QUrl sendUrl;
    } urls;

    int freeSMSCount;
    bool isLoggedIn;
    bool isDebugMode;
    QString postUri;
    QString postUriCancel;
    QString postUriSend;
    QString SMSnumber,SMSmessage,loginName,password;
    QHash<QNetworkReply*, Action> replyStateHash;
    QNetworkAccessManager accessManager;


    void login(Action);
    void logout();
    void prepareSMS(QString num,QString msg,QString choosePostUri);
    void handleSendSMS(Action);
    void logger(QString);


    // Network functions
    QNetworkRequest prepareRequest() const;
    QNetworkReply* get(const QNetworkRequest& request, Action action);
    QNetworkReply* post(const QNetworkRequest& request, const QByteArray& data, Action action);

public slots:
    void onReplyFinished(QNetworkReply* reply);

};


orangeCHPrivate::orangeCHPrivate(QObject *parent) :
    QObject(parent)
{
    urls.loginUrl    = QUrl("https://www.orange.ch/footer/login/loginForm");
    urls.sendUrl     = QUrl("https://www.orange.ch/myorange/sms/smsForm");
    urls.logoutUrl    = QUrl("https://www.orange.ch/vrtmyaccount?.portal_action=.logoutUser&amp;.portlet=myaccountwelcome");
    postUriSend ="wui_target_id=sendButton&wui_event_id=onclick&destinationNumberInput=NBR&messageInput=MSG&charNumberLeftOutput=0&signatureInput=";
    postUriCancel ="wui_target_id=cancelButton";
    connect(&accessManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(onReplyFinished(QNetworkReply*)));
    isDebugMode=false;
    isLoggedIn=false;
    freeSMSCount=-1;

}

void orangeCHPrivate::logger(QString txtToLog)
{
    if (isDebugMode)
    {
        QFile file("/home/david/log.txt");
        if (file.open(QFile::Append)) {
        QTextStream out(&file);
        out  << txtToLog << QString("\r\n");}
    }

}
void orangeCHPrivate::login(Action action)
{

        QNetworkRequest request = prepareRequest();
        QByteArray formData;
        formData += "wui_target_id=loginButton&wui_event_id=onclick&username="+loginName+"&password="+password+"&loginButton=Login";
        QString formDataS=QString("wui_target_id=loginButton&wui_event_id=onclick&username=%1&password=%2&loginButton=Login").arg(loginName).arg(password);

        request.setUrl(QUrl(urls.loginUrl));
        logger(QString("login process: %1").arg(formDataS));
        post(request, formData, action);
}
void orangeCHPrivate::logout()
{
    QNetworkRequest request = prepareRequest();
    request.setUrl(QUrl(urls.logoutUrl));
    get(request, Logout);
    isLoggedIn=false;

}



void orangeCHPrivate::onReplyFinished(QNetworkReply* reply)
{
    // Handle a redirection if needed
    QVariant redirectLocation = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);
    if ( !redirectLocation.isNull() )
    {
        // Where are we going to?
        QUrl newUrl = reply->url().resolved(redirectLocation.toUrl());

        // Prepare the request and send it
        QNetworkRequest request = prepareRequest();
        request.setRawHeader("Referer", reply->url().toEncoded());
        request.setUrl( newUrl );
        get(request, replyStateHash.value(reply));

        return; // Don't continue
    }

    QByteArray readData = reply->readAll();
    QString data = QString::fromLatin1(readData.constData(), readData.size());
    Action action=replyStateHash.value(reply);
    if (isDebugMode)
    {
        QFile file("/home/david/reply.html");
        if (file.open(QFile::WriteOnly)) {
            QTextStream out(&file);
            out  << data;}
    }

    // check login
    if (data.indexOf("pas disponibles") == -1 ) {
        if (action==Stop){
            // logging phase
            if (data!="" && data.indexOf("error") == -1 )
            {
                isLoggedIn=true;
                // Emit that the login
                logger(QString("onReplyFinished:logging check: logged OK "));
                emit self->loginReply( true );
            } else {
                action=Error;
                // Emit that the login
                emit self->error( tr("login failed") );
                emit self->loginReply( false );
            }
        }


        if (action==Sent || action==Balance)  {
            // check the balance:
            int formOffset = 0;
            QRegExp freeSmsExp("Il vous reste (\\d*) SMS gratuits");
            if ( freeSmsExp.indexIn(data) != -1 )
            {
                // Remember the offset to not search from the beginning every time
                formOffset = freeSmsExp.pos() + freeSmsExp.matchedLength();

                // How many free sms are left?
                freeSMSCount = freeSmsExp.cap(1).toInt();
                // Emit how many free-sms we have left
                logger(QString("handle balance: %1 ").arg(freeSMSCount));
                emit self->balanceReply( tr("%n free SMS", "", freeSMSCount) );
            } else {
                QRegExp freeSmsExp0("Votre crdit de SMS gratuits du mois est puis");
                if ( freeSmsExp0.indexIn(data) != -1 )
                {
                    freeSMSCount=0;
                    emit self->balanceReply( tr("%n free SMS", "", freeSMSCount) );
                } else {
                    action=Error;
                    logger(QString("no known reply from message sent "));
                }

            }
        }
    } else {

        emit self->loginReply( false );
        logger(QString("services are not accessible for maintenance reasons"));
        emit self->error( tr("services are not accessible for maintenance reasons") );
        action=Error;

    }

    // Decide what to do now
    switch(action)
    {

    case Balance:
        logger(QString("onReplyFinished:action: Balance Sent"));
        break;

    case Sent:
        {
            logger(QString("onReplyFinished:action:Sent "));
            // Emit a signal that we've sent the message
            Message message = qvariant_cast<Message>(reply->property("message"));
            emit self->sendMessageReply(true, message);
        }
        break;

    case Error:
        isLoggedIn=false;
        freeSMSCount = -1;
        logger(QString("onReplyFinished:action: Error "));
        break;

    case Stop:
        logger(QString("onReplyFinished:action: Stop "));
        self->updateBalance();
        break;

    case Logout:
        logger(QString("onReplyFinished:action: Logout "));
        emit self->logoutReply(true);
        break;

    default:
        logger(QString("onReplyFinished:action: default "));
        break;
    }

    // Remove hash values for this reply (if any)
    replyStateHash.remove(reply);

    // Delete the reply at a later time
    reply->deleteLater();

}

  QNetworkRequest orangeCHPrivate::prepareRequest() const
  {
    QNetworkRequest request;
    request.setRawHeader("User-Agent", "Opera/9.80 (Linux x86_64; U; en) Presto/2.5.18 Version/10.50 Gentoo");

    return request;
  }
  //////////////////////////////////////////////////////////////////////////

  QNetworkReply* orangeCHPrivate::get(const QNetworkRequest& request, Action action)
  {
    // Get the request
    QNetworkReply* reply = accessManager.get(request);

    // Remember the reply with the given state
    replyStateHash.insert(reply, action);

    return reply;
  }

  //////////////////////////////////////////////////////////////////////////

  QNetworkReply* orangeCHPrivate::post(const QNetworkRequest& request, const QByteArray& data, Action action)
  {
    // Get the post
    QNetworkReply* reply = accessManager.post(request, data);

    // Remember the reply with the given state
    replyStateHash.insert(reply, action);

    return reply;
  }

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

orangeCH::orangeCH(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new orangeCHPrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

orangeCH::~orangeCH()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void orangeCH::loadSettings(const QByteArray& element)
{
  QDataStream stream(element);
  stream >> d->loginName;
  stream >> d->password;
}

//////////////////////////////////////////////////////////////////////////

QByteArray orangeCH::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->loginName;
  stream << d->password;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool orangeCH::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle( tr("Orange CH") );
  dialog.setUserName(d->loginName);
  dialog.setPassword(d->password);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->loginName = dialog.userName();
    d->password  = dialog.password();

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

bool orangeCH::isLoggedIn() const
{
  // Simple .. heh?
  return d->isLoggedIn;
}

//////////////////////////////////////////////////////////////////////////

CallbackType orangeCH::login()
{
    d->logger(QString("starting loging"));
    d->login(orangeCHPrivate::Stop);

  return Callback_Async;
}

//////////////////////////////////////////////////////////////////////////

CallbackType orangeCH::logout()
{
    d->logout();

  return Callback_Async;
}

//////////////////////////////////////////////////////////////////////////

void orangeCH::sendMessage(const Message& message)
{
    d->logger(QString("starting sendMessage"));
  QString smsTo;
  // Merge the numbers
  foreach(const ContactInfo& receiver, message.receivers())
  {
    if ( !smsTo.isEmpty() )
      smsTo += ", ";

//    smsTo += QString("%1 <%2>").arg(receiver.name()).arg(receiver.number());
      smsTo += QString("%1").arg(receiver.number());

}
  QString postUri=d->postUriSend;
  //  postUri=postUri.replace(QString("MSG"),toUrlPercentEncoding(message.text().toLocal8Bit()));
  postUri=postUri.replace(QString("MSG"),message.text().toLocal8Bit());
  QByteArray formData;
  QNetworkRequest request;
  formData += postUri.replace(QString("NBR"),smsTo);
  request.setUrl(QUrl(d->urls.sendUrl));

  QNetworkReply* reply;
  // Prepare the send-message request
  if (d->isLoggedIn){
      reply=d->post(request,formData,orangeCHPrivate::Sent);
  } else {
      // Not logged in: Abort
      d->logger(QString("Not logged in: Abort"));
  }
  reply->setProperty("message", QVariant::fromValue(message));
}

//////////////////////////////////////////////////////////////////////////

void orangeCH::updateBalance()
{
    d->logger(QString("starting updateBalance"));
    QByteArray formData;
    QNetworkRequest request;
    formData += d->postUriCancel;
    request.setUrl(QUrl(d->urls.sendUrl));
    QNetworkReply* reply;
    if (d->isLoggedIn) {
        reply=d->post(request,formData,orangeCHPrivate::Balance);
    } else {
        // Not logged in: Abort
        d->logger(QString("Not logged in: Abort"));

    }
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> orangeCH::messageTypes() const
{
  return QList<MessageType>()
      << MessageType(tr("Standard"), "standard", QIcon(), QString(), 144, 2000, 10);
}
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#define ORANGE_CH_UUID "852ee4b8-e829-4a4e-bae5-fc5a729d01ef"

class orangeCHProviderPlugin : public QObject
                              , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

public:
  orangeCHProviderPlugin()
  {
    infoList << ProviderInfo(tr("Orange CH"), ORANGE_CH_UUID, QIcon(":/providers/orange_CH/orange_CH.png"), QLocale::Germany);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& uuid) const
  {
    if ( uuid == ORANGE_CH_UUID )
      return new orangeCH;

    return NULL;
  }

protected:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(orange_CH, orangeCHProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "orange_CH.moc"
