/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "o2_germany.hpp"
#include "../providerplugin.hpp"
#include "../accountsettingsdialog.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QRegExp>
#include <QtCore/QDebug> 
#include <QtCore/QDataStream>
#include <QtCore/QFile>
#include <QtCore/QTimer>
#include <QtGui/QLabel>
#include <QtGui/QHBoxLayout>
#include <QtGui/QDialogButtonBox>
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class o2GermanySendStateDialog : public QDialog
{
  Q_OBJECT

public:
  o2GermanySendStateDialog(QWidget* parent = 0, Qt::WindowFlags f = 0);
  ~o2GermanySendStateDialog();

  void setState(int state);

  QDialogButtonBox buttonBox;
  QLabel label;

  Message message;
  QString messageId;

  QObject* privClass;
};

//////////////////////////////////////////////////////////////////////////

o2GermanySendStateDialog::o2GermanySendStateDialog(QWidget* parent /* = 0 */, Qt::WindowFlags f /* = 0 */)
: QDialog(parent, f)
, buttonBox(this)
{
  setWindowTitle( tr("Delivery state") );

  buttonBox.setOrientation(Qt::Vertical);
  buttonBox.setStandardButtons(QDialogButtonBox::Close);

  QHBoxLayout* layout = new QHBoxLayout;
  layout->addWidget(&label);
  layout->addWidget(&buttonBox);
  setLayout(layout);

  connect(&buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
  connect(&buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
}

//////////////////////////////////////////////////////////////////////////

o2GermanySendStateDialog::~o2GermanySendStateDialog()
{
}

//////////////////////////////////////////////////////////////////////////

void o2GermanySendStateDialog::setState(int state)
{
  QString text;

  switch(state)
  {
  case 0:
    text = tr("Awaiting information ...");
    break;

  case 1: // DeliveryPending
    text = tr("Waiting for receipt");
    break;

  case 2: // Delivered
    text = tr("Delivered");
    break;

  case 3: // ReceiptConfirmed
    text = tr("Receipt confirmed");
    break;

  case 4: // NotDelivered
    text = tr("Not yet delivered");
    break;

  case 5: // NotAdmitted
    text = tr("Not admitted");
    break;

  case 6: // InsuffBalance
    text = tr("Insufficient balance");
    break;

  case 7: // DeliveredPartly
    text = tr("Partially delivered");
    break;

  default:
    text = tr("Unknown");
    break;
  }

  label.setText(text);

  if ( state != 5 && state != 2 && state != 0 )
    QTimer::singleShot(5000, privClass, SLOT(updateSendDetails()));

#ifdef Q_WS_MAEMO_5
  setAttribute(Qt::WA_Maemo5ShowProgressIndicator, state != 5 && state != 2);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class o2GermanyPrivate : public QObject
{
  Q_OBJECT

public:
  o2GermanyPrivate()
  : sentMessages(0)
  , freeSMSCount(-1)
  {
    sendStateDialog.privClass = this;

    // We define some default urls (may be replaced by code later)
    urls.preLoginUrl      = QUrl("http://www.o2online.de");
    urls.loginUrl         = QUrl("https://login.o2online.de/auth/login");
    urls.logoutUrl        = QUrl("https://login.o2online.de/loginRegistration/loginAction.do?_flowId=logout");
    urls.smsCenterUrl     = QUrl("https://email.o2online.de/smscenter_new.osp");
    urls.smsCenterSendUrl = QUrl("https://email.o2online.de/smscenter_send.osp");
    urls.comCenterUrl     = QUrl("https://email.o2online.de/ssomanager.osp?APIID=AUTH-WEBSSO&TargetApp=m_folder_detail.osp%3FFolder%3DO3sis%3A%3AInbox");

    connect(&accessManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(onReplyFinished(QNetworkReply*)));
  }

  enum State
  {
    Unknown,
    PreLogin,
    Login,
    Logout,
    ComCenter,
    SmsCenter,
    SmsCenterSend,
    FolderSent
  };

  o2Germany* self;

  // Properties
  o2GermanySendStateDialog sendStateDialog;

  QNetworkAccessManager accessManager;

  struct
  {
    QUrl preLoginUrl;
    QUrl loginUrl;
    QUrl logoutUrl;

    QUrl comCenterUrl;
    QUrl smsCenterUrl;
    QUrl smsCenterSendUrl;

    QUrl folderSentUrl;
  } urls;

  QHash<QNetworkReply*, State> replyStateHash;

  // User settings
  QString loginName;
  QString password;

  QString sessionId;

  int sentMessages;
  int freeSMSCount;

  QHash<QString, QByteArray> sendFormDefaultValues;

  // Functions
  QNetworkRequest prepareRequest() const;

  // Network functions
  QNetworkReply* get(const QNetworkRequest& request, State state);
  QNetworkReply* post(const QNetworkRequest& request, const QByteArray& data, State state);

  // Handler functions
  void handlePreLogin(QNetworkReply* reply);
  void handleLogin(QNetworkReply* reply);
  void handleComCenter(QNetworkReply* reply);
  void handleSmsCenter(QNetworkReply* reply);
  void handleSmsCenterSend(QNetworkReply* reply);
  void handleLogout(QNetworkReply* reply);
  void handleFolderSent(QNetworkReply* reply);

public slots:
  void onReplyFinished(QNetworkReply* reply);
  void updateSendDetails();
};

//////////////////////////////////////////////////////////////////////////

QNetworkRequest o2GermanyPrivate::prepareRequest() const
{
  QNetworkRequest request;
  request.setRawHeader("User-Agent", "Opera/9.80 (Linux x86_64; U; en) Presto/2.5.18 Version/10.50 Gentoo");

  return request;
}

//////////////////////////////////////////////////////////////////////////

QNetworkReply* o2GermanyPrivate::get(const QNetworkRequest& request, State state)
{
  // Get the request
  QNetworkReply* reply = accessManager.get(request);
  reply->ignoreSslErrors();

  // Remember the reply with the given state
  replyStateHash.insert(reply, state);

  return reply;
}

//////////////////////////////////////////////////////////////////////////

QNetworkReply* o2GermanyPrivate::post(const QNetworkRequest& request, const QByteArray& data, State state)
{
  // Get the post
  QNetworkReply* reply = accessManager.post(request, data);
  reply->ignoreSslErrors();

  // Remember the reply with the given state
  replyStateHash.insert(reply, state);

  return reply;
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handlePreLogin(QNetworkReply* reply)
{
  // Get all data from the reply
  QByteArray readData = reply->readAll();
  QString data = QString::fromUtf8(readData.constData(), readData.size());

  // Find the login url
  //QRegExp formExp("loginForm\" action=\"([^\"]*)\"");
  //if ( formExp.indexIn(data) != -1 )
  {
    // Set the login url
    //urls.loginUrl = QUrl(formExp.cap(1));

    //QRegExp urlExp("<a href=\"([^\"]*).*title=\"E-Mail\">");
    //if ( urlExp.indexIn(data, formExp.pos() + formExp.matchedLength()) != -1 )
    //  urls.comCenterUrl = urlExp.cap(1);

    // Ok, the fields are known:
    // loginName
    // password
    // url

    QByteArray formData;
    formData += "loginName=" + loginName.toUtf8();
    formData += "&password=" + password.toUtf8();
    formData += "&url=" + urls.comCenterUrl.toString().toUtf8().toPercentEncoding();

    // Prepare a request and post it
    QNetworkRequest request = prepareRequest();
    request.setRawHeader("Referer", reply->url().toEncoded());
    request.setUrl( urls.loginUrl );

    post(request, formData, Login);
  }
  /*else
  {
    // Emit that we did not find the loginForm
    emit self->error( tr("loginForm could not be found!") );

    // Emit that the login failed
    emit self->loginReply(false);
  }*/
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handleLogin(QNetworkReply* reply)
{
  // Get all data from the reply
  QByteArray readData = reply->readAll();
  QString data = QString::fromUtf8(readData.constData(), readData.size());

  bool success = data.indexOf("Fehler beim Login") == -1;

  if ( success )
  {
    // Open a new request to the sms center
    QNetworkRequest request = prepareRequest();
    request.setRawHeader("Referer", reply->url().toEncoded());
    request.setUrl( urls.comCenterUrl );
    get(request, ComCenter);
  }

  // Emit that the login failed
  emit self->loginReply( success );
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handleComCenter(QNetworkReply* reply)
{
  // Get all data from the reply
  QByteArray readData = reply->readAll();

  QNetworkRequest request = prepareRequest();
  request.setRawHeader("Referer", reply->url().toEncoded());
  request.setUrl( urls.smsCenterUrl );
  get(request, SmsCenter);
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handleSmsCenter(QNetworkReply* reply)
{
  // Get all data from the reply
  QByteArray readData = reply->readAll();
  QString data = QString::fromUtf8(readData.constData(), readData.size());

  int formOffset = 0;

  // Find the message which tells us something about our remaining free-sms
  // If it's not found we don't care ...
  QRegExp freeSmsExp("Frei-SMS: (\\d*) Web2SMS");
  if ( freeSmsExp.indexIn(data) != -1 )
  {
    // Remember the offset to not search from the beginning every time
    formOffset = freeSmsExp.pos() + freeSmsExp.matchedLength();

    // How many free sms are left?
    freeSMSCount = freeSmsExp.cap(1).toInt();

    // Emit how many free-sms we have left
    emit self->balanceReply( tr("%n free SMS", "", freeSMSCount) );
  }

  // Try to determine how many free sms we have left
  QRegExp formExp("name=\"frmSMS\" action=\"([^\"]*)\"");
  if ( formExp.indexIn(data, formOffset) != -1 )
  {
    // Remember the offset
    int offset = formExp.pos() + formExp.matchedLength();

    // Change the free sms center send-url if wanted
    urls.smsCenterSendUrl = reply->url().resolved(formExp.cap(1));

    // Fill the default fields
    sendFormDefaultValues.clear();

    QRegExp fieldExp("<input type=\"hidden\" name=\"([^\"]*)\" value=\"([^\"]*)\"", Qt::CaseInsensitive);
    int pos = fieldExp.indexIn(data, offset);
    while ( pos != -1 )
    {
      // Get the field name
      const QString fieldName = fieldExp.cap(1);

      // Update the folder urls
      if ( fieldName == "SID" )
      {
        sessionId = fieldExp.cap(2);
        urls.folderSentUrl = QString("https://email.o2online.de/smscenter_search.osp?SID=%1&FolderID=0").arg(fieldExp.cap(2));
      }

      sendFormDefaultValues.insert( fieldName, fieldExp.cap(2).toUtf8().toPercentEncoding() );
      
      // Find the next entry
      pos = fieldExp.indexIn(data, pos+fieldExp.matchedLength());
    }

    sendFormDefaultValues.insert("Frequency", "5");
    sendFormDefaultValues.insert("StartDateDay", QByteArray());
    sendFormDefaultValues.insert("StartDateMonth", QByteArray());
    sendFormDefaultValues.insert("StartDateYear", QByteArray());
    sendFormDefaultValues.insert("StartDateHour", QByteArray());
    sendFormDefaultValues.insert("StartDateMin", QByteArray());
    sendFormDefaultValues.insert("EndDateDay", QByteArray());
    sendFormDefaultValues.insert("EndDateMonth", QByteArray());
    sendFormDefaultValues.insert("EndDateYear", QByteArray());
    sendFormDefaultValues.insert("EndDateHour", QByteArray());
    sendFormDefaultValues.insert("EndDateMin", QByteArray());
  }
  else
  {
    // Emit that we could not find the frmSMS
    emit self->error( tr("frmSMS could not be found!") );
  }
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handleSmsCenterSend(QNetworkReply* reply)
{
  // Get all data from the reply
  QByteArray readData = reply->readAll();
  QString data = QString::fromUtf8(readData.constData(), readData.size());

  Message message = qvariant_cast<Message>(reply->property("message"));

  // Emit a signal that we've sent the message
  emit self->sendMessageReply(true, message);

  // We do some dummy analysis of the message ...
  MessageType messageType = message.messageType();
  QString messageText = message.text();

  int usedMessageCount = 0;
  if ( messageText.size() <= 160 )
    usedMessageCount = 1;
  else
    usedMessageCount = (messageText.size() / messageType.singleMessageLength()) + 1;

  // Subtract ...
  freeSMSCount -= usedMessageCount * message.receivers().size();

  emit self->balanceReply( tr("%n* free SMS", "", freeSMSCount) );

  // Now we check the send state
  QNetworkRequest request = prepareRequest();
  request.setRawHeader("Referer", reply->url().toEncoded());
  request.setUrl( urls.folderSentUrl );
  get(request, FolderSent);

  sendStateDialog.message = message;
  sendStateDialog.setState(0);
  sendStateDialog.show();
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handleLogout(QNetworkReply* reply)
{
  Q_UNUSED(reply);

  // No more free sms
  freeSMSCount = -1;

  // No need to read something, logout works always
  emit self->logoutReply(true);
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::handleFolderSent(QNetworkReply* reply)
{
  // Get all data from the reply
  QByteArray readData = reply->readAll();
  QString data = QString::fromUtf8(readData.constData(), readData.size());

  if ( !sendStateDialog.isVisible() )
    return;

  int state = -1;

  // Find the message id of the last message (first being found)
  QRegExp statusExp("displaySendStatus\\((\\d),'(\\d*)'");
  if ( statusExp.indexIn(data) == -1 )
  {
    // Second try
    statusExp.setPattern( "displaySendStatus\\((\\d)," );
    if ( statusExp.indexIn(data) == -1 )
      return;
  }

  state = statusExp.cap(1).toInt();
  sendStateDialog.messageId = statusExp.cap(2);
  sendStateDialog.setState(state);
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::updateSendDetails()
{
  QUrl url( QString("https://email.o2online.de/smscenter_group_status.osp?SID=%1&SortColumnType=Field&SortOrder=ASC&SortColumn=RECIPIENT&MsgContentID=%2").arg(sessionId).arg(sendStateDialog.messageId) );

  QNetworkRequest request = prepareRequest();
  request.setUrl(url);
  get(request, FolderSent);
}

//////////////////////////////////////////////////////////////////////////

void o2GermanyPrivate::onReplyFinished(QNetworkReply* reply)
{
  // Handle a redirection if needed
  QVariant redirectLocation = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);
  if ( !redirectLocation.isNull() )
  {
    // Where are we going to?
    QUrl newUrl = reply->url().resolved(redirectLocation.toUrl());

    // Prepare the request and send it
    QNetworkRequest request = prepareRequest();
    request.setRawHeader("Referer", reply->url().toEncoded());
    request.setUrl( newUrl );
    get(request, replyStateHash.value(reply));

    return; // Don't continue
  }

  // If some error happend while transmitting HTTP content, also signalize it
  if ( reply->error() )
    emit self->error( tr("HTTP error happened:\n%1").arg(reply->errorString()) );

  // Decide what to do now
  switch(replyStateHash.value(reply))
  {
  case PreLogin:
    handlePreLogin(reply);
    break;

  case Login:
    handleLogin(reply);
    break;

  case ComCenter:
    handleComCenter(reply);
    break;

  case SmsCenter:
    handleSmsCenter(reply);
    break;

  case SmsCenterSend:
    handleSmsCenterSend(reply);
    break;

  case Logout:
    handleLogout(reply);
    break;

  case FolderSent:
    handleFolderSent(reply);
    break;

  default:
    break;
  }

  // Remove hash values for this reply (if any)
  replyStateHash.remove(reply);

  // Delete the reply at a later time
  reply->deleteLater();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

o2Germany::o2Germany(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new o2GermanyPrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

o2Germany::~o2Germany()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void o2Germany::loadSettings(const QByteArray& element)
{
  QDataStream stream(element);
  stream >> d->loginName;
  stream >> d->password;
}

//////////////////////////////////////////////////////////////////////////

QByteArray o2Germany::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->loginName;
  stream << d->password;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool o2Germany::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle( tr("o2 Germany") );
  dialog.setUserName(d->loginName);
  dialog.setPassword(d->password);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->loginName = dialog.userName();
    d->password  = dialog.password();

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

bool o2Germany::isLoggedIn() const
{
  // Simple .. heh?
  return d->freeSMSCount != -1;
}

//////////////////////////////////////////////////////////////////////////

CallbackType o2Germany::login()
{
  // Start the first request
  QNetworkRequest request = d->prepareRequest();
  request.setUrl( d->urls.preLoginUrl );

  // Start the pre-login
  d->get(request, o2GermanyPrivate::PreLogin);

  return Callback_Async;
}

//////////////////////////////////////////////////////////////////////////

CallbackType o2Germany::logout()
{
  // Start the logout request
  QNetworkRequest request = d->prepareRequest();
  request.setUrl( d->urls.logoutUrl );

  // Start the logout request
  d->get(request, o2GermanyPrivate::Logout);

  return Callback_Async;
}

//////////////////////////////////////////////////////////////////////////

void o2Germany::sendMessage(const Message& message)
{
  QString smsTo;

  // Merge the numbers
  foreach(const ContactInfo& receiver, message.receivers())
  {
    if ( !smsTo.isEmpty() )
      smsTo += ", ";

    smsTo += QString("%1 <%2>").arg(receiver.name()).arg(receiver.number());
  }

  // Prepare the send-message request
  QNetworkRequest request = d->prepareRequest();
  request.setRawHeader("Referer", d->urls.smsCenterUrl.toEncoded());
  request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
  request.setUrl( d->urls.smsCenterSendUrl );

  qDebug() << d->urls.smsCenterSendUrl;

  // Build the form data
  QByteArray formData;

  // Fill the remaining fields
  d->sendFormDefaultValues.insert("SMSTo", smsTo.toAscii().toPercentEncoding());
  d->sendFormDefaultValues.insert("SMSText", toUrlPercentEncoding(toLatin1Euro(message.text())));

  // Fill the default values
  QHashIterator<QString, QByteArray> iter(d->sendFormDefaultValues);
  while (iter.hasNext())
  {
    iter.next();

    if ( !formData.isEmpty() )
      formData += '&';

    formData += iter.key().toLatin1() + '=' + iter.value();
  }

  QNetworkReply* reply = d->post(request, formData, o2GermanyPrivate::SmsCenterSend);
  reply->setProperty("message", QVariant::fromValue(message));
}

//////////////////////////////////////////////////////////////////////////

void o2Germany::updateBalance()
{
  // Nothing, this gets done due login
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> o2Germany::messageTypes() const
{
  return QList<MessageType>()
      << MessageType(tr("Standard"), "standard", QIcon(), QString(), 160, 2000, -1);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#define O2_GERMANY_UUID "7f1c4598-e675-4bed-ae64-9d35485f0b34"

class o2GermanyProviderPlugin : public QObject
                              , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

public:
  o2GermanyProviderPlugin()
  {
    infoList << ProviderInfo(tr("o2 Germany"), O2_GERMANY_UUID, QIcon(":/providers/o2_germany/o2_germany.png"), QLocale::Germany);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& uuid) const
  {
    if ( uuid == O2_GERMANY_UUID )
      return new o2Germany;
    
    return NULL;
  }

protected:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(o2_germany, o2GermanyProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "o2_germany.moc"
