/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "messagetextfield.hpp"
#include "providers/qgsmcodec.h"

// Global includes
#include <QtGui/QKeyEvent>
#include <QtGui/QPainter>
#include <QtGui/QFontMetrics>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MessageTextFieldPrivate
{
public:
  MessageTextFieldPrivate()
  : codec(&gsmCodec)
  , displayMode(MessageTextField::PlatformDisplayMode)
  {
  }

  QGsmCodec gsmCodec;
  QTextCodec* codec;
  MessageTextField::DisplayMode displayMode;

  QPalette standardPalette;
  QPalette platformPalette;
};

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MessageTextField::MessageTextField(QWidget* parent /* = 0 */)
: QPlainTextEdit(parent)
, d(new MessageTextFieldPrivate)
{
  // Save the palettes
  d->standardPalette = palette();

#ifdef Q_WS_MAEMO_5
  // Adjust the palette a little bit to stay with the Maemo style
  d->platformPalette = palette();
  d->platformPalette.setColor(QPalette::Text, d->platformPalette.color(QPalette::ButtonText));
  d->platformPalette.setColor(QPalette::Base, d->platformPalette.color(QPalette::Window));
  setPalette(d->platformPalette);

  // Maemo style has no frame
  setFrameShape(QFrame::NoFrame);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

MessageTextField::~MessageTextField()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MessageTextField::setDisplayMode(DisplayMode displayMode)
{
  // Don't set twice
  if ( d->displayMode == displayMode )
    return;

  // Remember the display mode
  d->displayMode = displayMode;

#ifdef Q_WS_MAEMO_5
  switch(displayMode)
  {
  case StandardDisplayMode:
    setPalette(d->standardPalette);
    setFrameShape(QFrame::StyledPanel);
    break;

  case PlatformDisplayMode:
    setPalette(d->platformPalette);
    setFrameShape(QFrame::NoFrame);
    break;

  default:
    break;
  }
#endif // Q_WS_MAEMO_5

  // Schedule the redraw event
  update();
}

//////////////////////////////////////////////////////////////////////////

MessageTextField::DisplayMode MessageTextField::displayMode() const
{
  return d->displayMode;
}

//////////////////////////////////////////////////////////////////////////

/*void MessageTextField::setCodec(QTextCodec* codec)
{
  if ( d->codec == codec )
    return;

  d->codec = codec;
}

//////////////////////////////////////////////////////////////////////////

QTextCodec* MessageTextField::codec() const
{
  return d->codec;
}*/

//////////////////////////////////////////////////////////////////////////

void MessageTextField::keyPressEvent(QKeyEvent* event)
{
  // Get the entered text
  QString text = event->text();
  if ( !text.isEmpty() && (text[0].isPrint() || text[0] == QLatin1Char('\n')) )
  {
    // Get the two byte code
    unsigned short twoByte = d->gsmCodec.twoByteFromUnicode( text.at(0) );

    if ( twoByte == 0x10 ) // Unsupported character
    {
      // Trash the event
      event->setAccepted(false);
      return;
    }
  }

  QPlainTextEdit::keyPressEvent(event);
}

//////////////////////////////////////////////////////////////////////////

void MessageTextField::paintEvent(QPaintEvent* event)
{
#ifdef Q_WS_MAEMO_5
  // Draw "prettier" on Maemo?
  if ( d->displayMode == PlatformDisplayMode )
  {
    // Draw the underlines like in the Maemo conversations application
    QFontMetrics fontMetrics = this->fontMetrics();
    QRect rect = this->rect();

    // The line color is taken from the Conversations application
    QColor lineColor(32, 32, 32);

    // Setup the painter
    QPainter painter;
    painter.begin(viewport());
    painter.setPen( QPen(lineColor) );

    // Define a jump offset
    int jump = fontMetrics.lineSpacing(); //fontMetrics.ascent() + fontMetrics.underlinePos();

    // Draw the lines
    for(int y=contentOffset().y()+jump; y < rect.height(); y += jump)
      painter.drawLine(rect.left(), y, rect.right(), y);

    painter.end();
  }
#endif // Q_WS_MAEMO_5

  QPlainTextEdit::paintEvent(event);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
