/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "accounts.hpp"
#include "accountconfig.hpp"
#include "providerplugin.hpp"

// Global includes
#include <QtCore/QSet>
#include <QtCore/QPluginLoader>
#include <QtCore/QFileInfo>
#include <QtCore/QDir>
#include <QtCore/QTextStream>
#include <QtCore/QCoreApplication>
#include <QtCore/QPointer>
#include <QtGui/QDesktopServices>
//#include <QSystemStorageInfo>

// Use the QtMobility namespace
//QTM_USE_NAMESPACE

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class AccountsPrivate
{
public:
  AccountsPrivate()
  : programDir( QDesktopServices::storageLocation(QDesktopServices::HomeLocation) )
  {
    /*QSystemStorageInfo storageInfo;
    foreach(const QString& drive, storageInfo.logicalDrives())
    {
      if ( storageInfo.typeForDrive(drive) != QSystemStorageInfo::InternalDrive )
        continue;

      // Try to create a program directory and step into it
      programDir = QDir(drive);
      programDir.mkdir(".web2sms");
      programDir.cd(".web2sms");
      break;
    }*/

    programDir.mkdir(".web2sms");
    programDir.cd(".web2sms");
  }

  // Properties
  QDir programDir;

  QHash<QString, AccountConfig> accounts;
  QList<ProviderPlugin*> providers;
  /// A hash map, key=alias; value=provider
  QHash<QString, QPointer<ProviderInterface> > aliasToProvider;

  // Functions
};

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

Accounts::Accounts(QObject* parent /* = 0 */)
: QObject(parent)
, d(new AccountsPrivate)
{
}

//////////////////////////////////////////////////////////////////////////

Accounts::~Accounts()
{
  // Clean up
  qDeleteAll(d->providers);
  d->providers.clear();

  // Those providers which still exist get deleted now
  typedef QPointer<ProviderInterface> ProviderInterfacePtr;
  foreach(const ProviderInterfacePtr& provider, d->aliasToProvider)
    provider->deleteLater();

  delete d;
}

//////////////////////////////////////////////////////////////////////////

void Accounts::loadSettings()
{
  // Open the config file in read only mode
  QFile file(d->programDir.absoluteFilePath("accounts.xml"));
  if ( !file.open(QIODevice::ReadOnly) )
    return;

  // Read the document
  QDomDocument document;
  document.setContent(&file);

  // Make sure we have opened the right format
  QDomElement accountsElement = document.documentElement();
  if ( accountsElement.tagName() != "accounts" )
    return;

  // List the "accounts" elements
  QDomNodeList nodeList = accountsElement.elementsByTagName("account"); // Foreach does not work with QDomNodeList
  for(int i=0; i < nodeList.size(); ++i)
  {
    // Convert to a element
    QDomElement element = nodeList.at(i).toElement();

    // Read the config entry
    AccountConfig config;
    config.setAlias( element.attribute("alias") );
    config.setProviderId( element.attribute("providerId") );
    config.setCustomData( QByteArray::fromBase64(element.attribute("customData").toUtf8()) );

    // Append the configuration
    d->accounts.insert(config.alias(), config);

    // Emit a signal that we've added the account
    emit accountAdded(config);
  }
}

//////////////////////////////////////////////////////////////////////////

void Accounts::saveSettings() const
{
  QDomDocument document;
  QDomElement accountsElement = document.createElement("accounts");

  // List the configurations
  foreach(const AccountConfig& config, d->accounts)
  {
    // Save the information
    QDomElement element = document.createElement("account");
    element.setAttribute("alias", config.alias());
    element.setAttribute("providerId", config.providerId());

    // Use the running interface or the old settings if it was not even loaded
    ProviderInterface* ptr = d->aliasToProvider.value(config.alias());
    QByteArray customData = ptr ? ptr->saveSettings() : config.customData();

    // Save the custom data
    element.setAttribute("customData", QString(customData.toBase64()));

    // Make us a child of "accounts"
    accountsElement.appendChild(element);
  }

  // Make "accounts" the document element
  document.appendChild(accountsElement);

  // Open a text file
  QFile file(d->programDir.absoluteFilePath("accounts.xml"));
  if ( !file.open(QIODevice::WriteOnly) )
    return; // TODO: Error handling

  // Save the data to the text stream
  QTextStream stream(&file);
  document.save(stream, 0);
}

//////////////////////////////////////////////////////////////////////////

void Accounts::loadPlugins()
{
  QDir dir = QCoreApplication::applicationDirPath();

  // Try to change into the directory
  if ( !dir.cd("providers") )
    return;

  // Iterate the files in the directory
  foreach(const QFileInfo& fileInfo, dir.entryInfoList(QDir::Files | QDir::NoSymLinks | QDir::NoDotAndDotDot))
  {
    // Is it a valid library file?
    if ( !QLibrary::isLibrary(fileInfo.absoluteFilePath()) )
      continue;

    // Try to load the plugin
    QPluginLoader loader(fileInfo.absoluteFilePath());
    loader.setLoadHints(QLibrary::ExportExternalSymbolsHint);

    if (QObject* instance = loader.instance())
    {
      // Check if we can get the plugin
      ProviderPlugin* plugin = qobject_cast<ProviderPlugin*>(instance);
      if ( plugin )
      {
        // Register the provider
        d->providers.append(plugin);

        // Stop here, don't unload the plugin
        continue;
      }
    }

    // Unload the plugin if we couldn't load it correctly
    loader.unload();
  }
}

//////////////////////////////////////////////////////////////////////////

QList<ProviderInfo> Accounts::providers() const
{
  QList<ProviderInfo> infoList;

  // Merge the provider info entries
  foreach(const ProviderPlugin* plugin, d->providers)
    infoList.append( plugin->info() );

  return infoList;
}

//////////////////////////////////////////////////////////////////////////

void Accounts::addAccount(const AccountConfig& config)
{
  // Already there?
  if ( d->accounts.contains(config.alias()) )
    return;

  // Simply insert the configuration
  d->accounts.insert(config.alias(), config);

  // Emit a signal that we've added the account
  emit accountAdded(config);
}

//////////////////////////////////////////////////////////////////////////

void Accounts::removeAccount(const QString& alias)
{
  // Not found?
  if ( !d->accounts.contains(alias) )
    return;

  // Get the account config
  AccountConfig config = d->accounts.value(alias);

  // Remove the provider pointer
  d->aliasToProvider.remove(alias);
  d->accounts.remove(alias);

  // Emit a signal that we've removed the account
  emit accountRemoved(config);
}

//////////////////////////////////////////////////////////////////////////

QList<AccountConfig> Accounts::accounts() const
{
  return d->accounts.values();
}

//////////////////////////////////////////////////////////////////////////

AccountConfig Accounts::accountForAlias(const QString& alias) const
{
  return d->accounts.value(alias);
}

//////////////////////////////////////////////////////////////////////////

ProviderInterface* Accounts::providerForAlias(const QString& alias)
{
  // Find a existing pointer, if any
  QHash<QString, QPointer<ProviderInterface> >::const_iterator iter = d->aliasToProvider.find(alias);
  if ( iter != d->aliasToProvider.constEnd() )
    return iter.value();

  // Get the provider id
  QUuid providerId = d->accounts.value(alias).providerId();

  // Make sure the providerId is valid (so is the alias)
  if ( providerId.isNull() )
    return NULL;

  // Create a new instance of the provider
  foreach(const ProviderPlugin* plugin, d->providers)
  {
    // Try to create a new instance using the current plugin
    ProviderInterface* ptr = plugin->createProvider(providerId);

    // Continue if nothing happened
    if ( !ptr )
      continue;

    // Load the account config
    AccountConfig config = accountForAlias(alias);
    if ( config.isValid() )
      ptr->loadSettings(config.customData());

    // Insert the provider into our hash map
    d->aliasToProvider.insert(alias, ptr);

    // Stop here
    return ptr;
  }

  // Nothing worked :(
  return NULL;
}

//////////////////////////////////////////////////////////////////////////

ProviderInfo Accounts::providerInfoForAlias(const QString& alias) const
{
  // Get the account config
  AccountConfig config = accountForAlias(alias);

  if ( !config.isValid() )
    return ProviderInfo();

  // Iterate the plugins
  foreach(const ProviderPlugin* plugin, d->providers)
  {
    // Get the plugin info list
    QList<ProviderInfo> infoList = plugin->info();

    // Iterate the list
    foreach(const ProviderInfo& info, infoList)
    {
      // If the id is the searched one, we're done
      if ( info.providerId() == config.providerId() )
        return info;
    }
  }

  // Nothing found :(
  return ProviderInfo();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
