/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "python.hpp"
#include "../providerplugin.hpp"
#include "sip/sipAPIweb2sms.h"

// Global includes
#include <QtCore/QDir>
#include <QtCore/QCoreApplication>
#include <QtCore/QSharedPointer>
#include <Python.h>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

extern "C" void initweb2sms();

static void doDecRef(PyObject* object)
{
  Py_DECREF(object);
}

typedef QSharedPointer<PyObject> pyptr;

pyptr py(PyObject* object)
{
  return pyptr(object, doDecRef);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class PythonPlugin : public QObject
                   , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

  struct ScriptInfo
  {
    pyptr script;
    ProviderPlugin* pluginInstance;
  };

public:
  PythonPlugin()
  {
    // Initialize python
    Py_Initialize();

    // Initialize the web2sms module
    struct _inittab buildin_modules[] = {
      {"web2sms", initweb2sms},
      {NULL, NULL}
    };

    PyImport_ExtendInittab(buildin_modules);

    PyObject* sysPath = PySys_GetObject("path");
    if ( sysPath )
    {
      QDir dir = QCoreApplication::applicationDirPath();
      dir.cd("providers");

      QByteArray currentPath = dir.absolutePath().toUtf8();
      PyObject* path = PyString_FromString(currentPath.constData());
      PyList_Append(sysPath, path);
    }

    // Load the python scripts
    loadScripts();
  }

  ~PythonPlugin()
  {
    // Delete the instances
    foreach(const ScriptInfo& scriptInfo, scripts)
      delete scriptInfo.pluginInstance;

    // The scripts have to be cleared before python is finalized
    scripts.clear();

    // Cleanup python code
    Py_Finalize();
  }

  void loadScripts()
  {
    // TODO: Add cleanup code for reload

    // Get the application directory
    QDir dir = QCoreApplication::applicationDirPath();

    // Try to move into the providers directory
    if ( !dir.cd("providers") )
      return;

    // Setup the file name filter
    QStringList nameFilter;
    nameFilter.append( "provider_*.py" );

    // List all scripts inside
    foreach(const QFileInfo& fileInfo, dir.entryInfoList(nameFilter, QDir::Files))
    {
      // Create the script name
      PyObject* scriptName = PyString_FromString( QString("%1").arg(fileInfo.baseName()).toUtf8().constData() );

      // Load the script
      pyptr script = py(PyImport_Import(scriptName));
      Py_DECREF(scriptName);

      if ( !script )
      {
        if ( PyErr_Occurred() )
          PyErr_Print();

        continue;
      }
      
      // Get the entry function
      pyptr entryFunction = py(PyObject_GetAttrString(script.data(), "qt_plugin_instance"));
      if ( !(entryFunction && PyCallable_Check(entryFunction.data())) )
      {
        if ( PyErr_Occurred() )
          PyErr_Print();

        continue;
      }

      pyptr args = py(PyTuple_New(0));
      PyObject* pluginInstance = PyObject_CallObject(entryFunction.data(), args.data());
      if ( pluginInstance )
      {
        if ( !sipCanConvertToType(pluginInstance, sipType_ProviderPlugin, 0) )
        {
          Py_DECREF(pluginInstance);
          continue;
        }

        int isErr = 0;
        void* voidInstance = sipConvertToType(pluginInstance, sipType_ProviderPlugin, NULL, 0, NULL, &isErr);
        if ( voidInstance )
        {
          sipTransferBreak(pluginInstance);

          ProviderPlugin* plugin = reinterpret_cast<ProviderPlugin*>(voidInstance);

          ScriptInfo info;
          info.pluginInstance = plugin;
          info.script = script;

          scripts.append(info);
        }
      }
    }
  }

  // ProviderPlugin overrides
  QList<ProviderInfo> info() const
  {
    QList<ProviderInfo> info;
    foreach(const ScriptInfo& scriptInfo, scripts)
      info.append( scriptInfo.pluginInstance->info() );
    return info;
  }

  ProviderInterface* createProvider(const QString& name) const
  {
    foreach(const ScriptInfo& scriptInfo, scripts)
    {
      ProviderInterface* provider = scriptInfo.pluginInstance->createProvider(name);
      if ( provider )
        return provider;
    }

    return NULL;
  }

protected:
  QList<ScriptInfo> scripts;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(python, PythonPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "python.moc"
