/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "mainwindow.hpp"
#include "messagewidget.hpp"
#include "providers/providerinterface.hpp"
#include "providers/accountsdialog.hpp"
#include "providers/accounts.hpp"

// Global includes
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QMenuBar>
#include <QtGui/QCloseEvent>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MainWindowPrivate : public QObject
{
  Q_OBJECT

public:
  MainWindowPrivate()
  : accounts(new Accounts)
  {}

  MainWindow* self;

  // Properties
  AccountsPtr accounts;
  MessageWidget* messageWidget;

  // Functions
  void init();

  void login();
  void logout();

public slots:
  // Ui callbacks
  void onEditAccounts();
};

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::init()
{
  self->setWindowTitle( tr("Web2SMS") );

  accounts->loadPlugins();
  accounts->loadSettings();

  messageWidget = new MessageWidget(self);
  messageWidget->setAccounts(accounts);

  // Set the message widget as central widget
  self->setCentralWidget(messageWidget);

  // Setup the menu
  QMenu* menu = self->menuBar()->addMenu(QString());
  menu->addAction(tr("Accounts ..."), this, SLOT(onEditAccounts()));

#if !defined(Q_WS_S60) && !defined(Q_WS_MAEMO_5)
  // We have to set a window title, otherwise Windows won't show the menu
  menu->setTitle("menu");
#endif
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onEditAccounts()
{
  // Show the accounts dialog
  AccountsDialog dialog(self);
  dialog.setAccounts(accounts);

  // Run the dialog (we don't care about the exit code)
  dialog.exec();

  // Always save the account settings when the dialog is finished
  accounts->saveSettings();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MainWindow::MainWindow(QWidget *parent /* = 0 */)
: QMainWindow(parent)
, d(new MainWindowPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MainWindow::~MainWindow()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::init()
{
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::closeEvent(QCloseEvent* closeEvent)
{
  // Logout first?
  ProviderInterface* provider = d->messageWidget->provider();

  if ( provider && provider->isLoggedIn() )
  {
    // Ignore the close event
    closeEvent->ignore();

    // Show a dialog that we're logging out first
    QProgressDialog* progressDialog = new QProgressDialog(this);
    progressDialog->setAttribute(Qt::WA_DeleteOnClose);
    progressDialog->setWindowTitle( tr("Logging out ...") );
    progressDialog->setRange(0, 0);
    progressDialog->show();

    // Close the information dialog and this window
    connect(provider, SIGNAL(logoutReply(bool)), progressDialog, SLOT(accept()));
    connect(provider, SIGNAL(logoutReply(bool)), this, SLOT(close()));

    // Log out now
    provider->logout();
  }
  else
  {
    QWidget::closeEvent(closeEvent);
  }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "mainwindow.moc"
