/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local webtexter includes
// These have to come before Qt is included because it redefines "signals"
#include "src/settings.h"
#include "src/http_sender.h"
#include "src/vodafone_sender.h"

// Local includes
#include "webtexter.hpp"
#include "webtexteraccountsettingsdialog.hpp"
#include "../providerplugin.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtCore/QLibrary>
#include <QtCore/QDebug>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class WebTexterProviderPrivate : public QObject
{
  Q_OBJECT

public:
  // Properties
  AppSettings2 settings;

  // Functions
};

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

WebTexterProvider::WebTexterProvider(int provider, QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new WebTexterProviderPrivate)
{
  // Set the default provider
  d->settings.setProvider( static_cast<Providers>(provider) );
}

//////////////////////////////////////////////////////////////////////////

WebTexterProvider::~WebTexterProvider()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void WebTexterProvider::loadSettings(const QByteArray& data)
{
  // No data? Nothing to do
  if ( data.isEmpty() )
    return;

  QDataStream stream(data);

  QByteArray userName, password, number, proxyUrl;
  int provider, curlTimeout;
  bool useProxyScript, saveMsg, orientationEnabled, emulator, extraLogging;

  stream >> userName
         >> password
         >> number
         >> provider
         >> useProxyScript
         >> saveMsg
         >> orientationEnabled
         >> emulator
         >> extraLogging
         >> proxyUrl
         >> curlTimeout;

  d->settings.setUsername(userName);
  d->settings.setPassword(password);
  d->settings.setNumber(number);
  d->settings.setProvider( static_cast<Providers>(provider) );
  d->settings.setUseProxyScript(useProxyScript);
  d->settings.setSaveMsg(saveMsg);
  d->settings.setOrientationEnabled(orientationEnabled);
  d->settings.setEmulator(emulator);
  d->settings.setExtraLogging(extraLogging);
  d->settings.setProxyUrl(proxyUrl);
  d->settings.setCurlTimeout(curlTimeout);
}

//////////////////////////////////////////////////////////////////////////

QByteArray WebTexterProvider::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->settings.userName()
         << d->settings.password()
         << d->settings.number()
         << (qint32)d->settings.provider()
         << d->settings.useProxyScript()
         << d->settings.saveMsg()
         << d->settings.orientationEnabled()
         << d->settings.emulator()
         << d->settings.extraLogging()
         << d->settings.proxyUrl()
         << d->settings.curlTimeout();

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool WebTexterProvider::hasLoginData() const
{
  return !d->settings.userName().isEmpty();
}

//////////////////////////////////////////////////////////////////////////

bool WebTexterProvider::isLoggedIn() const
{
  return false;
}

//////////////////////////////////////////////////////////////////////////

int WebTexterProvider::freeSMSCount() const
{
  return -1;
}

//////////////////////////////////////////////////////////////////////////

bool WebTexterProvider::showAccountSettingsDialog(QWidget* parent)
{
  WebTexterAccountSettingsDialog dialog(parent);
  dialog.setSettings(d->settings);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->settings = dialog.settings();
    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<QAction*> WebTexterProvider::actions() const
{
  return QList<QAction*>();
}

//////////////////////////////////////////////////////////////////////////

void WebTexterProvider::login()
{
  // Simply return true
  emit loginReply(true);
}

//////////////////////////////////////////////////////////////////////////

void WebTexterProvider::logout()
{
}

//////////////////////////////////////////////////////////////////////////

int WebTexterProvider::sendMessage(const QList<ContactInfo>& receivers, const QString& text)
{
  // Currently we only accept one receiver here
  if ( receivers.size() != 1 )
    return -1;

  QByteArray to = receivers.first().number().toUtf8();
  QByteArray msg = text.toUtf8();

  // Construct a AppSettings instance
  AppSettings settings;
  d->settings.fillAppSettings(&settings);

  gint sent = send_msg(&settings, to.data(), msg.data(), NULL );

  switch(sent)
  {
  case SUCCESS:
    break;

  case ERROR_LOGIN:
    emit error( tr("Error logging in!") );
    break;

  case ERROR_SEND:
    emit error( tr("Unable to send message!") );
    break;

  default:
    emit error( tr("Unknown error happened!") );
    break;
  }

  return -1;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class WebTexterProviderPlugin : public QObject
                              , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

public:
  WebTexterProviderPlugin()
  {
    // We have to load libcurl by ourself, don't know why ...
    QLibrary curlLoader("libcurl");
    curlLoader.load();

    infoList
        << ProviderInfo(tr("Vodafone Ireland"), "vodafone_ireland", QIcon(":/providers/webtexter/vodafone_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("o2 Ireland"), "o2_ireland", QIcon(":/providers/webtexter/o2_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("3 Ireland"), "3_ireland", QIcon(":/providers/webtexter/three_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("Meteor Ireland"), "meteor_ireland", QIcon(":/providers/webtexter/meteor_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("Blueface Ireland"), "blueface_ireland", QIcon(":/providers/webtexter/blueface_ireland.png"), QLocale::Ireland)
        << ProviderInfo(tr("VoipCheap.com"), "voipcheap", QIcon(":/providers/webtexter/voipcheap.com.png"), QLocale::AnyCountry)
        << ProviderInfo(tr("SMSdiscount.com"), "smsdiscount", QIcon(":/providers/webtexter/smsdiscount.com.png"), QLocale::AnyCountry)
        << ProviderInfo(tr("Lowratevoip.com"), "lowratevoip", QIcon(), QLocale::AnyCountry)
        << ProviderInfo(tr("Other Betamax"), "otherbetamax", QIcon(), QLocale::AnyCountry)
        << ProviderInfo(tr("WebSMS.ru"), "websms.ru", QIcon(":/providers/webtexter/websms.ru.png"), QLocale::RussianFederation)
        << ProviderInfo(tr("Exetel Australia"), "exetel_australia", QIcon(), QLocale::Australia);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterfacePtr createProvider(const QString& name) const
  {
    int provider = -1;

    // Now check the big list ..
    if ( name == "vodafone_ireland" )
      provider = VODAFONE;
    else if ( name == "o2_ireland" )
      provider = O2;
    else if ( name == "3_ireland" )
      provider = THREE;
    else if ( name == "meteor_ireland" )
      provider = METEOR;
    else if ( name == "blueface_ireland" )
      provider = BLUEFACE;
    else if ( name == "voipcheap" )
      provider = VOIPCHEAP;
    else if ( name == "smsdiscount" )
      provider = SMSDISCOUNT;
    else if ( name == "lowratevoip" )
      provider = LOWRATEVOIP;
    else if ( name == "otherbetamax" )
      provider = OTHER_BETAMAX;
    else if ( name == "websms.ru" )
      provider = WEBSMSRU;
    else if ( name == "exetel_australia" )
      provider = EXETEL;

    // Should we create a new provider?
    if ( provider != -1 )
      return ProviderInterfacePtr(new WebTexterProvider(provider));

    return ProviderInterfacePtr();
  }

protected:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(webtexter, WebTexterProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "webtexter.moc"
