/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "mainwindow.hpp"
#include "messagewidget.hpp"
#include "providers/providerinterface.hpp"
#include "contacts.h"

// REMOVE THIS INCLUDE
#include "providers/o2_germany/o2_germany.hpp"

// Global includes
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>
#include <QtGui/QMenuBar>
#include <QtGui/QCloseEvent>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MainWindowPrivate : public QObject
{
  Q_OBJECT

public:
  MainWindowPrivate()
  {}

  MainWindow* self;

  // Properties
  MessageWidget messageWidget;

  ProviderInterface* provider;

  // Functions
  void init();

  void login();
  void logout();

public slots:
  // Provider callbacks
  void onLoginReply(bool success);
  void onError(const QString& message);

  // Ui callbacks
  void onShowAccountSettings();
  void onSelectProvider();
};

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::init()
{
  // Update the window title
  self->setWindowTitle( tr("Web2SMS") );

  // Set the message widget as central widget
  self->setCentralWidget(&messageWidget);

  // Initialize a o2 Germany instance
  provider = new o2Germany(this);

  // Set the provider to the message widget
  messageWidget.setProvider(provider);

  // Connect signals and slots
  connect(provider, SIGNAL(error(QString)), this, SLOT(onError(QString)));
  connect(provider, SIGNAL(loginReply(bool)), this, SLOT(onLoginReply(bool)));

  // Setup the menu
  QMenu* menu = self->menuBar()->addMenu(QString());
  menu->addAction(tr("Account settings"), this, SLOT(onShowAccountSettings()));
  menu->addAction(tr("Select provider"), this, SLOT(onSelectProvider()));

#ifdef Q_OS_WIN
  // We have to set a window title, otherwise Windows won't show the menu
  menu->setTitle("menu");
#endif
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::login()
{
  QProgressDialog* progressDialog = new QProgressDialog(self);
  progressDialog->setAttribute(Qt::WA_DeleteOnClose);
  progressDialog->setWindowTitle( tr("Logging in ...") );
  progressDialog->setRange(0, 0);
  progressDialog->show();

  connect(provider, SIGNAL(loginReply(bool)), progressDialog, SLOT(close()));

  provider->login();
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onLoginReply(bool success)
{
  if ( success )
    return;

  // Show a message that tells the user that's something wrong
  QMessageBox::critical(self, self->windowTitle(), tr("Login failed, please check your username and password"));
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onError(const QString& message)
{
  QMessageBox::critical(self, self->windowTitle(), message);
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onShowAccountSettings()
{
  // Show the account settings dialog
  if ( !provider->showAccountSettingsDialog(self) )
    return;

  // If the provider now has login data, do it!
  if ( provider->hasLoginData() )
    login();
}

//////////////////////////////////////////////////////////////////////////

void MainWindowPrivate::onSelectProvider()
{
  // Blame me ...
  QMessageBox::information(self, self->windowTitle(), tr("Currently there is only o2 Germany :("));
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MainWindow::MainWindow(QWidget *parent /* = 0 */)
: QMainWindow(parent)
, d(new MainWindowPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MainWindow::~MainWindow()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::init()
{
  if ( !d->provider->hasLoginData() )
  {
    QMessageBox::information(this, tr("First start notice"), tr("This program only works with o2 Germany. Use at your own risk, bugs may be reported on the project website."));

    d->onShowAccountSettings();
  }
  else
    d->login();
}

//////////////////////////////////////////////////////////////////////////

void MainWindow::closeEvent(QCloseEvent* closeEvent)
{
  // Logout first?
  if ( d->provider && d->provider->isLoggedIn() )
  {
    // Ignore the close event
    closeEvent->ignore();

    // Show a dialog that we're logging out first
    QProgressDialog* progressDialog = new QProgressDialog(this);
    progressDialog->setAttribute(Qt::WA_DeleteOnClose);
    progressDialog->setWindowTitle( tr("Logging out ...") );
    progressDialog->setRange(0, 0);
    progressDialog->show();

    // Close the information dialog and this window
    connect(d->provider, SIGNAL(logoutReply(bool)), progressDialog, SLOT(close()));
    connect(d->provider, SIGNAL(logoutReply(bool)), this, SLOT(close()));

    // Log out now
    d->provider->logout();
  }
  else
    QWidget::closeEvent(closeEvent);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "mainwindow.moc"
