/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/* IMPORTS */
#define _GNU_SOURCE
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <glib.h>

#include "profile.h"
#include "config_data_vpnc.h"
#include "fileaccess_vpnc.h"

/* EXPORTS */

/**
 * Load vpnc-configuration from specified file.
 * This function does not manipulate path.
 * Result must be freed if not longer used.
 * 
 * @param path pointer to a string containing the path
 * @return freshly g_alloced config_data_vpnc with parsed config-values
 *         or NULL in case of an error
 */
config_data_vpnc *fileaccess_vpnc_load_file(char *path) {

	config_data_vpnc *data = config_data_vpnc_alloc();

	FILE *fp=fopen(path, "r");
	if(fp==NULL) {
		perror("fopen failed");
		return NULL;
	}


	char *line = NULL;
	size_t line_length = 0;


	for(;;) {
		ssize_t llen;

		llen = getline(&line, &line_length, fp);
		if (llen == -1 && feof(fp))
			break;
		if (llen == -1) {
			perror("getline failed");
			break;
		}

		if(line[0]=='#')
			continue;

		/* strip \n */
		line[llen-1]='\0';

#ifdef DEBUG
		printf("line: '%s'\n", line);
#endif

		if(strstr(line, "IPSec gateway ")==line) {
			data->ipsec_gw=g_strdup(line+sizeof("IPSec gateway"));
		}
		else if(strstr(line, "IPSec ID ")==line) {
			data->ipsec_id=g_strdup(line+sizeof("IPSec ID"));
		}
		else if(strstr(line, "IPSec Secret ")==line) {
			data->ipsec_secret=g_strdup(line+sizeof("IPSec Secret"));
		}
		else if(strstr(line, "Xauth username ")==line) {
			data->xauth_user=g_strdup(line+sizeof("Xauth username"));
		}
		else if(strstr(line, "Xauth password ")==line) {
			data->xauth_pw=g_strdup(line+sizeof("Xauth password"));
		}
		else if(strstr(line, "Xauth interactive")==line) {
			data->xauth_interactive=TRUE;
		}
		else if(strstr(line, "Domain ")==line) {
			data->domain=g_strdup(line+sizeof("Domain"));
		}
	}

	if(line!=NULL)
		free(line);


	return data;

}

/**
 * Write vpnc config-data to specified file.
 * This function does not manipulate path or data.
 * 
 * @param path a pointer to a string containing the path
 * @param data the configuration data, NULL items will not be written
 * @return TRUE if data has been successfully written
 */
gboolean fileaccess_vpnc_save_file(char *path, config_data_vpnc *data) {

	FILE *fp=fopen(path, "w");
	if(fp==NULL) {
		perror("fopen failed");
		return FALSE;
	}


	fprintf(fp, "# Generated by maemo vpnc-gui\n");
	fprintf(fp, "# Michael 'ScriptKiller' Arndt <scriptkiller@gmx.de>\n");
	fprintf(fp, "# DON'T EDIT BY HAND (unless u really know what u're doing)\n\n");


	if(data->ipsec_gw)
		fprintf(fp, "IPSec gateway %s\n", data->ipsec_gw);
	if(data->ipsec_id)
		fprintf(fp, "IPSec ID %s\n", data->ipsec_id);
	if(data->ipsec_secret)
		fprintf(fp, "IPSec Secret %s\n", data->ipsec_secret);
	if(data->xauth_user)
		fprintf(fp, "Xauth username %s\n", data->xauth_user);
	if(data->xauth_pw)
		fprintf(fp, "Xauth password %s\n", data->xauth_pw);
	if(data->domain)
		fprintf(fp, "Domain %s\n", data->domain);
	if(data->xauth_interactive==TRUE)
		fprintf(fp, "Xauth interactive\n");

	fclose(fp);

#ifdef DEBUG
	puts("Config written");
#endif

	return TRUE;
}
