/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#include "printjob.h"
#include "urpoconnection.h"
#include <QFileInfo>
#include <stdexcept> // Thowing execption for start()
#include <cstdlib>  // For random numbers generating temp file name

PrintJob::PrintJob(UrpoConnection* connection) :
    UrpoJob(connection)
{
    currentProcess_ = 0;
}

PrintJob::~PrintJob()
{
    if( currentProcess_ )
        delete currentProcess_;
}


void PrintJob::freeProcess()
{
    if( currentProcess_)
    {
        // Delete current process when event loop
        //
        // If we are processing a signal from this process
        // (process is finished), we can't delete process
        // (actually, we can delete, but we get null pointer
        // and segmentation fault !)
        currentProcess_->deleteLater();

        // disvalidate pointer
        currentProcess_ = 0;
    }
}

void PrintJob::startJob()
{
    // See API documentation on header...
    throw( new std::logic_error("PrintJob::starJob() NOT for use!") );
}


void PrintJob::printFile(const QString& path, const QString& printOptions)
{
    // Get information about file
    QFileInfo fileInfo(path);
    if( fileInfo.exists() == false)
    {
        // File not found!
        sendDebugMessage( tr("File %1 not exists.").arg(path));
        fail( tr("File not found") );
        return;
    }

    // new process for use
    freeProcess();
    currentProcess_= newProcess();


    // Copy temperary file to temperary directory
    pathToPrint_ = "/var/tmp/urpo-";

    // Securing unique of temperary files, add random
    // number between 0 and ffffff
    pathToPrint_.append( QString::number( rand() % 0xffffff, 16));

    // In server path, replace space with _
    pathToPrint_.append( fileInfo.fileName().replace(QChar(' '),QChar('_')));

    printOptions_ = printOptions;

    // Copy command
    QString command = "scp ";
    command.append( getConnection()->getKeyOption());
    command.append(fileInfo.absoluteFilePath().replace(QString(" "),QString("\\ ")) );
    command.append(" ");
    command.append( getConnection()->getHostString() );
    command.append(":");
    command.append(pathToPrint_);

    // Next Copy Ready
    connect( currentProcess_, SIGNAL(finished(bool)), this, SLOT(copyReady(bool)));

    currentProcess_->start(command);
}

void PrintJob::copyReady(bool success)
{
    freeProcess();

    if( success == false)
    {
        fail( tr("Failed copying file"));
        return;
    }

    // Running remote printing command
    currentProcess_ = newProcess();
    QString command = " ssh ";
    command.append( getConnection()->getKeyOption()) ;
    command.append(getConnection()->getHostString());
    command.append(" lp ");
    command.append(printOptions_);
    command.append(" ");
    command.append(pathToPrint_);

    // Next print ready
    connect( currentProcess_, SIGNAL(finished(bool)), this, SLOT(printReady(bool)));
    currentProcess_->start(command);

}

void PrintJob::printReady(bool success)
{
    // Print step is ready
    freeProcess();

    if( success == false )
    {
        fail( tr("Print failed"));
        return;
    }

    // delete temperary file
    currentProcess_ = newProcess();
    QString command = " ssh ";
    command.append( getConnection()->getKeyOption()) ;
    command.append(getConnection()->getHostString());
    command.append(" rm ");
    command.append(pathToPrint_);

    // Next delete ready
    connect( currentProcess_, SIGNAL(finished(bool)),this,SLOT(deleteReady(bool)));
    currentProcess_->start(command);
}

void PrintJob::deleteReady(bool success)
{
    // All done!
    freeProcess();
    if( success == false )
    {
        fail(tr("Failed deleting file"));
    }
    else
    {
        // Successed !!!
        finish( Successed );
    }
}

void PrintJob::cancelJob()
{
    // Cancelled by user.
    if( currentProcess_ )
    {
        currentProcess_->terminate();
        freeProcess();
    }
    finish( Cancelled );
}

