/*
 *  txtmsgtomail (Textmessage to EMail)
 *  Copyright (C) 2011 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#define MODEST_ACCOUNTS_GCONF_PATH "/apps/modest/accounts"
#define MODEST_DEFAULT_ACCOUNT_GCONF_PATH "/apps/modest/default_account"
#define GC_ROOT "/apps/maemo/txtmsgtomail"
#define TXTMSG_TO_MAIL_KEY_ENABLED GC_ROOT "/enabled"
#define TXTMSG_TO_MAIL_KEY_SENDER GC_ROOT "/mail_sender"
#define TXTMSG_TO_MAIL_KEY_ACCOUNT GC_ROOT "/mail_account"
#define TXTMSG_TO_MAIL_KEY_RECEIVER GC_ROOT "/mail_receiver"
#define TXTMSG_TO_MAIL_KEY_DISABLE_SMS_FORWARDING GC_ROOT "/disable_sms_forwarding"
#define TXTMSG_TO_MAIL_KEY_ENABLE_CHAT_FORWARDING GC_ROOT "/enable_chat_forwarding"
#define TXTMSG_TO_MAIL_KEY_DISABLE_INCOMING_FORWARDING GC_ROOT "/disable_incoming_forwarding"
#define TXTMSG_TO_MAIL_KEY_ENABLE_OUTGOING_FORWARDING GC_ROOT "/enable_outgoing_forwarding"

static void
_enable_one_of(GtkToggleButton* button, gpointer user_data)
{
  if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
  {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(user_data), TRUE);
  }
}

static void
_enable_disable_forwarding_option_box(HildonCheckButton* button, gpointer user_data)
{
  gtk_widget_set_sensitive(GTK_WIDGET(user_data), hildon_check_button_get_active(button));
}

static void
_fill_account_list(GtkListStore* list_store, const gchar* active_account)
{
  GConfClient* client = gconf_client_get_default();
  g_assert(GCONF_IS_CLIENT(client));
  
  gchar* default_account = NULL;
  if(active_account == NULL)
  {
    default_account = gconf_client_get_string(client, MODEST_DEFAULT_ACCOUNT_GCONF_PATH, NULL);
  }
  else
  {
    default_account = g_strdup(active_account);
  }
  if(default_account == NULL)
  {
    g_warning("no account defined\n");
  }

  GSList* accounts = gconf_client_all_dirs(client, "/apps/modest/accounts", NULL);
  GSList* account_entry = accounts;

  while(account_entry)
  {
    gchar* account = account_entry->data;
    const size_t prefix_len = strlen(MODEST_ACCOUNTS_GCONF_PATH) +1;

    gchar* account_name = gconf_unescape_key(account + prefix_len, strlen(account)-prefix_len);

    gchar* name_key = g_strdup_printf("%s/email", account);
    gchar* display_name_key = g_strdup_printf("%s/fullname", account);

    gchar* name = gconf_client_get_string(client, name_key, NULL);
    gchar* display_name = gconf_client_get_string(client, display_name_key, NULL);

    g_free(display_name_key);
    g_free(name_key);

    GtkTreeIter iter;
    if(g_strcmp0(default_account, account_name)==0)
      gtk_list_store_prepend(list_store, &iter);
    else
      gtk_list_store_append(list_store, &iter);

    gtk_list_store_set(list_store, &iter,
		       0, g_strdup_printf("%s <%s>", display_name, name),
		       1, g_strdup(account_name),
		       2, g_strdup(name),
		       -1);

    g_free(name);
    g_free(account_name);
    g_free(display_name);
    g_free(account);
    account_entry = account_entry->next;
  }
  g_object_unref(client);
  g_free(default_account);
  g_slist_free(accounts);
}

osso_return_t execute(osso_context_t* osso, gpointer user_data, gboolean user_activated)
{
  GConfClient* gconf_client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(gconf_client))
  {
    return;
  }

  GtkWidget* dialog = gtk_dialog_new_with_buttons("Settings",
						  GTK_WINDOW(user_data),
						  GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
						  dgettext("hildon-libs", "wdgt_bd_done"), GTK_RESPONSE_ACCEPT,
						  NULL);
  
  gboolean forwarding_active = gconf_client_get_bool(gconf_client, TXTMSG_TO_MAIL_KEY_ENABLED, NULL);
  gboolean disable_sms_forwarding  = gconf_client_get_bool(gconf_client, TXTMSG_TO_MAIL_KEY_DISABLE_SMS_FORWARDING, NULL);
  gboolean enable_chat_forwarding  = gconf_client_get_bool(gconf_client, TXTMSG_TO_MAIL_KEY_ENABLE_CHAT_FORWARDING, NULL);
  gboolean disable_incoming_forwarding  = gconf_client_get_bool(gconf_client, TXTMSG_TO_MAIL_KEY_DISABLE_INCOMING_FORWARDING, NULL);
  gboolean enable_outgoing_forwarding  = gconf_client_get_bool(gconf_client, TXTMSG_TO_MAIL_KEY_ENABLE_OUTGOING_FORWARDING, NULL);

  gchar* receiver = gconf_client_get_string(gconf_client, TXTMSG_TO_MAIL_KEY_RECEIVER, NULL);
  gchar* account =  gconf_client_get_string(gconf_client, TXTMSG_TO_MAIL_KEY_ACCOUNT, NULL);

  GtkWidget* active_check_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(active_check_button), "Enable Forwarding");
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(active_check_button), forwarding_active);
  
  GtkWidget* disable_sms_forwarding_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* enable_chat_forwarding_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);

  GtkWidget* disable_incoming_forwarding_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  GtkWidget* enable_outgoing_forwarding_button = hildon_gtk_toggle_button_new(HILDON_SIZE_FINGER_HEIGHT);
  
  gtk_button_set_label(GTK_BUTTON(disable_sms_forwarding_button), "SMS");
  gtk_button_set_label(GTK_BUTTON(enable_chat_forwarding_button), "CHAT");
  gtk_button_set_image(GTK_BUTTON(disable_sms_forwarding_button), gtk_image_new_from_icon_name("general_sms", HILDON_ICON_SIZE_FINGER));
  gtk_button_set_image(GTK_BUTTON(enable_chat_forwarding_button), gtk_image_new_from_icon_name("general_chat", HILDON_ICON_SIZE_FINGER));
  gtk_button_set_label(GTK_BUTTON(disable_incoming_forwarding_button), "Incoming");
  gtk_button_set_label(GTK_BUTTON(enable_outgoing_forwarding_button), "Outgoing");

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(disable_sms_forwarding_button), !disable_sms_forwarding);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_chat_forwarding_button), enable_chat_forwarding);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(disable_incoming_forwarding_button), !disable_incoming_forwarding);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_outgoing_forwarding_button), enable_outgoing_forwarding);

  GtkWidget* account_picker_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
							      HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
  GtkListStore* account_store = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
  _fill_account_list(account_store, account);
  GtkWidget* account_selector = hildon_touch_selector_new_text();
  hildon_touch_selector_set_model(HILDON_TOUCH_SELECTOR(account_selector),
				  0, GTK_TREE_MODEL(account_store));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(account_selector),
				   0,0);
  hildon_button_set_title(HILDON_BUTTON(account_picker_button),
			  dgettext("modest", "mail_va_from"));
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(account_picker_button),
				    HILDON_TOUCH_SELECTOR(account_selector));
  GtkWidget* to_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  hildon_gtk_entry_set_input_mode(GTK_ENTRY(to_entry), HILDON_GTK_INPUT_MODE_FULL);
  if(receiver != NULL)
    hildon_entry_set_text(HILDON_ENTRY(to_entry), receiver);
  GtkWidget* to_label = gtk_label_new(dgettext("modest", "mail_va_to"));
  GtkWidget* to_box = gtk_hbox_new(FALSE, 0);
  GtkWidget* forwarding_option_box = gtk_vbox_new(FALSE, 0);
  GtkWidget* sms_chat_box = gtk_hbox_new(TRUE, 0);
  GtkWidget* incoming_outgoing_box = gtk_hbox_new(TRUE, 0);
  GtkWidget* sms_chat_in_out_box = gtk_hbox_new(FALSE, 0);

  gtk_box_pack_start(GTK_BOX(to_box), to_label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(to_box), to_entry, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(sms_chat_box), disable_sms_forwarding_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(sms_chat_box), enable_chat_forwarding_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(incoming_outgoing_box), disable_incoming_forwarding_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(incoming_outgoing_box), enable_outgoing_forwarding_button, TRUE, TRUE, 0);
  g_signal_connect(disable_sms_forwarding_button, "toggled", G_CALLBACK(_enable_one_of), enable_chat_forwarding_button);
  g_signal_connect(enable_chat_forwarding_button, "toggled", G_CALLBACK(_enable_one_of), disable_sms_forwarding_button);
  g_signal_connect(disable_incoming_forwarding_button, "toggled", G_CALLBACK(_enable_one_of), enable_outgoing_forwarding_button);
  g_signal_connect(enable_outgoing_forwarding_button, "toggled", G_CALLBACK(_enable_one_of), disable_incoming_forwarding_button);
  g_signal_connect(active_check_button, "toggled", G_CALLBACK(_enable_disable_forwarding_option_box), forwarding_option_box);

  GtkWidget* contents = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(contents), active_check_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(sms_chat_in_out_box), sms_chat_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(sms_chat_in_out_box), incoming_outgoing_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(forwarding_option_box), sms_chat_in_out_box, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(forwarding_option_box), account_picker_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(forwarding_option_box), to_box, FALSE, FALSE, 0);
  gtk_widget_set_sensitive(forwarding_option_box, forwarding_active);
  gtk_box_pack_start(GTK_BOX(contents), forwarding_option_box, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), contents, TRUE, TRUE, 0);
  gtk_widget_show_all(dialog);
  
  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    if(hildon_check_button_get_active(HILDON_CHECK_BUTTON(active_check_button)) != forwarding_active)
    {
      gconf_client_set_bool(gconf_client, TXTMSG_TO_MAIL_KEY_ENABLED, hildon_check_button_get_active(HILDON_CHECK_BUTTON(active_check_button)), NULL);
    }
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(disable_sms_forwarding_button)) == disable_sms_forwarding)
    {
      gconf_client_set_bool(gconf_client, TXTMSG_TO_MAIL_KEY_DISABLE_SMS_FORWARDING, !disable_sms_forwarding, NULL);
    }
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(enable_chat_forwarding_button)) != enable_chat_forwarding)
    {
      gconf_client_set_bool(gconf_client, TXTMSG_TO_MAIL_KEY_ENABLE_CHAT_FORWARDING, !enable_chat_forwarding, NULL);
    }
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(disable_incoming_forwarding_button)) == disable_incoming_forwarding)
    {
      gconf_client_set_bool(gconf_client, TXTMSG_TO_MAIL_KEY_DISABLE_INCOMING_FORWARDING, !disable_incoming_forwarding, NULL);
    }
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(enable_outgoing_forwarding_button)) != enable_outgoing_forwarding)
    {
      gconf_client_set_bool(gconf_client, TXTMSG_TO_MAIL_KEY_ENABLE_OUTGOING_FORWARDING, !enable_outgoing_forwarding, NULL);
    }
    const gchar* to_address = hildon_entry_get_text(HILDON_ENTRY(to_entry));
    if(g_strcmp0(to_address, receiver)!=0)
    {
      gconf_client_set_string(gconf_client, TXTMSG_TO_MAIL_KEY_RECEIVER, to_address, NULL);
    }
    GtkTreeIter iter;
    hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(account_selector),
				       0, &iter);
    gchar* new_account = NULL;
    gchar* new_address = NULL;
    gtk_tree_model_get(GTK_TREE_MODEL(account_store), &iter,
		       1, &new_account, 2, &new_address,
		       -1);
    if(g_strcmp0(new_account, account)!=0)
    {
      gconf_client_set_string(gconf_client, TXTMSG_TO_MAIL_KEY_ACCOUNT, new_account, NULL);
    }
    if(g_strcmp0(new_address, receiver)!=0)
    {
      gconf_client_set_string(gconf_client, TXTMSG_TO_MAIL_KEY_SENDER, new_address, NULL);
    }
    g_free(new_address);
    g_free(new_account);
  }

  gtk_widget_destroy(dialog);
  g_free(receiver);
  g_free(account);
  g_object_unref(gconf_client);

  return OSSO_OK;
}

osso_return_t save_state(osso_context_t* osso, gpointer user_data)
{
  return OSSO_OK;
}

