/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loginview.h"

/*!
 * \brief LoginView constructor.
 */
LoginView::LoginView(void)
{
    _exitAnim = NULL;
}

/*!
 * \brief Creates the LoginView objects and animations.
 */
void LoginView::createUi(void)
{
    _createLoginBackground();
    _createButtons();
    _createLogo();
    _createSettings();
    _createOauthWizard();
    _createUsersList();

    _unlockRotation();

    _changeOrientation();
    connect(QApplication::desktop(), SIGNAL(resized(int)), SLOT(_changeOrientation()));

    _lockRotation();

    _entryAnim = _createEntryAnimation();
    _entryAnim->start();

    _createAnimation();

    connect(_entryAnim, SIGNAL(finished()), SLOT(_unlockRotation()));
    connect(_entryAnim, SIGNAL(finished()), SLOT(_changeOrientation()));
    connect(_exitButton, SIGNAL(released()), SLOT(exitLogin()));
    connect(_loginButton, SIGNAL(released()), SLOT(emitLoginAccount()));
    connect(_cancelButton, SIGNAL(released()), SLOT(_endLogin()));
    connect(_settingsButton, SIGNAL(released()), _settingsDialog, SLOT(startAnimation()));
    connect(_newAccountButton, SIGNAL(released()), _oauthWizard, SLOT(startAnimation()));
}

void LoginView::_createAnimation(void)
{
    if(_exitAnim != NULL) {
        if(_exitAnim->state() == QAbstractAnimation::Running)
            return;

        delete _exitAnim;
    }

    _exitAnim = _createExitAnimation();
    connect(_exitAnim, SIGNAL(finished()), SIGNAL(done()));
}

/*!
 * \brief This method creates the entry animation for all the objects in the screen.
 * \return anim the animation entry animation.
 */
QAbstractAnimation *LoginView::_createEntryAnimation(void)
{
    QParallelAnimationGroup *anim = new QParallelAnimationGroup(this);

    QEasingCurve easingCurve;
    easingCurve.setType(QEasingCurve::OutBack);
    easingCurve.setOvershoot(2.5);

    _addOpacityAnim(anim, _loginButton, 0);
    _addPosAnim(anim, _loginButton, _loginButtonPos + QPointF(0, 30), _loginButtonPos, easingCurve, 0);

    _addOpacityAnim(anim, _exitButton, 0);
    _addPosAnim(anim, _exitButton, _exitButtonPos + QPointF(0, -30), _exitButtonPos, easingCurve, 0);

    _addOpacityAnim(anim, _exposeButton, 0);
    _addPosAnim(anim, _exposeButton, QPointF(0, -30), QPointF(0, 0), easingCurve, 0);

    _addOpacityAnim(anim, _settingsButton, 0);
    _addPosAnim(anim, _settingsButton, _settingsButtonPos + QPointF(0, 30), _settingsButtonPos, easingCurve, 0);

    _addOpacityAnim(anim, _newAccountButton, 0);
    _addPosAnim(anim, _newAccountButton, _newAccountButtonPos + QPointF(0, 30), _newAccountButtonPos, easingCurve, 0);

    return anim;
}

/*!
 * \brief This method creates the exit animation for all the objects in the screen.
 * \return anim the exit animation.
 * This method takes the entry animation and sets it backward for the exit animation.
 */
QAbstractAnimation *LoginView::_createExitAnimation(void)
{
    QParallelAnimationGroup *anim = qobject_cast<QParallelAnimationGroup *>(_createEntryAnimation());

    anim->setDirection(QAbstractAnimation::Backward);
    return anim;
}

/*!
 * \brief Creates login background.
 */
void LoginView::_createLoginBackground(void)
{
    _background = new QGraphicsRectItem(QRectF(0, 0, 314, 480),this);
    _background->setPos(800 - _background->boundingRect().width(), 0);
    _background->setBrush(QBrush(QColor(255, 255, 255, 100)));
    _background->setPen(Qt::NoPen);
}

/*!
 * \brief Creates all buttons.
 */
void LoginView::_createButtons(void)
{
    _exitButton = new TwitterButton(TwitterButton::Close, "",this);
    _exitButton->setOpacity(0);
    _exitButton->setZValue(4.0);

    _loginButton = new TwitterButton(TwitterButton::GenericDownRight, "Login", this);
    _loginButton->setOpacity(0);
    _loginButton->enable(false);

    _exposeButton = new TwitterButton(TwitterButton::Expose, "", this);
    _exposeButton->setOpacity(0);
    _exposeButton->setZValue(4.0);

    _settingsButton = new TwitterButton(TwitterButton::LoginSettings,"",this);
    _settingsButton->setOpacity(0);
    _settingsButton->enable(false);

    _newAccountButton = new TwitterButton(TwitterButton::NewAccount,"New Account",this);
    _newAccountButton->setOpacity(0);

    _cancelButton = new TwitterButton(TwitterButton::Generic, "Cancel", this);
    _cancelButton->setOpacity(0);
    _cancelButton->setZValue(4.0);
}

/*!
 * \brief Creates logo.
 */
void LoginView::_createLogo(void)
{
    _logo = new QGraphicsPixmapItem(this);
    _logo->setPixmap(QPixmap(":images/resources/logo.png"));
}

void LoginView::_createSettings(void)
{
    _settingsDialog = new LoginSettings(this);
    _settingsDialog->setPos(800,0);
    _settingsDialog->setZValue(5.0);

    connect(_settingsDialog, SIGNAL(removeAcc()),SLOT(_removeAccountDialog()));
    connect(_settingsDialog, SIGNAL(accRemoved()), SLOT(_accountRemoved()));
}

void LoginView::_createOauthWizard(void)
{
    _oauthWizard = new OauthWizard(this);
    _oauthWizard->setPos(800,0);
    _oauthWizard->setZValue(5.0);

    connect(_oauthWizard, SIGNAL(newAccount()),SIGNAL(newAccount()));
}

void LoginView::_createUsersList(void)
{
    usersList = new KineticList(KineticList::Vertical, QSizeF(314,270),this);
    usersList->setScrollBarPolicy(KineticList::Vertical, Qt::ScrollBarAsNeeded);
    usersList->setZValue(2.0);

    connect(usersList, SIGNAL(selected(bool)), _loginButton, SLOT(enable(bool)));
    connect(usersList, SIGNAL(selected(bool)), _settingsButton, SLOT(enable(bool)));
}

/*!
 * \brief Add the opacity Animation
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 */
void LoginView::_addOpacityAnim(QParallelAnimationGroup *anim, TwitterApplicationItem *item, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("opacity");
    animation->setStartValue(0.0);
    animation->setEndValue(1.0);
    animation->setDuration(500);
    animation->setEasingCurve(QEasingCurve::Linear);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Add the position Animation
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void LoginView::_addPosAnim(QParallelAnimationGroup *anim,
                            TwitterApplicationItem *item,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(1000);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Emits login signal if user presses login button.
 */
void LoginView::emitLoginAccount(void)
{
    setLoaderPosition(size().width()/2 - 12, size().height() - 96);

    loaderEntryAnimation();

    usersList->setEnabled(false);
    _newAccountButton->enable(false);
    _settingsButton->enable(false);

    _loginButton->setOpacity(0);
    _cancelButton->setOpacity(1.0);

    emit loginAccount();
}

/*!
 * \brief This slot is called when there is a connection error to delete the loader object.
 */
void LoginView::connectionError()
{
    loaderExitAnimation();

    _exitButton->enable(true);
    _loginButton->enable(true);

    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0);
}

/*!
 * \brief This slot starts the exit animation when user is successfully connected
 * or emits done signal if the animation could not be created.
 */
void LoginView::connected(void)
{
    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0.0);

    loaderExitAnimation();

    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief When user hits cancel, this method deletes loader, emits a signal to
 * abort the login, and puts back login button on the screen.
 */
void LoginView::_endLogin(void)
{
    emit abort();

    usersList->setEnabled(true);
    _newAccountButton->enable(true);
    _settingsButton->enable(true);

    loaderExitAnimation();

    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0);
}

void LoginView::exitLogin(void)
{
    _dialog = new DialogView(DialogView::Warning, QString("Are you sure you want to quit?"), this);
    _dialog->setZValue(6.0);
    connect(_dialog, SIGNAL(yesPressed()), SIGNAL(exitApp()));
}

void LoginView::_removeAccountDialog(void)
{
    _dialog = new DialogView(DialogView::Warning, QString("Are you sure you want <br>to delete this account?"), this);
    _dialog->setZValue(6.0);
    connect(_dialog, SIGNAL(yesPressed()), _settingsDialog, SLOT(_removeUser()));
}

void LoginView::_accountRemoved(void)
{
    usersList->setSelected(-1);
    emit accRemoved();
}

void LoginView::_changeOrientation(void)
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    resize(screenGeometry.size());

    if(screenGeometry.width() > screenGeometry.height()) {
        // Set buttons positions
        _settingsButtonPos = QPointF(_background->pos().x(), _background->boundingRect().height() -
                                     _settingsButton->boundingRect().height());
        _newAccountButtonPos = QPointF((size().width() - _background->pos().x() - _newAccountButton->boundingRect().width())/2 +
                                       _background->pos().x(), _settingsButtonPos.y() - _newAccountButton->boundingRect().height() - 13);

        // Logo image and position
        _logo->setPos(35, 100);

        // Users list position and size
        usersList->setPos(486, 70);
        usersList->setSize(QSizeF(314, 270));
    } else if (screenGeometry.width() < screenGeometry.height()) {
        // Set buttons positions
        _settingsButtonPos = QPointF(0, size().height() - _settingsButton->boundingRect().height());
        _newAccountButtonPos = QPointF((size().width() - _newAccountButton->boundingRect().width())/2,
                                       size().height() - _newAccountButton->boundingRect().height());

        // Logo image and position
        _logo->setPos(23, 60);

        // Users list position and size
        usersList->setPos(0, 300);
        usersList->setSize(QSizeF(480, 410));
    }

    setLoaderPosition(size().width()/2 - 12, size().height() - 96);

    _exitButtonPos = QPointF(size().width() - _exitButton->boundingRect().width(), 0);
    _loginButtonPos = QPointF(size().width() - _loginButton->boundingRect().width(),
                              size().height() - _loginButton->boundingRect().height());

    _cancelButton->setPos(_loginButtonPos);
    _loginButton->setPos(_loginButtonPos);
    _exitButton->setPos(_exitButtonPos);
    _settingsButton->setPos(_settingsButtonPos);
    _newAccountButton->setPos(_newAccountButtonPos);

    _createAnimation();
}

void LoginView::_unlockRotation(void)
{
    _emitLockRotation(false);
}

void LoginView::_lockRotation(void)
{
    _emitLockRotation(true);
}

void LoginView::_emitLockRotation(bool lock)
{
    emit lockRotation(lock);
}
