#include "accountManager.h"

#include<QDebug>
#include <QtCore/QFile>

// Old format
typedef struct Acc {

    bool operator == (const Acc &other);

    QString username;
    QString token;
    QString tokenSecret;
    QString imageUrl;
    bool showAvatar;
    bool showUsername;
    bool showTimestamp;
    bool showReplyTo;
    bool highlightUrls;
    bool highlightMentions;
    bool highlightTags;
    QImage avatar;
}Acc;

AccountManager *AccountManager::_instance = 0;

AccountManager* AccountManager::instance(void)
{
    if(!AccountManager::_instance) {
        AccountManager::_instance = new AccountManager();
    }
    return AccountManager::_instance;
}

AccountManager::AccountManager() :
    _autoLogin(-1),
    _selected(NULL),
    _version(0.0)
{
    qRegisterMetaType<Account>("Account");
    qRegisterMetaTypeStreamOperators<Account>("Account");

    _settings = new QSettings("Zagaia","TweeGo");

    _loadVersion();
    _loadSettings();
}

void AccountManager::_loadVersion(void)
{
    _settings->beginGroup("VERSION");
    _version = _settings->value("version").toFloat();
    _settings->endGroup();

    _checkVersion();

    _settings->beginGroup("VERSION");
    _settings->setValue("version", 0.65);
    _settings->endGroup();
}

void AccountManager::_checkVersion(void)
{
    if(_version <= 0.64) {
        _convertNewSettings();
    }
}

void AccountManager::_convertNewSettings(void)
{
    QList<Acc> oldAccounts;

    // Read old format
    int size = _settings->beginReadArray("ACCOUNTS");
    for(int i = 0; i < size; i++) {
        _settings->setArrayIndex(i);

        Acc account;
        account.username = _settings->value("username").toString();
        account.token = _settings->value("token").toString();
        account.tokenSecret = _settings->value("tokenSecret").toString();
        account.imageUrl = _settings->value("imageUrl").toString();
        account.avatar = _settings->value("avatar").value<QImage>();
        account.showAvatar = _settings->value("showAvatar").toInt();
        account.showUsername = _settings->value("showUsername").toInt();
        account.showTimestamp = _settings->value("showTimestamp").toInt();
        account.showReplyTo = _settings->value("showReplyTo").toInt();
        account.highlightUrls = _settings->value("highlightUrls").toInt();
        account.highlightMentions = _settings->value("highlightMentions").toInt();
        account.highlightTags = _settings->value("highlightTags").toInt();

        oldAccounts.append(account);
    }
    _settings->endArray();

    // Remove file
    QFile configs(QString("%1/.config/Zagaia/TweeGo.conf").arg(getenv("HOME")));
    if(configs.exists()) {
        configs.remove();
    }

    // Clear all settings
    _settings->clear();

    // Remove old accounts
    _removeAllAccounts();

    // Save new format
    size = oldAccounts.size();

    for(int i = 0; i < size; i++) {
        saveNewAccount(oldAccounts.at(i).username, oldAccounts.at(i).token, oldAccounts.at(i).tokenSecret);
        saveImage(oldAccounts.at(i).avatar, oldAccounts.at(i).username);
    }
}

void AccountManager::_loadSettings(void)
{
    int size = _settings->beginReadArray("ACCOUNTS");

    for(int i = 0; i < size; i++) {
        _settings->setArrayIndex(i);

        Account account;
        account = _settings->value("accountInfos").value<Account>();

        QMap<QString, Account>::const_iterator j = _accounts.find(account.getUsername());
        if(j == _accounts.end() || j.key() != account.getUsername()) {
            _accounts[account.getUsername()] = account;
            _orderedAccounts.append(account);
        }
    }
    _settings->endArray();

    _settings->beginGroup("AUTOLOGIN");
    _autoLogin = _settings->value("user").toString();
    _settings->endGroup();

    QMap<QString, Account>::const_iterator i = _accounts.find(_autoLogin);
    if(i == _accounts.end() || i.key() != _autoLogin) {
        _autoLogin = "";
    }
}

void AccountManager::_reloadSettings(void)
{
    _accounts.clear();
    _orderedAccounts.clear();

    _loadSettings();
}

void AccountManager::setSelected(QString user)
{
    _selected = &_accounts[user];
}

Account* AccountManager::getSelected(void)
{
    return _selected;
}

QImage AccountManager::getSelectedAvatar(void)
{
    return _selected->avatar;
}

QString AccountManager::getSelectedImageUrl(void)
{
    return _selected->user.imageUrl;
}

void AccountManager::setSelectedAutoLogin(bool option)
{
    _settings->beginGroup("AUTOLOGIN");

    if(option) {
        _settings->setValue("user",_selected->getUsername());
        _autoLogin = _selected->getUsername();
    } else {
        _settings->setValue("user","");
        _autoLogin = "";
    }

    _settings->endGroup();
}

bool AccountManager::isSelectedAutoLogin(void)
{
    if (_selected->getUsername() == _autoLogin)
        return true;
    return false;
}

int AccountManager::autoLoginPosition(void)
{
    return _orderedAccounts.indexOf(_accounts[_autoLogin]);
}

bool AccountManager::hasAutoLogin(void)
{
    if(_autoLogin == "" || _autoLogin == "0")
        return false;
    return true;
}

QList<Account>* AccountManager::getUsers(void)
{
    return &_orderedAccounts;
}

void AccountManager::saveImage(QImage avatar)
{
    saveImage(avatar, _selected->getUsername());
}

bool AccountManager::userExists(QString user)
{
    QMap<QString, Account>::const_iterator i = _accounts.find(user);
    if(i != _accounts.end() && i.key() == user) {
        return true;
    }
    return false;
}

void AccountManager::_removeAllAccounts(void)
{
    int size = _settings->beginReadArray("ACCOUNTS");
    _settings->endArray();

    for(int i = 0; i < size; ++i) {
        _settings->beginWriteArray("ACCOUNTS");
        _settings->setArrayIndex(i);
        _settings->remove("");
    }

    _selected = NULL;
    _autoLogin = "";
}

void AccountManager::removeSelected(void)
{
    int pos = _orderedAccounts.indexOf(_accounts[_selected->getUsername()]);

    int size = _settings->beginReadArray("ACCOUNTS");
    _settings->endArray();

    _settings->beginWriteArray("ACCOUNTS");
    _settings->setArrayIndex(pos);
    _settings->remove("");

    Account account;
    int j;

    for(int i = pos; i < size-1; i++) {
        // Save next
        j = i + 1;

        // Read next
        _settings->setArrayIndex(j);
        account = _settings->value("accountInfos").value<Account>();

        // Write next on the current position
        _settings->setArrayIndex(i);
        _settings->setValue("accountInfos", qVariantFromValue(account));
    }

    // Remove Last Account
    _settings->setArrayIndex(size-1);
    _settings->remove("");
    _settings->endArray();

    // Decrease accounts size
    _settings->beginWriteArray("ACCOUNTS", size - 1);
    _settings->endArray();

    // If autologin is enable to the removed account, reset autologin
    if(_selected->getUsername() == _autoLogin) {
        _settings->beginGroup("AUTOLOGIN");
        _settings->setValue("autologin", "");
        _settings->setValue("user", -1);
        _settings->endGroup();

        _autoLogin = "";
    }

    _selected = NULL;
    _reloadSettings();
}

void AccountManager::saveNewAccount(QString username, QString token, QString tokenSecret)
{
    int size = _settings->beginReadArray("ACCOUNTS");
    _settings->endArray();

    Account newAccount;
    newAccount.initialize();
    newAccount.setTokens(token, tokenSecret);
    newAccount.setUsername(username);

    // Write the list to file
    _settings->beginWriteArray("ACCOUNTS");
    _settings->setArrayIndex(size);
    _settings->setValue("accountInfos", qVariantFromValue(newAccount));
    _settings->endArray();

    _accounts[username] = newAccount;
    _orderedAccounts.append(newAccount);
}

void AccountManager::setUser(UserInfo *user)
{
    int pos = _orderedAccounts.indexOf(_accounts[_selected->getUsername()]);

    _settings->beginReadArray("ACCOUNTS");
    _settings->setArrayIndex(pos);
    Account account = _settings->value("accountInfos").value<Account>();
    account.user = UserInfo(user);

    _settings->setValue("accountInfos", qVariantFromValue(account));
    _settings->endArray();

    _reloadSettings();

    setSelected(user->screenName);
}

void AccountManager::saveImage(QImage avatar, QString username)
{
    int pos = _orderedAccounts.indexOf(_accounts[username]);

    _settings->beginReadArray("ACCOUNTS");
    _settings->setArrayIndex(pos);
    Account account = _settings->value("accountInfos").value<Account>();
    account.avatar = avatar;

    _settings->setValue("accountInfos", qVariantFromValue(account));
    _settings->endArray();

    _reloadSettings();
}
