/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "backend.h"

/*! \namespace Connection Contains types of connections.
  */
namespace Connection
{
    enum ConnectionType { VerifyCredentials, Update, GetFriendsTimeline, RemoveStatus,
                          AddFavorite, RemoveFavorite, addFriendship, RemoveFriendship,
                          SendDirectMessage, GetMentions };
}

Backend *Backend::_instance = 0;

/*!
 * \brief Gets the current instance of the backend object.
 * \return If there is a instance returns it, else creates a new instance then returns it.
 *
 *  Singleton Design Pattern.
 */
Backend *Backend::getInstance(void)
{
    if (!Backend::_instance) {
        Backend::_instance = new Backend();
    }
    return Backend::_instance;
}

/*!
 * \brief TwitterButton constructor.
 */
Backend::Backend()
    : _twitterapi( new TwitterAPI ),
      _mentionsPage(1),
      _friendsPage(1),
      _userPage(1)
{
    _twitterapi->setServiceUrl("https://twitter.com");
    _twitterapi->setUsingOAuth(true);

    connect(_twitterapi, SIGNAL(credentials(EntryList)), SLOT(_credentials(EntryList)));
    connect(_twitterapi, SIGNAL(newEntries(EntryList)), SLOT(_parseEntries(EntryList)));
    connect(_twitterapi, SIGNAL(deleteEntry(quint64)), SLOT(_deleteEntry(quint64)));
    connect(_twitterapi, SIGNAL(errorMessage(QNetworkReply::NetworkError)), SLOT(_error(QNetworkReply::NetworkError)));
}

void Backend::_deleteEntry(quint64 id)
{
    _userInformations.statusesCount -= 1;
    emit deletedTweet(id);
}

void Backend::_credentials(EntryList entries)
{
    _userInformations = entries.at(0).userInfo;
    emit signedOn();
}

void Backend::_parseEntries(EntryList entries)
{
    int lastConn = _conns.at(0);
    _conns.removeAt(0);

    switch(lastConn)
    {
        case Connection::GetFriendsTimeline:
            _addNewHomeTweets(entries);
            emit gotFriendsTimeline();
            break;

        case Connection::Update:
            _updateTweet = entries.at(0);
            emit updatePosted();
            break;

        case Connection::GetMentions:
            _addNewMentionsTweets(entries);
            emit gotMentions();
            break;
    }
}

void Backend::_addNewHomeTweets(EntryList entries)
{
    foreach( Entry entry, entries) {
        if(!_homeTweetsMap.contains(entry.id)) {
            _homeTweetsMap[entry.id] = entry;
            _homeTweets << entry;
        }
    }
}

void Backend::_addNewMentionsTweets(EntryList entries)
{
    foreach( Entry entry, entries) {
        if(!_mentionsTweetsMap.contains(entry.id)) {
            _mentionsTweetsMap[entry.id] = entry;
            _mentionsTweets << entry;
        }
    }
}

/*!
 * \brief Try to post a new update on the twitter.
 * \param status Status message.
 */
void Backend::update(QString status, QString lat, QString lon, quint64 inReplyToStatusId)
{
    Q_UNUSED(lat);
    Q_UNUSED(lon);
    Q_UNUSED(inReplyToStatusId);
    _conns << Connection::Update;
    _twitterapi->postUpdate(status/*,inReplyToStatusId*/);
}

/*!
 * \brief Try to get a timeline from the twitter.
 * \param type The type of timeline that is requested.
 *
 * The app can download the user, friends and public timelines.
 */
void Backend::getTimeline(Timeline::TimelineType type)
{
    switch(type)
    {
        case Timeline::Users:
            // TODO: implement twitterapi->userTimeline().
            break;

        case Timeline::Friends:
            _homeTweets.clear();

            _conns << Connection::GetFriendsTimeline;
            _twitterapi->friendsTimeline(_friendsPage);
            _friendsPage++;
            break;

        case Timeline::Public:
            _twitterapi->publicTimeline();
            break;

        case Timeline::Mentions:
            _mentionsTweets.clear();

            _conns << Connection::GetMentions;
            _twitterapi->mentions(_mentionsPage);
            _mentionsPage++;
            break;
    }
}

/*!
 * \brief Verify the user informations.
 *
 * This function is used when the app needs to get the user informations
 */
void Backend::verifyCredentials(void)
{
    _twitterapi->verifyCredentials();
}

/*!
 * \brief This function returns a list with the user informations.
 * \return A list with the user informations.
 */
UserInfo Backend::userInfo(void)
{
    return _userInformations;
}

EntryList Backend::getHomeTweets(void)
{
    return _homeTweets;
}

EntryList Backend::getMentionsTweets(void)
{
    return _mentionsTweets;
}

Entry Backend::getUpdateTweet(void)
{
    return _updateTweet;
}

void Backend::resetTimelinePage(Timeline::TimelineType type)
{
    switch(type) {
        case Timeline::Friends:
            _homeTweetsMap.clear();
            _homeTweets.clear();

            _friendsPage = 1;
            break;

        case Timeline::Mentions:
            _mentionsTweetsMap.clear();
            _mentionsTweets.clear();

            _mentionsPage = 1;
            break;

        case Timeline::Users:
            _userPage = 1;
            break;
    }
}

/*!
 * \brief Adds the status(tweet) as favorite.
 * \param id Tweet's id.
 */
void Backend::addFavorite(quint64 id)
{
    _conns << Connection::AddFavorite;
    _twitterapi->createFavorite(id);
}

/*!
 * \brief Remove the status(tweet) as favorite.
 * \param id Tweet's id.
 */
void Backend::removeFavorite(quint64 id)
{
    _conns << Connection::RemoveFavorite;
    _twitterapi->destroyFavorite(id);
}

/*!
 * \brief Removes the status(tweet).
 * \param id Status id.
 */
void Backend::removeStatus(quint64 id)
{
    _conns << Connection::RemoveStatus;
    _twitterapi->deleteUpdate(id);
}

/*!
 * \brief Removes the user from the following list.
 * \param id User id.
 */
void Backend::removeFriendship(quint64 id)
{
    _conns << Connection::RemoveFriendship;
    _twitterapi->unfollow(id);
}

/*!
 * \brief Adds the user in the following list.
 * \param id User id.
 */
void Backend::addFriendship(quint64 id)
{
    _conns << Connection::addFriendship;
    _twitterapi->follow(id);
}

/*!
 * \brief Send the user a message.
 * \param user Username.
 * \param text The message.
 * \return true if successful or false otherwise.
 */
void Backend::sendDirectMessage(QString user, QString msg)
{
    _conns << Connection::SendDirectMessage;
    _twitterapi->postDM(user,msg);
}

/*!
 * \brief Change user's profile image.
 * \param image The image.
 * \return true if successful or false otherwise.
 */
bool Backend::updateProfileImage(QString image)
{
    // TODO: implement on twitterapi :(
    Q_UNUSED(image);
    return false;
}

void Backend::setConsumerKey(const QByteArray &consumerKey)
{
    _twitterapi->setConsumerKey(consumerKey);
}

void Backend::setConsumerSecret(const QByteArray &consumerSecret)
{
    _twitterapi->setConsumerSecret(consumerSecret);
}

void Backend::setLogin(const QString &login)
{
    _twitterapi->setLogin(login);
}

void Backend::setPassword(const QString &password)
{
    _twitterapi->setPassword(password);
}

void Backend::abort(void)
{
    _twitterapi->abort();
}

void Backend::_error(QNetworkReply::NetworkError error)
{
    qDebug() << error;
}
