/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twitterimage.h"

/*! \brief TwitPic constructor.
 */
TwitterImage::TwitterImage(QGraphicsWidget *parent) :
        TwitterApplicationItem(parent)
{
    m_twitPicManager = new QNetworkAccessManager( this );
    m_tweetPhotoManager = new QNetworkAccessManager( this );
    connect(m_twitPicManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(handleTwitPicRequest(QNetworkReply*)));
    connect(m_tweetPhotoManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(handleTweetPhotoRequest(QNetworkReply*)));

    setGeometry(0, 0, 800, 480);
    setMinimumSize(800, 480);
    setMaximumSize(800, 480);
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void TwitterImage::downloadTwitPic(QString Url)
{
    m_isImage = false;
    m_paintImage = false;

    QNetworkRequest request;
    request.setUrl(Url);
    m_twitPicManager->get(request);
}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void TwitterImage::handleTwitPicRequest(QNetworkReply *reply)
{
    if(m_isImage) {
        if(!m_image.load(reply,0))
            m_image = QImage();
        m_paintImage = true;
        update();
    }
    else
        parseHtml(QString(reply->readAll()));

    reply->deleteLater();
}

void TwitterImage::parseHtml(QString html)
{
    QString a;
    a = html.section("img src=\"", 2, 2);
    QStringList stringList = a.split("\"");

    m_isTweetPhoto = false;
    downloadImage(stringList[0]);
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void TwitterImage::downloadTweetPhoto(QString Url)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_isImage = false;
    m_paintImage = false;

    QString apiUrl = "http://api.tweetphoto.com/api/tpapi.svc/photos/";

    QString tweetPhotoId;
    tweetPhotoId = Url.section("/", 3, 3);

    QNetworkRequest request;
    request.setUrl(apiUrl + tweetPhotoId);
    m_tweetPhotoManager->get(request);
}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void TwitterImage::handleTweetPhotoRequest(QNetworkReply *reply)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(m_isImage) {
        if(!m_image.load(reply,0))
            m_image = QImage();
        m_paintImage = true;
        update();
    }
    else
        parseTweetPhotoXml(QString(reply->readAll()));

    reply->deleteLater();
}

void TwitterImage::parseTweetPhotoXml(QString xml)
{
    QDomDocument doc;
    QDomNodeList nodeList;

    doc.setContent(xml);
    nodeList = doc.elementsByTagName("Photo");

    for(int i=0; i<nodeList.count(); i++)
    {
            QDomElement node;

            node = nodeList.at(i).toElement();

            QString bigImageUrl = node.namedItem("BigImageUrl").toElement().text();
            QString thumbnailurl = node.namedItem("ThumbnailUrl").toElement().text();
            m_message = node.namedItem("Message").toElement().text();

            m_imageUrlList.append(bigImageUrl);
            m_imageUrlList.append(thumbnailurl);
            m_imageUrlList.append(m_message);
    }
    m_isTweetPhoto = true;
    downloadImage(m_imageUrlList[0]);
}

void TwitterImage::downloadImage(QString imageUrl)
{
    m_isImage = true;

    QNetworkRequest request;

    QByteArray array = imageUrl.toAscii();
    QUrl newUrl;
    newUrl.setEncodedUrl(array);

    request.setUrl(newUrl);
    qDebug() << newUrl;

    if(!m_isTweetPhoto)
        m_twitPicManager->get(request);
    else
        m_tweetPhotoManager->get(request);
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterImage::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    Q_UNUSED(option);

    painter->setBrush(QBrush(QColor(0, 0, 0, 150)));
    painter->setPen(Qt::NoPen);
    painter->drawRoundedRect(rect(), 5, 5);

    /* Draw Image */
    if (m_paintImage) {
        m_imageScaled = m_image.scaled(760, 440, Qt::KeepAspectRatio);
        painter->drawImage((800 - m_imageScaled.width()) / 2, 20, m_imageScaled);
    }

    /* Draw Message */
    if (!m_message.isEmpty()) {

        QFont font;
        font.setFamily("Nokia Sans");
        font.setStyle(QFont::StyleNormal);
        font.setStyleStrategy(QFont::PreferAntialias);
        font.setPixelSize(22);
        font.setWeight(QFont::Normal);
        painter->setFont(font);

        QRectF messageRect(QRectF(0 , m_imageScaled.rect().bottom() - 20,
                                  800, 40 + 20));

        painter->drawRoundedRect(messageRect, 5, 5);
        painter->setPen(QColor(Qt::white));
        painter->drawText(messageRect, Qt::TextWordWrap | Qt::AlignCenter, m_message);
    }
}

/*!
 * \brief Overridden.
 */
void TwitterImage::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

/*!
 * \brief Overridden.
 */
void TwitterImage::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    int exitSpace = (800 - m_imageScaled.width()) / 2;
    if(event->pos().x() < exitSpace || event->pos().x() > (m_imageScaled.width() + exitSpace)) {
        emit animExit();
        return;
    }

    if(m_paintImage) {
        QMessageBox msgBox;
        msgBox.setText("Save image?");
        msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
        msgBox.setDefaultButton(QMessageBox::Save);
        int ret = msgBox.exec();

        if(ret == QMessageBox::Save) {
            QFileDialog dialog;

            #ifdef Q_WS_MAEMO_5
                QString fileName = dialog.getSaveFileName(0, "Save File", "/home/user/.images/untitled.jpg","Images (*.png *.gif *.jpg)");
            #endif

            if(fileName != NULL)
                    m_image.save(fileName);
            else
                return;
        }
        else
            if(ret == QMessageBox::Cancel)
                return;
    }
    // TODO: animation inside this object
//    else
//        emit animExit();
}
