/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include <QDesktopServices>
#include "twitterbutton.h"

/*! \brief TwitterButton constructor.
 *  \param buttonType
 *  \param buttonLabel String to be written on the button.
 */
TwitterButton::TwitterButton(ButtonType buttonType, QString buttonLabel, QGraphicsWidget *parent)
        : TwitterApplicationItem(parent),
        _buttonLabel(buttonLabel)
{
    _pen.setColor(QColor(15, 149, 218));
    m_isPressed = false;

    setTypeButton(buttonType);

    if(buttonType == TwitterButton::Expose) {
        connect(this, SIGNAL(released()),SLOT(_emitExposeSignal()));
    } else if(buttonType == TwitterButton::Url) {
        _url = _buttonLabel;
        connect(this, SIGNAL(released()),SLOT(_openUrl()));
    }
}

/*!
 * \brief Overridden.
 */
void TwitterButton::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if(contains(event->pos())) {
        m_isPressed = true;
        emit pressed();
    }
    _pen.setColor(QColor(255,255,255));
    update();
}

/*!
 * \brief Overridden.
 */
void TwitterButton::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if(!contains(event->pos()))
        return;

    _pen.setColor(QColor(15, 149, 218));

    m_isPressed = false;
    update();

    emit released();
}

/*!
 * \brief Overridden.
 */
void TwitterButton::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    if(contains(event->pos())) {
        m_isPressed = true;
        _pen.setColor(QColor(255,255,255));
    } else {
        m_isPressed = false;
        _pen.setColor(QColor(15, 149, 218));
    }

    update();
}

/*!
 * \brief Overridden.
 */
QRectF TwitterButton::boundingRect() const
{
    return _buttonImage.rect();
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterButton::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    bool down = option->state & QStyle::State_Sunken;

    if (down && m_isPressed) {
        painter->translate(2, 2);
        painter->setOpacity(0.7);

        switch(_buttonType) {
            case Expose:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/expose-pressed.png");
                break;
            case Logout:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/logout-pressed.png");
                break;
            case Generic:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/generic-pressed.png");
                break;
            case Retweet:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/rt-pressed.png");
                break;
            case Reply:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/reply-pressed.png");
                break;
            case GoUrl:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/go-to-url-pressed.png");
                break;
            case Delete:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/delete-pressed.png");
                break;
            case Back:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/back-pressed.png");
                break;
            case Favorite:
            case NoFavorite:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/favorite-pressed.png");
                break;
            case More:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/more-pressed.png");
                break;
            case Close:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/close-pressed.png");
                break;
            case LoginSettings:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/settings-pressed.png");
                break;
            case GenericDownLeft:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/down-left-pressed.png");
                break;
            case GenericDownRight:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/down-right-pressed.png");
                break;
            case Url:
            case NewAccount:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/new-account-pressed.png");
                break;
            case Refresh:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/refresh-pressed.png");
                break;
        }
    } else
        _buttonSelectedImage = 0;

    painter->drawPixmap(0, 0, _buttonImage);
    painter->setOpacity(1.0);
    painter->drawPixmap(0, 0, _buttonSelectedImage);

    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setWeight(QFont::Bold);

    font.setPixelSize(16);

    painter->setFont(font);
    painter->setPen(_pen);
    painter->drawText(boundingRect(), Qt::TextWordWrap | Qt::AlignCenter, _buttonLabel);

}

/*!
 * \brief This function enables or disables a button.
 * \param option true to enable and false to disable button.
 */
void TwitterButton::enable(bool option)
{
    if(!option) {
        _pen.setColor(QColor(117, 117, 117));
        if (_buttonType == LoginSettings)
            _buttonImage = QPixmap(":images/resources/buttons/settings-disabled.png");
    } else {
        _pen.setColor(QColor(15, 149, 218));
        if (_buttonType == LoginSettings)
            _buttonImage = QPixmap(":images/resources/buttons/settings.png");
    }
    this->setEnabled(option);
}

/*!
 * \brief This function sets the button type, changing the image.
 * \param buttonType The type of button.
 */
void TwitterButton::setTypeButton(int buttonType)
{
    switch(buttonType){
        case Expose:
            _buttonImage = QPixmap(":images/resources/buttons/expose.png");
            break;
        case Logout:
            _buttonImage = QPixmap(":images/resources/buttons/logout.png");
            break;
        case Generic:
            _buttonImage = QPixmap(":images/resources/buttons/generic.png");
            break;
        case Retweet:
            _buttonImage = QPixmap(":images/resources/buttons/rt.png");
            break;
        case Reply:
            _buttonImage = QPixmap(":images/resources/buttons/reply.png");
            break;
        case Favorite:
            _buttonImage = QPixmap(":images/resources/buttons/favorite.png");
            break;
        case NoFavorite:
            _buttonImage = QPixmap(":images/resources/buttons/no-favorite.png");
            break;
        case GoUrl:
            _buttonImage = QPixmap(":images/resources/buttons/go-to-url.png");
            break;
        case Delete:
            _buttonImage = QPixmap(":images/resources/buttons/delete.png");
            break;
        case Back:
            _buttonImage = QPixmap(":images/resources/buttons/back.png");
            break;
        case More:
            _buttonImage = QPixmap(":images/resources/buttons/more.png");
            break;
        case Close:
            _buttonImage = QPixmap(":images/resources/buttons/close.png");
            break;
        case LoginSettings:
            _buttonImage = QPixmap(":images/resources/buttons/settings.png");
            break;
        case GenericDownLeft:
            _buttonImage = QPixmap(":images/resources/buttons/down-left.png");
            break;
        case GenericDownRight:
            _buttonImage = QPixmap(":images/resources/buttons/down-right.png");
            break;
        case Url:
        case NewAccount:
            _buttonImage = QPixmap(":images/resources/buttons/new-account.png");
            break;
        case Refresh:
            _buttonImage = QPixmap(":images/resources/buttons/refresh.png");
            break;
    }
    this->_buttonType = buttonType;
    update();
    setGeometry(_buttonImage.rect());
    setMinimumSize(_buttonImage.size());
    setMaximumSize(_buttonImage.size());
}

/*!
 * \brief This function changes a button's label.
 * \param label The button's label.
 */
void TwitterButton::setLabel(QString label)
{
    this->_buttonLabel = label;
}

/*!
 * \brief Emits a DBus signal to Hildon Desktop who manages the app window with.
 * the task swticher when the expose button is pressed.
 */
void TwitterButton::_emitExposeSignal()
{
    QDBusConnection bus = QDBusConnection::sessionBus();
    QDBusMessage message = QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view");
    bus.send(message);
}

void TwitterButton::_openUrl()
{
    QDesktopServices::openUrl(_url);
}
