/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loginctrl.h"

/*! 
 * \brief LoginCtrl constructor.
 */
LoginCtrl::LoginCtrl(bool autologin, QGraphicsScene *scene)
    : _autologin(autologin)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _scene = scene;
    _scene->setBackgroundBrush(QPixmap(":images/resources/Background.png"));

    _loginView = new LoginView();

    showView();

    connect(_loginView, SIGNAL(done()), SLOT(deleteLater()));
    connect(_loginView, SIGNAL(exitLogin()), SLOT(_exitButtonPressed()));
    connect(_loginView, SIGNAL(loginAccount()), SLOT(_loginTwitter()));
    connect(_loginView, SIGNAL(abort()), SLOT(_abort()));
    connect(_loginView, SIGNAL(newAccount()), SLOT(_populateUserList()));
    connect(Backend::getInstance(),SIGNAL(signedOn()),_loginView,SLOT(connected()));
//    connect(Backend::getInstance(), SIGNAL(connectionError(QString)), this, SLOT(_connectionError(QString)));

    _populateUserList();
    _loadSettings();
}

/*! 
 * \brief LoginCtrl destructor.
 */
LoginCtrl::~LoginCtrl(void)
{
    if (_loginView) {
        delete _loginView;
    }
    emit done();
}

/*! 
 * \brief Show a dialog window with a error message
 * \param message the message in the dialog
 */
void LoginCtrl::_connectionError(QString message)
{
    _loginView->_cancelButton->setOpacity(0);
    _dialog = new DialogView(DialogView::Error,message);
    _scene->addItem(_dialog);
    _loginView->connectionError();
}

/*! 
 * \brief Creates the view and adds the scene
 */
void LoginCtrl::showView(void)
{
    _loginView->createUi();
    _scene->addItem(_loginView);
}

/*! 
 * \brief Try to login on the twitter.
 */
void LoginCtrl::_loginTwitter(void)
{
    QSettings settings("Zagaia","TweeGo");

    // set ConsumerKey e ConsumerSecret
    Backend::getInstance()->setConsumerKey(OauthWizard::ConsumerKey);
    Backend::getInstance()->setConsumerSecret(OauthWizard::ConsumerSecret);

    // Get Selected user
    int selected = _loginView->usersList->getSelected();

    // Read the informations from the file
    settings.beginReadArray("ACCOUNTS");
    settings.setArrayIndex(selected);

    // And set them to connect
    Backend::getInstance()->setLogin(settings.value("username").toString());
    Backend::getInstance()->setPassword(settings.value("token").toString() + "&"
                                        + settings.value("tokenSecret").toString());

    // Finally, try to connect
    Backend::getInstance()->verifyCredentials();
    settings.endArray();
}

/*! 
 * \brief This slot is called when the exit button is pressed
 * quits the application
 */
void LoginCtrl::_exitButtonPressed(void)
{
    _dialog = new DialogView(DialogView::Warning, QString("Are you sure you want to quit?"));
    connect(_dialog, SIGNAL(yesPressed()), this, SLOT(_exit()));
    _scene->addItem(_dialog);
}

/*!
 * \brief Exits the application.
 */
void LoginCtrl::_exit(void)
{
    QApplication::quit();
}

/*!
 * \brief Loads username from a file if it exist.
 */
void LoginCtrl::_loadSettings(void)
{
    // TODO: Remove old tweego's files, remove later
    QFile userInfo(QString("%1/.info.xml").arg(getenv("HOME")));
    if(userInfo.exists())
        userInfo.remove();

    if(!_autologin)
        return;

    QSettings settings("Zagaia","TweeGo");
    settings.beginGroup("AUTOLOGIN");
    bool autologin = settings.value("autologin").toBool();

    if(autologin) {
        int sel = settings.value("user").toInt();
        _loginView->usersList->setSelected(sel);
        _loginView->emitLoginAccount();
    }
    settings.endGroup();
}

/*!
 * \brief Aborts any connections in progress.
 */
void LoginCtrl::_abort()
{
    Backend::getInstance()->abort();
}

void LoginCtrl::_populateUserList(void)
{  
    QSettings settings("Zagaia","TweeGo");

    _loginView->usersList->clearList();

    // Populate the list with any previous account
    int size = settings.beginReadArray("ACCOUNTS");

    QImage avatar;
    QString name;

    if (!size)
    {
        User *newUser = new User("No account. Register a new <br> account to proceed.", avatar, 0, 0);
        newUser->blockSignals(true);
        _loginView->usersList->addToList(newUser,0);
    }
    else
        for(int i=0; i < size; i++)
        {
            settings.setArrayIndex(i);

            name = settings.value("username").toString();
            avatar = QImage(settings.value("avatar").value<QImage>());

            User *newUser = new User(name, avatar, i /*pos*/);
            _loginView->usersList->addToList(newUser,i);
        }
    settings.endArray();
}
