/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loginview.h"

/*!
 * \brief LoginView constructor.
 */
LoginView::LoginView(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _exitAnim = 0;
}

/*!
 * \brief Creates the LoginView objects and animations.
 */
void LoginView::createUi(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _createLoginBackground();
    _createButtons();
    _createLogo();
    _createSettings();
    _createOauthWizard();
    _createUsersList();

    /* Animations */
    _entryAnim = _createEntryAnimation();
    _entryAnim->start();
    _exitAnim = _createExitAnimation();

    connect(qobject_cast<QObject *>(_exitAnim), SIGNAL(finished()), this,
            SIGNAL(done()));
    connect(_exitButton, SIGNAL(released()), this, SIGNAL(exitLogin()));
    connect(_loginButton, SIGNAL(released()), this, SLOT(emitLoginAccount()));
    connect(_settingsButton, SIGNAL(released()), this, SLOT(_animSettings()));
    connect(_cancelButton, SIGNAL(released()), this, SLOT(_endLogin()));
    connect(_newAccountButton, SIGNAL(released()), _oauthWizard, SLOT(startAnimation()));
}

/*!
 * \brief This method creates the entry animation for all the objects in the screen.
 * \return anim the animation entry animation.
 */
QAbstractAnimation *LoginView::_createEntryAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = new QParallelAnimationGroup(this);

    QEasingCurve easingCurve;
    easingCurve.setType(QEasingCurve::OutBack);
    easingCurve.setOvershoot(2.5);

    /* Buttons */
    _addOpacityAnim(anim, _loginButton, 0);
    _addPosAnim(anim, _loginButton, QPointF(715, 423 + 30), QPointF(715, 423),
                easingCurve, 0);
    _addOpacityAnim(anim, _exitButton, 0);
    _addPosAnim(anim, _exitButton, QPointF(716, 0 - 30), QPointF(716, 0),
                easingCurve, 0);
    _addOpacityAnim(anim, _exposeButton, 0);
    _addPosAnim(anim, _exposeButton, QPointF(0, 0 - 30), QPointF(0, 0),
                easingCurve, 0);
    _addOpacityAnim(anim, _settingsButton, 0);
    _addPosAnim(anim, _settingsButton, QPointF(485, 423 + 30), QPointF(485, 423),
                easingCurve, 0);
    _addOpacityAnim(anim, _newAccountButton, 0);
    _addPosAnim(anim, _newAccountButton, QPointF(509, 353 + 30), QPointF(509, 353),
                easingCurve, 0);

    return anim;
}

/*!
 * \brief This method creates the exit animation for all the objects in the screen.
 * \return anim the exit animation.
 * This method takes the entry animation and sets it backward for the exit animation.
 */
QAbstractAnimation *LoginView::_createExitAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = qobject_cast<QParallelAnimationGroup *>(_createEntryAnimation());

    anim->setDirection(QAbstractAnimation::Backward);
    return anim;
}

/*!
 * \brief Creates login background.
 */
void LoginView::_createLoginBackground(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_background = new QGraphicsRectItem(QRectF(0,0,314,480),this);
    m_background->setPos(800 - m_background->boundingRect().width(),0);
    m_background->setBrush(QBrush(QColor(255,255,255,100)));
    m_background->setPen(Qt::NoPen);
}

/*!
 * \brief Creates all buttons.
 */
void LoginView::_createButtons(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _exitButton = new TwitterButton(TwitterButton::Close, "",this);
    _exitButton->setOpacity(0);

    _loginButton = new TwitterButton(TwitterButton::GenericDownRight, "Login", this);
    _loginButton->setOpacity(0);
    _loginButton->enable(false);

    _exposeButton = new TwitterButton(TwitterButton::Expose, "", this);
    _exposeButton->setOpacity(0);
    _exposeButton->setZValue(4.0);

    _settingsButton = new TwitterButton(TwitterButton::LoginSettings,"",this);
    _settingsButton->setOpacity(0);
    _settingsButton->enable(false);

    _newAccountButton = new TwitterButton(TwitterButton::NewAccount,"New Account",this);
    _newAccountButton->setOpacity(0);

    _cancelButton = new TwitterButton(TwitterButton::Generic, "Cancel", this);
    _cancelButton->setOpacity(0);
    _cancelButton->setPos(QPoint(715, 423));
    _cancelButton->setZValue(4.0);
}

/*!
 * \brief Creates logo.
 */
void LoginView::_createLogo(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _logo = new QGraphicsPixmapItem(this);
    _logo->setPixmap(QPixmap(":images/resources/logo.png"));
    _logo->setPos(35,100);
}

void LoginView::_createSettings(void)
{
    _settingsDialog = new LoginSettings(this);
    _settingsDialog->setPos(800,0);
    _settingsDialog->setZValue(3.0);
}

void LoginView::_createOauthWizard(void)
{
    _oauthWizard = new OauthWizard(this);
    _oauthWizard->setPos(800,0);
    _oauthWizard->setZValue(3.0);

    connect(_oauthWizard, SIGNAL(newAccount()),SIGNAL(newAccount()));
}

void LoginView::_createUsersList(void)
{
    usersList = new KineticList(KineticList::Vertical, QSizeF(314,270),this);
    usersList->setScrollBarPolicy(KineticList::Vertical, Qt::ScrollBarAsNeeded);
    usersList->setPos(486,70);
    usersList->setZValue(2.0);

    connect(usersList, SIGNAL(selected(bool)), _loginButton, SLOT(enable(bool)));
    connect(usersList, SIGNAL(selected(bool)), _settingsButton, SLOT(enable(bool)));
}

/*!
 * \brief Add the opacity Animation
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 */
void LoginView::_addOpacityAnim(QParallelAnimationGroup *anim, TwitterApplicationItem *item, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("opacity");
    animation->setStartValue(0.0);
    animation->setEndValue(1.0);
    animation->setDuration(500);
    animation->setEasingCurve(QEasingCurve::Linear);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Add the position Animation
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void LoginView::_addPosAnim(QParallelAnimationGroup *anim,
                            TwitterApplicationItem *item,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(1000);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Emits login signal if user presses login button.
 */
void LoginView::emitLoginAccount(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _connecting();
    _loginButton->setOpacity(0);
    _cancelButton->setOpacity(1.0);
    emit loginAccount();
}

/*!
 * \brief This slot is called when there is a connection error to delete the loader object.
 */
void LoginView::connectionError()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _deleteLoader();
    _exitButton->enable(true);
    _loginButton->enable(true);
    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0);
}

/*!
 * \brief This slot starts the exit animation when user is successfully connected
 * or emits done signal if the animation could not be created.
 */
void LoginView::connected(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _loginButton->setOpacity(1.0);
    delete _cancelButton;
    _loader->startExitAnimation();
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief Deletes loader animation.
 */
void LoginView::_deleteLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    delete _loader;
}

/*!
 * \brief Calls loader after user presses the login button.
 */
void LoginView::_connecting(void)
{
    _loader = new Loader(this);
    _loader->setZValue(3.0);
    _loader->startEntryAnimation();
    connect(_loader, SIGNAL(finished()), this, SLOT(_deleteLoader()));
}

/*!
 * \brief When user hits cancel, this method deletes loader, emits a signal to
 * abort the login, and puts back login button on the screen.
 */
void LoginView::_endLogin(void)
{
    emit abort();
    _deleteLoader();
    _loginButton->setOpacity(1.0);
    _cancelButton->setOpacity(0);
}

void LoginView::_animSettings(void)
{
    _settingsDialog->setSelected(usersList->getSelected());
    _settingsDialog->startAnimation();
}
