/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "tweet.h"

/*!
 * \brief Class Constructor
 * \param entry object with all the tweet's informations
 */
Tweet::Tweet(Entry entry, QGraphicsWidget *parent)
    : TwitterApplicationItem(parent),
    m_entry(entry),
    m_image(QImage(":images/resources/avatar.png"))
{
    paintImage = false;
    this->setMinimumSize(776,74);

    createTweetMessage();

    if(m_entry.favorited) {
        m_favorited = new QGraphicsPixmapItem(QPixmap::fromImage(QImage(":images/resources/fave_ico.png")),this);
        m_favorited->setPos(768,5);
    }

    if(m_entry.text.contains("RT")) {
        m_rtImage = new QGraphicsPixmapItem(QPixmap::fromImage(QImage(":images/resources/rt_timeline.png")),this);
        m_rtImage->setPos(77,9);
    }

    connect(&m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(handleNetworkRequest(QNetworkReply*)));
    download(m_entry.userInfo.imageUrl);
}

/*!
 * \brief Reimplements mouse press event
 * \param event Used to get the mouse click position
 */
void Tweet::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

/*!
 * \brief Reimplements mouse press event
 * \param event Used to get the mouse click position
 */
void Tweet::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if(this->contains(event->pos()))
        emit this->createTweetInfo(this);
}

/*!
 * \brief Writes The whole tweet message
 * \param entry
 */
void Tweet::createTweetMessage()
{
    QString messageHtml = "<p style=\" font-size:18px; padding-bottom:0px; margin-bottom:0px; font-family:'Nokia Sans';\"> ";
    if(m_entry.text.contains("RT"))
        messageHtml += "<span style=\" color:#0f95da;\">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + m_entry.userInfo.screenName + "</span>";
    else
        messageHtml += "<span style=\" color:#0f95da;\">" + m_entry.userInfo.screenName + "</span>";
    messageHtml += "<span style=\" color:#000000;\"> " + parseText(m_entry.originalText) + "</span></p>";

    QString messageInfo = "<p align=\"right\"; style=\" padding-top:5px; margin-top:5px; font-size:14px; font-family:'Nokia Sans';\"> ";
    messageInfo += "<span style=\" color=#757575;\">" + parseCreatedAt(m_entry.localTime);
    messageInfo += " via " + parseSource(m_entry.source) + "</span></p>";

    message = new QGraphicsTextItem(this);
    message->setPos(74,0);
    message->setTextWidth(702);
    message->setDefaultTextColor(QColor(0,0,0));
    message->setHtml(messageHtml + messageInfo);

    if(message->document()->size().height() < 56) {
        messageHtml.replace("</p>","<br></p>");
        message->setHtml(messageHtml + messageInfo);
    }

    if (m_entry.hasInReplyToStatusId) {
        messageInfo.replace("</span></p>"," in reply to " + m_entry.inReplyToScreenName + "</span></p>");
        message->setHtml(messageHtml + messageInfo);
    }

    if(message->document()->size().height() > 74)
        setMinimumHeight(message->document()->size().height());
    setMaximumHeight(this->minimumHeight());
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void Tweet::download(QString Url)
{
    QNetworkRequest request;
    request.setUrl(Url);
    m_manager.get(request);
}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void Tweet::handleNetworkRequest(QNetworkReply *reply)
{
    if(!m_image.load(reply,0))
        m_image = QImage();

    emit imageDownloaded();

    reply->deleteLater();
    update();
}

/*!
 * \brief Paints the Tweet white rectangle and the user avatar
 * \param event Unused
 */
void Tweet::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    // Transparent Rect
    painter->setPen(Qt::NoPen);

    if(m_entry.isOwn)
        painter->setBrush(QColor(255, 240, 204));
    else
        painter->setBrush(QColor(246, 253, 255));

    painter->drawRoundedRect(0, 0, rect().width(), rect().height(),2,2);
    painter->setRenderHint(QPainter::Antialiasing);

    // Avatar Rect
    painter->drawImage(QRect(13,13,48,48),m_image);
}

/*!
 * \brief This funcion is a parser for writting the user tweet's source app
 * \param source The source list
 * \return A list with only the app name
 */
QString Tweet::parseSource(QString source)
{
    return source.remove(QRegExp("<[^<]*>"));
}

/*!
 * \brief This funcion is a parser for writting the time of user tweet
 * \param source The list with the time of the tweets
 * \return A list with only the time of the tweets
 */
QString Tweet::parseCreatedAt(QDateTime timestamp)
{
    QLocale::setDefault(QLocale::English);

    QString newCreatedAt;
    int seconds = timestamp.secsTo(QDateTime::currentDateTime().toUTC());

    if (seconds < 20) {
        newCreatedAt = "Less than 20 seconds ago";
    } else if(seconds < 30) {
        newCreatedAt = "About half minute ago";
    } else if(seconds < 60) {
        newCreatedAt = "Less than 1 minute ago";
    } else if(seconds < 3600) {
        newCreatedAt = QString::number(seconds/60) + " minutes ago";
    } else if(seconds < 7200) {
        newCreatedAt = "About 1 hour ago";
    } else if(seconds < 86400) {
        newCreatedAt = "About " + QString::number(seconds/3600) + " hours ago";
    } else {
        newCreatedAt = QString::number(timestamp.time().hour()) + ":" +
                       QString::number(timestamp.time().minute()) + " " +
                       QDate::shortMonthName(timestamp.date().month()) + " " +
                       QString::number(timestamp.date().day());

        switch(timestamp.date().day()) {
            case 1:
            case 21:
                newCreatedAt += "st "; break;
            case 2:
            case 22:
                newCreatedAt += "nd "; break;
            case 3:
                newCreatedAt += "rd "; break;
            default:
                newCreatedAt += "th "; break;
        }
    }

    return newCreatedAt;
}

/*!
 * \brief This funcion is a parser for writting the mentions, urls and trends with blue color
 * \param text The tweet message
 * \return The text with the mentions, urls and trends with blue color
 */
QString Tweet::parseText(QString text)
{  
    // Paint #Trends
    QRegExp tag( "#([\\w\\d-]+)( ?)", Qt::CaseInsensitive );
    text.replace( tag, "<font color=#0f95da>#\\1</font>\\2");

    // Paint Links
    QRegExp ahref( "((https?|ftp)://[^ ]+)( ?)", Qt::CaseInsensitive );
    text.replace( ahref, "<font color=#0f95da>\\1</font>\\3" );

    // Paint @Mentions
    text.replace( QRegExp( "(^|[^a-zA-Z0-9])@([\\w\\d_]+(/[\\w\\d-]+)?)" ),
                  QString( "\\1@<font color=#0f95da>\\2</font>"));
    return text;
}

/*!
 * \brief Method to get the avatar's image from tweet
 * \return The avatar image
 */
QImage Tweet::getImage(void)
{
    return this->m_image;
}

/*!
 * \brief Method to get the tweet's position in the list from tweet
 * \return The tweet's position
 */
int Tweet::getPosInList(void)
{
//    return this->m_posInList;
}

/*!
 * \brief Method to get the tweet's coordinates from tweet
 * \return The tweet's coordinates
 */
bool Tweet::getFollowing(void)
{
//    return this->m_following;
}

/*!
 * \brief Method to get the Tweet message in html format
 * \return The message in html format
 */
QTextDocument* Tweet::getTweetDocument()
{
    return this->message->document();
}

/*!
 * \brief Method to set the tweet's follow flag from tweet
 * \param username The username
 * \param foloow The follow flag
 */
void Tweet::setFollowing(QString username, bool follow)
{
//    if(m_username == username)
//        m_following = follow;
}

/*!
 * \brief Method to set the favorite flag in tweet
 * \param favorite The new favorite flag
 */
void Tweet::setFavorite(bool favorite)
{
//    this->m_favorite = favorite;
}

/*!
 * \brief Method to set the position of the tweet on the list
 * \param position Tweet's position
 */
void Tweet::setPosition(int position)
{
//    m_posInList = position;
}

Entry Tweet::getEntry(void)
{
    return m_entry;
}
