/*
 * vim:ts=4:sw=4:et:cindent:cino=(0
 */ 

#include <config.h>
#include <gmodule.h>

#include "tweakr-module.h"


enum
{
    PROP_0,
    PROP_FILENAME
};


static void      tweakr_module_finalize      (GObject      *object);
static void      tweakr_module_get_property  (GObject      *object,
                                                    guint         param_id,
                                                    GValue       *value,
                                                    GParamSpec   *pspec);
static void      tweakr_module_set_property  (GObject      *object,
                                                    guint         param_id,
                                                    const GValue *value,
                                                    GParamSpec   *pspec);
static gboolean  tweakr_module_load_module   (GTypeModule  *gmodule);
static void      tweakr_module_unload_module (GTypeModule  *gmodule);


G_DEFINE_TYPE (TweakrModule, tweakr_module, G_TYPE_TYPE_MODULE);


static void
tweakr_module_class_init (TweakrModuleClass *class)
{
    GObjectClass     *object_class      = G_OBJECT_CLASS (class);
    GTypeModuleClass *type_module_class = G_TYPE_MODULE_CLASS (class);

    object_class->finalize     = tweakr_module_finalize;
    object_class->get_property = tweakr_module_get_property;
    object_class->set_property = tweakr_module_set_property;

    type_module_class->load    = tweakr_module_load_module;
    type_module_class->unload  = tweakr_module_unload_module;

    g_object_class_install_property
        (object_class, PROP_FILENAME,
         g_param_spec_string ("filename",
                              "Filename",
                              "The filaname of the module",
                              NULL,
                              G_PARAM_READWRITE |
                              G_PARAM_CONSTRUCT_ONLY));
}

static void
tweakr_module_init (TweakrModule *module)
{
    module->filename = NULL;
    module->library  = NULL;
    module->load     = NULL;
    module->unload   = NULL;
}

static void
tweakr_module_finalize (GObject *object)
{
    TweakrModule *module = TWEAKR_MODULE (object);

    g_free (module->filename);

    G_OBJECT_CLASS (tweakr_module_parent_class)->finalize (object);
}

static void
tweakr_module_get_property (GObject    *object,
                                  guint       param_id,
                                  GValue     *value,
                                  GParamSpec *pspec)
{
    TweakrModule *module = TWEAKR_MODULE (object);

    switch (param_id)
    {
        case PROP_FILENAME:
            g_value_set_string (value, module->filename);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
tweakr_module_set_property (GObject      *object,
                                  guint         param_id,
                                  const GValue *value,
                                  GParamSpec   *pspec)
{
    TweakrModule *module = TWEAKR_MODULE (object);

    switch (param_id)
    {
        case PROP_FILENAME:
            g_free (module->filename);
            module->filename = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static gboolean
tweakr_module_load_module (GTypeModule *gmodule)
{
    TweakrModule *module = TWEAKR_MODULE (gmodule);

    if (!module->filename)
    {
        g_warning ("Module path not set");
        return FALSE;
    }

    module->library = g_module_open (module->filename, 0);

    if (!module->library)
    {
        g_printerr ("%s\n", g_module_error ());
        return FALSE;
    }

    /* Make sure that the loaded library contains the required methods */
    if (! g_module_symbol (module->library,
                           "tweakr_module_load",
                           (gpointer) &module->load) ||
        ! g_module_symbol (module->library,
                           "tweakr_module_unload",
                           (gpointer) &module->unload))
    {
        g_printerr ("%s\n", g_module_error ());
        g_module_close (module->library);

        return FALSE;
    }

    /* Initialize the loaded module */
    module->load (module);

    return TRUE;
}

static void
tweakr_module_unload_module (GTypeModule *gmodule)
{
    TweakrModule *module = TWEAKR_MODULE (gmodule);

    module->unload (module);

    g_module_close (module->library);
    module->library = NULL;

    module->load   = NULL;
    module->unload = NULL;
}

TweakrModule *
tweakr_module_new (const gchar *filename)
{
    TweakrModule *module;

    g_return_val_if_fail (filename != NULL, NULL);

    module = g_object_new (TWEAKR_TYPE_MODULE,
                           "filename", filename,
                           NULL);

    return module;
}

