#include "playlistmanager.h"
#include <QDir>
#include <QUrl>
#include <QMap>
#include <QtDebug>
#include <QTemporaryFile>
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkReply>

QStringList PlaylistManager::allowedExtensions;


PlaylistManager::PlaylistManager(QWidget* parent)
    : parentWidget (parent), lastMetaRead (-1)
{
    if (!allowedExtensions.size ())
    {
        allowedExtensions << "mp3" << "ogg" << "wav" << "mlp" << "wma" << "flac" << "amr";
        allowedExtensions << "midi" << "mid" << "mpeg" << "qcp" << "ac3" << "aiff" << "act";
        allowedExtensions << "alac" << "alaw" << "au" << "eac3" << "aac" << "ape" << "gsm";
        allowedExtensions << "mod" << "mp2" << "sbc" << "spx" << "tta" << "ra" << "riff";
        allowedExtensions << "m4a" << "mp4" << "m4p" << "pcm" << "amr";
    }
    metaInformationResolver = new Phonon::MediaObject(parent);
    connect(metaInformationResolver, SIGNAL(stateChanged(Phonon::State,Phonon::State)),
        this, SLOT(metaStateChanged(Phonon::State,Phonon::State)));
    downloadmanager = new QNetworkAccessManager(this);
    connect(downloadmanager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(playlistDownloadFinished(QNetworkReply*)));
    qDebug () << "Database drivers: " << QSqlDatabase::drivers();
    cachedb = QSqlDatabase::addDatabase("QSQLITE");
    cachedb.setDatabaseName(getCacheDbName());
    cachedb.open();
    if (cachedb.tables().indexOf("metadata") == -1)
    {
        QSqlQuery q (cachedb);
       
        q.exec("CREATE TABLE IF NOT EXISTS metadata(id INTEGER PRIMARY KEY,filename TEXT,artist TEXT,title TEXT,album TEXT)");
        q.exec("CREATE INDEX fnameind ON metadata(filename)");
    }
}

PlaylistManager::~PlaylistManager()
{
    cachedb.commit();
    cachedb.close();
}

int PlaylistManager::indexOf(const Phonon::MediaSource &s) const
{
    for (int i = 0; i < items.size(); ++i)
    {
        if (items[i].source == s)
            return i;
    }
    return -1;
}

void PlaylistManager::parseAndAddFolder(const QString &dir, bool recursive)
{
    QStringList filters;

    QStringList files = QDir (dir).entryList(filters);

    if (files.isEmpty())
        return;


    int index = items.size();
    foreach (QString string, files)
    {
        if (string == "."  || string == "..")
            continue;
        QString fname = dir + "/" + string;
        QFileInfo fi (fname);
        if (fi.isDir())
        {
            if (recursive)
                parseAndAddFolder(fname, true);
            continue;
        }
        if (fileSupported(fname))
        {
//            qDebug () << "Adding: " << fname;
            items.append(PlaylistItem (PlaylistItem (fname)));
        }
    }
    if (items.size () > index)
    {
        metaInformationResolver->setCurrentSource(items.at(index).source);
        lastMetaRead = index;
    }
    emit playlistChanged (index);
}

void PlaylistManager::addStringList(const QStringList& list, const QStringList& titles, const QVariantList& positions)
{
    int index = items.size();
    foreach (QString string, list)
    {
        if (fileSupported(string) || string.toLower().startsWith("http"))
        {
            if (string.toLower().startsWith("http") && ((string.toLower().endsWith("pls")
                                        || (string.toLower().endsWith("m3u")))))
            {
                downloadmanager->get(QNetworkRequest(QUrl(string)));
            }
            else
            {
                items.append(PlaylistItem (string));
                if (titles.size() >= items.size ())
                    items[items.size () - 1].title = titles[items.size () - 1];
                if (positions.size() >= items.size ())
                    items[items.size () - 1].subPos = positions[items.size () - 1].toInt ();
            }
        }
    }
    if (items.size () > index)
    {
        metaInformationResolver->setCurrentSource(items.at(index).source);
        lastMetaRead = index;
    }
    emit playlistChanged(index);
}

void PlaylistManager::playlistDownloadFinished(QNetworkReply* rep)
{
    QVariant redir = rep->attribute(QNetworkRequest::RedirectionTargetAttribute);
    if (!redir.isNull())
    {
        downloadmanager->get (QNetworkRequest (redir.toUrl()));
        return;
    }
    QTemporaryFile tmp;
    tmp.open ();
    tmp.write (rep->readAll());
    tmp.close ();
    QString filename = tmp.fileName();
    int index = items.size();
    QString ext = rep->url().toString().right(4).toLower();
    if (ext == ".m3u")
        appendPlaylist(filename);
    else if (ext == ".pls")
        appendPlaylistPLS(filename);
    else if (ext == ".cue")
        appendPlaylistCUE(filename);
    if (items.size () > index)
    {
        metaInformationResolver->setCurrentSource(items.at(index).source);
        lastMetaRead = index;
        emit playlistChanged (index);
    }
}


void PlaylistManager::metaStateChanged(Phonon::State newState, Phonon::State /*oldState*/)
{
    // NOTE: This is an ugly hack, since the metaInformationResolver doesn't properly load the assigned source when it's in the error state
    // In order to properly read the next file we have to set it as current source again when the resolver entered the stopped state after the error
    static bool wasError = false;
    if (wasError && newState == Phonon::StoppedState)
    {
        metaInformationResolver->setCurrentSource(items[lastMetaRead].source);
        wasError = false;
        return;
    }
    if (newState == Phonon::ErrorState)
    {
        wasError = true;
    }

    if (newState != Phonon::StoppedState && newState != Phonon::ErrorState)
    {
        return;
    }

    int index = lastMetaRead;

    QMap<QString, QString> metaData = metaInformationResolver->metaData();


    if (index >= 0 && newState != Phonon::ErrorState && index < items.size ())
    {
        items[index].artist = metaData.value("ARTIST");
        if (!items[index].title.size())
            items[index].title = metaData.value("TITLE");
        items[index].album = metaData.value("ALBUM");
        writeCache(items[index].uri, items[index].album, items[index].artist, items[index].title);
/*        items[index].year = metaData.value("DATE");
        items[index].genre = metaData.value("GENRE");
        qDebug () << "Meta " << metaData;
        qDebug () << "Info is: " << items[index].year << " - " << items[index].genre;*/
        if (!metaData.isEmpty())
            items[index].playable = true;
        emit itemUpdated (index);
    }
/*    if (index + 1 < items.size () && index >= 0)
    {
        metaInformationResolver->setCurrentSource(items[index + 1].source);
        lastMetaRead = index + 1;
    }*/
    while (index + 1 < items.size ())
    {
        QStringList l = checkCache(items[index + 1].uri);
        if (l.size() == 3)
        {
            items[index + 1].album = l[0];
            items[index + 1].artist = l[1];
            items[index + 1].title = l[2];
            items[index + 1].playable = true;
            emit itemUpdated(index + 1);
            lastMetaRead = index + 1;
            index++;
            continue;
        }
        if (!items[index + 1].title.size() || !items[index + 1].album.size() || !items[index + 1].artist.size ())
        {
            metaInformationResolver->setCurrentSource(items[index + 1].source);
            lastMetaRead = index + 1;
            break;
        }
        items[index + 1].playable = true;
        index++;
    }
}

bool PlaylistManager::savePlaylist(const QString& filenam)
{
    QString filename = filenam;
    if (filename.isEmpty())
        return false;
    bool writepls = false;
    if (filename.length() < 4 || (filename.right(4).toLower() != ".m3u" && filename.right(4).toLower() != ".pls"))
    {
        filename += ".pls";
        writepls = true;
    }
    else if (filename.right(4).toLower() == ".pls")
        writepls = true;
    QFile f (filename);
    try
    {
        f.open(QFile::WriteOnly);
        if (writepls)
        {
            f.write ("[playlist]\n");
            f.write (QString ("NumberOfEntries=%1\n").arg (items.size ()).toAscii());
        }
        for (int i = 0; i < items.size(); ++i)
        {
            if (writepls)
            {
                f.write (QString ("File%1=%2\n").arg (i + 1).arg (items[i].uri).toAscii());
                f.write (QString ("Title%1=%2 - %3 - %4\n").arg (i + 1).arg (items[i].artist).arg (items[i].title).arg (items[i].album).toAscii());
            }
            else
            {
                f.write (items[i].uri.toAscii());
                f.write ("\n");
            }
        }
        if (writepls)
            f.write ("Version=2\n");
        f.close ();
        return true;
    }
    catch (...)
    {
    }
    return false;
}

void PlaylistManager::loadPlaylist(const QString& filename)
{
    clearPlaylist();
    QString ext = filename.right(4).toLower();
    if (ext == ".m3u")
        appendPlaylist(filename);
    else if (ext == ".pls")
        appendPlaylistPLS(filename);
    else if (ext == ".cue")
        appendPlaylistCUE(filename);
    if (!items.isEmpty())
    {
        metaInformationResolver->setCurrentSource(items.at(0).source);
        lastMetaRead = 0;
    }
    emit playlistChanged (0);
}

void PlaylistManager::addPlaylist(const QString& filename)
{
    int index = items.size();
    QString ext = filename.right(4).toLower();
    if (ext == ".m3u")
        appendPlaylist(filename);
    else if (ext == ".pls")
        appendPlaylistPLS(filename);
    else if (ext == ".cue")
        appendPlaylistCUE(filename);
    if (items.size () > index)
    {
        metaInformationResolver->setCurrentSource(items.at(index).source);
        lastMetaRead = index;
        emit playlistChanged (index);
    }
}

void PlaylistManager::appendPlaylist(const QString& filename)
{
    QString dir = QFileInfo (filename).absoluteDir().path();
    QFile f(filename);
    if (!f.open (QFile::ReadOnly))
        return;
    QString tmp = f.readAll();
    f.close ();
    QStringList lines = tmp.split("\n");
    foreach (QString l, lines)
    {
        if (l.isEmpty() || l.trimmed().startsWith("#"))
        {
            continue;
        }
        l = l.replace("\r", "");
        QString fname = dir + QDir::separator() + l;
        if (QFile::exists(l))
            fname = l;
        if (QFile::exists(fname))
            items.append(PlaylistItem (fname));
    }
}

void PlaylistManager::appendPlaylistPLS(const QString& filename)
{
    QString dir = QFileInfo (filename).absoluteDir().path();
    QFile f(filename);
    if (!f.open (QFile::ReadOnly))
        return;
    QString tmp = f.readAll();
    f.close ();
    QStringList lines = tmp.split("\n");
    QMap<int, int> filemap;

    foreach (QString l, lines)
    {
//        qDebug() << "Playlist line " << l;
        if (l.isEmpty() || l.trimmed().toLower() == "[playlist]" || l.trimmed().toLower() == "version=2"
            || l.trimmed().startsWith(";"))
        {
            continue;
        }
        l = l.replace("\r", "");
        if (l.trimmed().toLower().left(4) == "file")
        {
            QStringList tokens = l.split('=');
            if (tokens.size () < 2)
                continue;
            tokens[0] = tokens[0].mid (4);
            filemap.insert(tokens[0].toInt (), items.size ());
            QString fname = dir + QDir::separator() + tokens[1];
            if (QFile::exists(l))
                fname = tokens[1];
            if (QFile::exists(fname))
                items.append(PlaylistItem (fname));
        }
        else if (l.trimmed().toLower().left(5) == "title")
        {
            QStringList tokens = l.split('=');
            if (tokens.size () < 2)
                continue;
            tokens[0] = tokens[0].mid (5);
            int toupdate = filemap[tokens[0].toInt()];
            QStringList metatok = tokens[1].split (" - ");
            if (metatok.size() > 2 && toupdate >= 0 && toupdate < items.size ())
            {
                items[toupdate].artist = metatok[0];
                items[toupdate].title = metatok[1];
                metatok = metatok.mid (2);
                items[toupdate].album = metatok.join (" - ");
            }
            else
            {
                items[toupdate].title = metatok.join (" - ");
            }
        }
    }
}

void PlaylistManager::appendPlaylistCUE(const QString& filename)
{
    QFile f(filename);
    if (!f.open (QFile::ReadOnly))
        return;
    QString tmp = f.readAll();
    f.close ();
    QStringList lines = tmp.split("\n");
    bool infile = false;
    bool intrack = false;
    int count = 0;
    QString dir = QFileInfo (filename).absoluteDir().path();
    QString curfile;
    QString curtitle;
    QString curindex;
    QString curperformer;
    QString curalbum;
    qDebug () << "Cue file " << filename;
    foreach (QString l, lines)
    {
        l = l.replace("\r", "");
/*        if (!infile && !intrack && (l.trimmed().toLower().left(5) != "file "))
        {
            continue;
        }*/
        if ((l.trimmed().toLower().left(5) == "file ") && 
            ((l.trimmed().toLower().right(5) == " wave") || (l.trimmed().toLower().right(4)) == " mp3"))
        {
            if (intrack && curtitle.size() && curindex.size())
            {
                QStringList tim = curindex.split(":");
                int pos = 0;
                if (tim.size() == 3 && count > 1)
                    pos = tim[0].toInt () * 60000 + tim[1].toInt () * 1000 + tim[2].toInt() * 10;
                QString fname = dir + QDir::separator() + curfile;
                qDebug () << "Add track " << curtitle <<  " index " << curindex << " calculated position " << pos << " in file " << fname;
                if (QFile::exists(fname))
                {
                    items.append(PlaylistItem (fname));
                    items[items.size () - 1].subPos = pos;
                    items[items.size () - 1].title = curtitle;
                    items[items.size () - 1].artist = curperformer;
                    items[items.size () - 1].album = curalbum;
                }
                else
                    qDebug () << "doesn't exist...";
            }
            infile = true;
            curfile = l.section("\"", 1, 1);
            qDebug () << "In file " << curfile;
            count = 0;
            curtitle = "";
            curindex = "";
            intrack = false;
        }
        if (infile && (l.trimmed().toLower().left(6) == "track "))
        {
            if (intrack && curtitle.size() && curindex.size())
            {
                QStringList tim = curindex.split(":");
                int pos = 0;
                if (tim.size() == 3 && count > 1)
                    pos = tim[0].toInt () * 60000 + tim[1].toInt () * 1000 + tim[2].toInt() * 10;
                QString fname = dir + QDir::separator() + curfile;
                qDebug () << "Add track " << curtitle <<  " index " << curindex << " calculated position " << pos << " in file " << fname;
                if (QFile::exists(fname))
                {
                    items.append(PlaylistItem (fname));
                    items[items.size () - 1].subPos = pos;
                    items[items.size () - 1].title = curtitle;
                    items[items.size () - 1].artist = curperformer;
                    items[items.size () - 1].album = curalbum;
                }
            }
            intrack = true;
            count++;
        }
        if (infile && intrack && (l.trimmed().toLower().left(6) == "title "))
        {
            curtitle = l.section ("\"", 1, 1);
        }
        if (!infile && !intrack && (l.trimmed().toLower().left(6) == "title "))
        {
            curalbum = l.section ("\"", 1, 1);
        }
        if (infile && intrack && (l.trimmed().toLower().left(10) == "performer "))
        {
            curperformer = l.section ("\"", 1, 1);
        }
        if (infile && intrack && (l.trimmed().toLower().left(6) == "index "))
        {
            QStringList tmp = l.trimmed().split(" ");
            if (tmp.size() > 2)
                curindex = tmp[2];
        }
        
    }
    if (intrack && curtitle.size() && curindex.size())
    {
        QStringList tim = curindex.split(":");
        int pos = 0;
        if (tim.size() == 3 && count > 1)
            pos = tim[0].toInt () * 60000 + tim[1].toInt () * 1000 + tim[2].toInt() * 10;
        QString fname = dir + QDir::separator() + curfile;
        qDebug () << "Add track " << curtitle <<  " index " << curindex << " calculated position " << pos << " in file " << fname;
        if (QFile::exists(fname))
        {
            items.append(PlaylistItem (fname));
            items[items.size () - 1].subPos = pos;
            items[items.size () - 1].title = curtitle;
            items[items.size () - 1].artist = curperformer;
            items[items.size () - 1].album = curalbum;
        }
    }
}



void PlaylistManager::clearPlaylist()
{
    items.clear();
    emit playlistChanged(0);
}

QStringList PlaylistManager::playlistStrings() const
{
    QStringList ret;
    for (int i = 0; i < items.size (); ++i)
        ret << items[i].uri;
    return ret;
}

QStringList PlaylistManager::playlistTitles() const
{
    QStringList ret;
    for (int i = 0; i < items.size (); ++i)
        ret << items[i].title;
    return ret;
}

QVariantList PlaylistManager::playlistPositions() const
{
    QVariantList ret;
    for (int i = 0; i < items.size (); ++i)
        ret << items[i].subPos;
    return ret;
}

void PlaylistManager::removeItem(int i)
{
    items.removeAt (i);
    emit itemRemoved(i);
}


bool PlaylistManager::fileSupported (const QString& fname) const
{
    if (fname.lastIndexOf('.') < 0)
        return false;
    QString ext = fname.right(fname.size() - fname.lastIndexOf('.') - 1).toLower();
    foreach (QString e, allowedExtensions)
    {
        if (ext == e)
            return true;
    }

    return false;
}

bool PlaylistManager::moveItemUp (int i)
{
    if (i)
    {
        PlaylistItem tmp = items[i - 1];
        items[i - 1] = items[i];
        items[i] = tmp;
        return true;
    }
    return false;
}

bool PlaylistManager::moveItemDown (int i)
{
    if (i < items.size () - 1)
    {
        PlaylistItem tmp = items[i + 1];
        items[i + 1] = items[i];
        items[i] = tmp;
        return true;
    }
    return false;
}

QString PlaylistManager::getCacheDbName () const
{
    QString ret = QDir::homePath() + QDir::separator() + ".tomamp-cache";
#ifdef Q_WS_MAEMO_5
    ret = QDir::homePath() + QDir::separator() + "MyDocs/.tomamp-cache";
#endif
    return ret;
}

QStringList PlaylistManager::checkCache (const QString& fname)
{
    QSqlQuery q (cachedb);
    q.prepare("SELECT artist,album,title FROM metadata WHERE filename=:fname");
    q.bindValue(":fname", fname);
    q.exec ();
    if (!q.first ())
        return QStringList();
    QSqlRecord r = q.record();
    if (r.isEmpty())
        return QStringList();
    else
    {
        QStringList ret;
        ret << r.value("album").toString() << r.value("artist").toString() << r.value("title").toString();
        qDebug () << "Returned from cache for " << fname << " => " << ret;
        return ret;
    }
}

void PlaylistManager::writeCache (const QString& fname, const QString& album, const QString& artist, const QString& title)
{
    QSqlQuery q (cachedb);
    q.prepare("INSERT INTO metadata (filename,artist,album,title) VALUES (:fname,:art,:alb,:tit)");
    q.bindValue(":fname", fname);
    q.bindValue(":alb", album);
    q.bindValue(":art", artist);
    q.bindValue(":tit", title);
    q.exec ();
    cachedb.commit();
}
