#ifndef NODEMODEL_H
#define NODEMODEL_H

#include "nodelist.h"
#include "reply.h"
#include <QAbstractItemModel>

using namespace QtUbuntuOne;

class NodeRoles : public QObject
{
    Q_OBJECT

    Q_ENUMS(Role)

public:
    enum Role {
        NodeTypeRole = Qt::UserRole + 1,
        PathRole,
        ResourcePathRole,
        ContentPathRole,
        ParentPathRole,
        VolumePathRole,
        NodePathRole,
        NameRole,
        SuffixRole,
        KeyRole,
        WhenCreatedRole,
        LastModifiedRole,
        GenerationRole,
        GenerationCreatedRole,
        HashRole,
        SizeRole,
        IsPublicRole,
        PublicUrlRole,
        HasChildrenRole,
        IsRootRole
    };
};

class NodeModel : public QAbstractItemModel
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ count
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)

public:
    ~NodeModel();

    static NodeModel* instance();

#if QT_VERSION >= 0x050000
    QHash<int QByteArray> roleNames() const;
#endif
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    int columnCount(const QModelIndex &parent = QModelIndex()) const;
    int count() const;
    
    Q_INVOKABLE QVariant data(const QModelIndex &index, int role) const;

    Q_INVOKABLE bool setData(const QModelIndex &index, const QVariant &value, int role);

    Q_INVOKABLE QMap<int, QVariant> itemData(const QModelIndex &index) const;

    QModelIndex index(int row, int column, const QModelIndex &parent = QModelIndex()) const;
    QModelIndex parent(const QModelIndex &child) const;

    QModelIndex activeIndex() const;

    Q_INVOKABLE Node* get(const QModelIndex &index) const;

    bool loading() const;

public slots:
    void listDirectory(const QModelIndex &parent = QModelIndex());
    void makeDirectory(const QModelIndex &parent, const QString &name);

    void renameNode(const QModelIndex &index, const QString &name);
    void moveNode(const QModelIndex &index, const QModelIndex &destination);
    void deleteNode(const QModelIndex &index);

    void setFilePublic(const QModelIndex &index, bool isPublic);

    void setActiveIndex(const QModelIndex &index);

    void reload(const QModelIndex &parent = QModelIndex());
    void clear(const QModelIndex &parent = QModelIndex());

private:
    NodeModel();

    void setLoading(bool loading);

    QModelIndex destinationIndex() const;
    void setDestinationIndex(const QModelIndex &index);

    QModelIndex actionIndex() const;
    void setActionIndex(const QModelIndex &index);

    void insertNode(int pos, Node *node, const QModelIndex &parent);
    void insertNodeAlphabetically(Node *node, const QModelIndex &parent);
    void appendNode(Node *node, const QModelIndex &parent);
    void removeNode(int pos, const QModelIndex &parent);
    void removeNodes(int pos, int count, const QModelIndex &parent);

private slots:
    void addNodes(NodeList *nodes);

    void onNodeCreated(Node *node);
    void onNodeDeleted(Reply *reply);
    void onNodeDataChanged(Node *node);

signals:
    void countChanged();
    void loadingChanged(bool loading);

private:
    static NodeModel *m_instance;

    Node *m_rootNode;

    QModelIndex m_activeIndex;
    QModelIndex m_destinationIndex;
    QModelIndex m_actionIndex;

    bool m_loading;

    QHash<int, QByteArray> m_roleNames;
};

#endif // NODEMODEL_H
