#include "transferworker.h"
#include "baseurls.h"
#include "utils.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QRegExp>
#include <QDir>

#define ILLEGAL_CHARS "[\"@&~=\\/:?#!|<>*^]"

using namespace QtJson;

TransferWorker::TransferWorker(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_ubuntu(0),
    m_busy(false),
    m_dreply(0),
    m_ureply(0)
{
}

void TransferWorker::downloadFile(QSharedPointer<TransferItem> transfer) {
    m_transfer = transfer;
    m_transferTime.start();
    setTransferInProgress(true);
    performDownload(QUrl(CONTENT_ROOT_FILES + m_transfer.data()->contentPath().toUtf8().toPercentEncoding(":/~_?=")));

    emit transferStarted(m_transfer);
}

void TransferWorker::performDownload(const QUrl &url) {
    QString filePath(m_transfer.data()->downloadPath());

    if (!filePath.endsWith('/')) {
        filePath.append('/');
    }

    QDir dir;
    dir.mkpath(filePath);
    m_downloadFile.setFileName(filePath + QString(m_transfer.data()->fileName()).replace(QRegExp(ILLEGAL_CHARS), "_") + ".partial");

    if (m_downloadFile.exists()) {
        if (!m_downloadFile.open(QIODevice::Append)) {
            setTransferInProgress(false);
            emit transferFailed(m_transfer, tr("Cannot write to file"));
            return;
        }
    }
    else if (!m_downloadFile.open(QIODevice::WriteOnly)) {
        setTransferInProgress(false);
        emit transferFailed(m_transfer, tr("Cannot write to file"));
        return;
    }

    m_transfer.data()->setFilePath(m_downloadFile.fileName());

    QNetworkRequest request(url);
    request.setRawHeader("Authorization", m_ubuntu->getOAuthHeader("GET", CONTENT_ROOT_FILES + m_transfer.data()->contentPath().toUtf8().toPercentEncoding(":/~_?="), QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());

    if (m_downloadFile.size() > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(m_downloadFile.size()) + "-"); // Set 'Range' header if resuming a download
    }

    m_dreply = networkAccessManager()->get(request);
    connect(m_dreply, SIGNAL(downloadProgress(qint64, qint64)), this, SLOT(updateDownloadProgress(qint64,qint64)));
    connect(m_dreply, SIGNAL(finished()), this, SLOT(downloadFinished()));
    connect(m_dreply, SIGNAL(readyRead()), this, SLOT(downloadReadyRead()));
}

void TransferWorker::pauseDownload() {
    m_dreply->abort();
}

void TransferWorker::cancelDownload() {
    if (transferInProgress()) {
        m_dreply->abort();
    }

    m_downloadFile.close();
    m_downloadFile.remove();
    emit transferCancelled(m_transfer);
}

void TransferWorker::deleteIncompleteDownload(const QString &filePath) {
    if (!filePath.isEmpty()) {
        QFile::remove(filePath);
    }
}

void TransferWorker::updateSize() {
    qint64 bytes = m_dreply->header(QNetworkRequest::ContentLengthHeader).toLongLong();

    if (bytes > 0) {
        m_transfer.data()->setSize(m_downloadFile.size() + bytes);
    }
}

void TransferWorker::updateDownloadProgress(qint64 bytesReceived, qint64 bytesTotal) {
    Q_UNUSED(bytesTotal); // Ubuntu One server reports incorrect content-length header.

    if (bytesReceived) {
        float progress = float (bytesReceived) / (m_transfer.data()->size());
        qint64 elapsed = m_transferTime.elapsed();
        int eta = int ((elapsed / progress) - elapsed);

        emit progressChanged(progress, eta);
    }
}

void TransferWorker::downloadFinished() {
    m_downloadFile.close();
    setTransferInProgress(false);
    QString redirect = m_dreply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (!redirect.isEmpty()) {
        performDownload(redirect); // Follow redirect :P
    }
    else {
        if (m_dreply->error()) {
            if (m_dreply->error() == QNetworkReply::OperationCanceledError) {
                emit transferPaused(m_transfer);
            }
            else {
                QString statusText = m_dreply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                emit transferFailed(m_transfer, statusText);
            }
        }
        else {
            QString fileName = m_downloadFile.fileName().left(m_downloadFile.fileName().lastIndexOf("."));
            int num = 1;
            bool fileSaved = m_downloadFile.rename(fileName);

            while ((!fileSaved) && (num < 10)) {
                if (num == 1) {
                    fileName = fileName.insert(fileName.lastIndexOf("."), "(" + QByteArray::number(num) + ")");
                }
                else {
                    fileName = fileName.replace(fileName.lastIndexOf("(" + QByteArray::number(num - 1) + ")"), 3, "(" + QByteArray::number(num) + ")");
                }
                fileSaved = m_downloadFile.rename(fileName);
                num++;
            }

            emit transferCompleted(m_transfer);
        }
    }

    m_dreply->deleteLater();
}

void TransferWorker::downloadReadyRead() {
    m_downloadFile.write(m_dreply->readAll());
}

void TransferWorker::uploadFile(QSharedPointer<TransferItem> transfer) {
    m_transfer = transfer;
    m_uploadFile.setFileName(m_transfer.data()->filePath());

    if (!m_uploadFile.exists()) {
        transferFailed(m_transfer, tr("File cannot be found"));
        return;
    }

    m_uploadFile.open(QIODevice::ReadOnly);
    qint64 size = m_uploadFile.size();
    m_transfer.data()->setSize(size);
    QUrl url(CONTENT_ROOT_FILES + m_transfer.data()->contentPath().toUtf8().toPercentEncoding(":/~_?=") + '/' + m_transfer.data()->fileName());
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    request.setHeader(QNetworkRequest::ContentLengthHeader, size);
    request.setRawHeader("Authorization", m_ubuntu->getOAuthHeader("PUT", CONTENT_ROOT_FILES + m_transfer.data()->contentPath().toUtf8().toPercentEncoding(":/~_?=") + '/' + m_transfer.data()->fileName(), QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    m_transferTime.start();
    m_ureply = networkAccessManager()->put(request, &m_uploadFile);
    connect(m_ureply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(updateUploadProgress(qint64,qint64)));
    connect(m_ureply, SIGNAL(finished()), this, SLOT(uploadFinished()));

    m_transferTime.start();
    setTransferInProgress(true);
    emit transferStarted(m_transfer);
}

void TransferWorker::updateUploadProgress(qint64 bytesSent, qint64 bytesTotal) {
    if (bytesSent) {
        float progress = float (bytesSent) / bytesTotal;
        qint64 elapsed = m_transferTime.elapsed();
        int eta = int ((elapsed / progress) - elapsed);
        emit progressChanged(progress, eta);
    }
}

void TransferWorker::cancelUpload() {
    m_ureply->abort();
}

void TransferWorker::uploadFinished() {
    m_uploadFile.close();
    setTransferInProgress(false);

    if (m_ureply->error()) {
        if (m_ureply->error() == QNetworkReply::OperationCanceledError) {
            emit transferCancelled(m_transfer);
        }
        else {
            QString statusText = m_ureply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit transferFailed(m_transfer, statusText);
        }
    }
    else {
        QString response(m_ureply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            if (m_transfer.data()->publish()) {
                m_ubuntu->setPublished(result.value("resource_path").toString(), true);
                connect(m_ubuntu, SIGNAL(publishedChanged(QVariantMap)), this, SLOT(uploadPublished(QVariantMap)));
            }
            else {
                emit fileCreated(result);
            }
        }

        emit transferCompleted(m_transfer);
    }    

    m_ureply->deleteLater();
}

void TransferWorker::uploadPublished(const QVariantMap &file) {
    emit fileCreated(file);
    disconnect(m_ubuntu, SIGNAL(publishedChanged(QVariantMap)), this, SLOT(uploadPublished(QVariantMap)));
}
