#include "transferlistmodel.h"

TransferListModel::TransferListModel(QObject *parent) :
    QAbstractListModel(parent),
    m_worker(0),
    m_transferStatus(TransferItem::Queued),
    m_transferFailed(false),
    m_viewVisible(false)
{
}

TransferListModel::~TransferListModel() {
    clearTransfers();
}

void TransferListModel::setWorker(TransferWorker *aworker) {
    m_worker = aworker;
    connect(worker(), SIGNAL(transferCompleted(QSharedPointer<TransferItem>)), this, SLOT(onTransferCompleted(QSharedPointer<TransferItem>)));
    connect(worker(), SIGNAL(transferFailed(QSharedPointer<TransferItem>,QString)), this, SLOT(onTransferFailed(QSharedPointer<TransferItem>,QString)));
    connect(worker(), SIGNAL(transferPaused(QSharedPointer<TransferItem>)), this, SLOT(onTransferPaused(QSharedPointer<TransferItem>)));
    connect(worker(), SIGNAL(transferStarted(QSharedPointer<TransferItem>)), this, SLOT(onTransferStarted(QSharedPointer<TransferItem>)));
    connect(worker(), SIGNAL(transferCancelled(QSharedPointer<TransferItem>)), this, SLOT(onTransferCancelled(QSharedPointer<TransferItem>)));
    connect(worker(), SIGNAL(sizeChanged(qint64)), this, SLOT(onSizeChanged(qint64)));
}

void TransferListModel::setViewVisible(bool visible) {
    m_viewVisible = visible;

    if (m_viewVisible) {
        connect(worker(), SIGNAL(progressChanged(float,int)), this, SLOT(onProgressChanged(float,int)));
    }
    else {
        disconnect(worker(), SIGNAL(progressChanged(float,int)), this, SLOT(onProgressChanged(float,int)));
    }
}

int TransferListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent);

    return m_list.size();
}

QVariant TransferListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case FileNameRole:
        return m_list.at(index.row()).data()->fileName();
    case TransferTypeRole:
        return m_list.at(index.row()).data()->transferType();
    case StatusRole:
        return m_list.at(index.row()).data()->status();
    case StatusTextRole:
        return m_list.at(index.row()).data()->statusText();
    case StatusInfoRole:
        return m_list.at(index.row()).data()->statusInfo();
    case EtaRole:
        return m_list.at(index.row()).data()->eta();
    case ProgressRole:
        return m_list.at(index.row()).data()->progress();
    case SizeRole:
        return m_list.at(index.row()).data()->size();
    case PriorityRole:
        return m_list.at(index.row()).data()->priority();
    case PriorityTextRole:
        return m_list.at(index.row()).data()->priorityText();
    default:
        return QVariant();
    }
}

QSharedPointer<TransferItem> TransferListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<TransferItem>(m_list.at(row));
    }

    return QSharedPointer<TransferItem>();
}

void TransferListModel::insertTransfer(int row, QSharedPointer<TransferItem> transfer) {
    beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, transfer);
    endInsertRows();

    if (transfer.data()->transferType() != TransferItem::Upload) {
        emit transferAdded(transfer);
    }

    emit countChanged(rowCount());
}

void TransferListModel::appendTransfer(QSharedPointer<TransferItem> transfer) {
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    m_list.append(transfer);
    endInsertRows();

    if (transfer.data()->transferType() != TransferItem::Upload) {
        emit transferAdded(transfer);
    }

    emit countChanged(rowCount());
}

void TransferListModel::addDownloadTransfer(FileItem *file, const QString &downloadPath) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(file->key(), TransferItem::Download, m_transferStatus));
    transfer.data()->setContentPath(file->contentPath());
    transfer.data()->setFileName(file->fileName());
    transfer.data()->setDownloadPath(downloadPath);
    transfer.data()->setSize(file->size());
    appendTransfer(transfer);
    emit alert(tr("File added to transfer queue"));

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }

    emit countChanged(rowCount());
}

void TransferListModel::addDownloadTransfers(QList<FileItem*> files, const QString &downloadPath) {
    while (!files.isEmpty()) {
        FileItem *file = files.takeFirst();
        QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(file->key(), TransferItem::Download, m_transferStatus));
        transfer.data()->setContentPath(file->contentPath());
        transfer.data()->setFileName(file->fileName());
        transfer.data()->setDownloadPath(downloadPath);
        transfer.data()->setSize(file->size());
        appendTransfer(transfer);
    }

    emit alert(tr("File(s) added to transfer queue"));

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }
}

void TransferListModel::addUploadTransfer(const QString &filePath, const QString &contentPath, bool publish) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem("", TransferItem::Upload, m_transferStatus));
    transfer.data()->setFileName(filePath.section('/', -1));
    transfer.data()->setFilePath(filePath);
    transfer.data()->setContentPath(contentPath);
    transfer.data()->setPublish(publish);
    appendTransfer(transfer);
    emit alert(tr("File upload added to transfer queue"));

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }
}

void TransferListModel::addTransfers(QList< QSharedPointer<TransferItem> > transfers) {
    while (!transfers.isEmpty()) {
        QSharedPointer<TransferItem> transfer = transfers.takeFirst();
        transfer.data()->setStatus(m_transferStatus);
        appendTransfer(transfer);
    }

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }
}

void TransferListModel::pauseTransfer(int row) {
    QSharedPointer<TransferItem> transfer = get(row);

    if ((transfer == m_currentTransfer) && (worker()->transferInProgress())) {
        if (transfer.data()->transferType() == TransferItem::Upload) {
            emit alert(tr("File uploads cannot be paused once in progress"));
        }
        else {
            worker()->pauseDownload();
        }
    }
    else {
        transfer.data()->setStatus(TransferItem::Paused);
        emit dataChanged(index(row), index(row));
    }
}

void TransferListModel::resumeTransfer(int row) {
    QSharedPointer<TransferItem> transfer = get(row);

    if (transfer.data()->status() != TransferItem::Active) {
        transfer.data()->setStatus(TransferItem::Queued);
    }

    emit dataChanged(index(row), index(row));

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }
}

void TransferListModel::cancelTransfer(int row) {
    QSharedPointer<TransferItem> transfer = get(row);

    if ((transfer == m_currentTransfer) && (worker()->transferInProgress())) {
        if (transfer.data()->transferType() == TransferItem::Upload) {
            worker()->cancelUpload();
        }
        else {
            worker()->cancelDownload();
        }
    }
    else {
        onTransferCancelled(transfer);
    }
}

void TransferListModel::setTransferPriority(int row, TransferItem::Priority priority) {
    get(row).data()->setPriority(priority);
    emit dataChanged(index(row), index(row));
}

void TransferListModel::onTransferStarted(QSharedPointer<TransferItem> transfer) {
    m_currentTransfer = transfer;
    m_currentTransfer.data()->setStatus(TransferItem::Active);
    onDataChanged(transfer);
}

void TransferListModel::onTransferPaused(QSharedPointer<TransferItem> transfer) {
    transfer.data()->setStatus(TransferItem::Paused);
    onDataChanged(transfer);

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }
}

void TransferListModel::onTransferCancelled(QSharedPointer<TransferItem> transfer) {
    transfer.data()->setStatus(TransferItem::Cancelled);

    if (!transfer.data()->filePath().isEmpty()) {
        worker()->deleteIncompleteDownload(transfer.data()->filePath());
    }

    onDataChanged(transfer);
    emit transferCancelled(transfer);
}

void TransferListModel::onTransferCompleted(QSharedPointer<TransferItem> transfer) {
    transfer.data()->setStatus(TransferItem::Completed);
    onDataChanged(transfer);
    if (transfer.data()->transferType() == TransferItem::Upload) {
        emit alert(tr("Upload of") + " '" + transfer.data()->fileName() + "' " + tr("completed"));
    }
    else {
        emit alert(tr("Download of") + " '" + transfer.data()->fileName() + "' " + tr("completed"));
    }

    emit transferCompleted(transfer);
}

void TransferListModel::onTransferFailed(QSharedPointer<TransferItem> transfer, const QString &reason) {
    setTransferHasFailed(true);
    transfer.data()->setStatus(TransferItem::Failed);
    transfer.data()->setStatusInfo(reason);
    onDataChanged(transfer);

    if (!worker()->transferInProgress()) {
        getNextTransfer();
    }
}

void TransferListModel::onTransferCleared(QSharedPointer<TransferItem> transfer) {
    if ((transfer.data()->status() == TransferItem::Completed) || (transfer.data()->status() == TransferItem::Cancelled)) {
        QModelIndex index = indexFromItem(transfer);
        removeRow(index.row());

        if (!worker()->transferInProgress()) {
            getNextTransfer();
        }
    }
}

void TransferListModel::onProgressChanged(float progress, int eta) {
    if (viewVisible()) {
        m_currentTransfer.data()->setProgress(progress, eta);
        onDataChanged(m_currentTransfer);
    }
}

void TransferListModel::onSizeChanged(qint64 size) {
    m_currentTransfer.data()->setSize(size);
    onDataChanged(m_currentTransfer);
}

void TransferListModel::onDataChanged(QSharedPointer<TransferItem> transfer) {
    QModelIndex index = indexFromItem(transfer);
    emit dataChanged(index, index);
}

void TransferListModel::getNextTransfer() {
    bool found = false;
    int i = 0;
    int priority = TransferItem::HighPriority;

    while ((priority <= TransferItem::LowPriority) && (!found)) {

        while ((i < m_list.size()) && (!found)) {
            QSharedPointer<TransferItem> transfer = get(i);

            if ((transfer.data()->priority() == priority) && (transfer.data()->status() == TransferItem::Queued)) {
                found = true;

                if (transfer.data()->transferType() == TransferItem::Upload) {
                    worker()->uploadFile(transfer);
                }
                else {
                    worker()->downloadFile(transfer);
                }
            }

            i++;
        }

        priority++;
        i = 0;
    }
}

QModelIndex TransferListModel::indexFromItem(QSharedPointer<TransferItem> transfer) const
{
    for (int i = 0; i < m_list.size(); i++) {
        if (m_list.at(i) == transfer) {
            return index(i);
        }
    }

    return QModelIndex();
}

bool TransferListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent);

    if ((row >= 0) && (row < m_list.size())) {
        beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        endRemoveRows();
        emit countChanged(rowCount());

        return true;
    }

    return false;
}

void TransferListModel::clearTransfers() {
    if (worker()->transferInProgress()) {
        if (m_currentTransfer.data()->transferType() == TransferItem::Upload) {
            worker()->cancelUpload();
        }
        else {
            worker()->pauseDownload();
        }
    }

    m_list.clear();
}
