#include "session.h"
#include <QMaemo5InformationBox>
#include <QNetworkAccessManager>

Session::Session(QObject *parent) :
    QObject(parent),
    m_settings(new Settings(this)),
    m_database(new Database(this)),
    m_ubuntu(new UbuntuOne(this)),
    m_fileModel(new FileModel(this)),
    m_transferModel(new TransferListModel(this)),
    m_transferWorker(new TransferWorker(this)),
    m_nam(new QNetworkAccessManager(this)),
    m_progressDialog(0)
{
    ubuntu()->setNetworkAccessManager(networkAccessManager());
    transferWorker()->setNetworkAccessManager(networkAccessManager());
    transferWorker()->setUbuntuOneSession(ubuntu());
    transferManager()->setWorker(transferWorker());
    fileManager()->setRootNode(ubuntu()->rootNodePath());

    connect(ubuntu(), SIGNAL(signedIn(QVariantMap)), database(), SLOT(storeUbuntuOneAccount(QVariantMap)));
    connect(ubuntu(), SIGNAL(signedIn(QVariantMap)), fileManager(), SLOT(getFiles()));
    connect(ubuntu(), SIGNAL(signedOut()), fileManager(), SLOT(clear()));
    connect(ubuntu(), SIGNAL(signedOut()), database(), SLOT(removeUbuntuOneAccount()));
    connect(ubuntu(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    connect(ubuntu(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(ubuntu(), SIGNAL(warning(QString)), this, SLOT(showNoteBanner(QString)));
    connect(ubuntu(), SIGNAL(gotFiles(QVariantMap)), fileManager(), SLOT(addFiles(QVariantMap)));
    connect(ubuntu(), SIGNAL(itemDeleted()), fileManager(), SLOT(onItemDeleted()));
    connect(ubuntu(), SIGNAL(itemMoved(QVariantMap)), fileManager(), SLOT(onItemMoved(QVariantMap)));
    connect(ubuntu(), SIGNAL(itemRenamed(QVariantMap)), fileManager(), SLOT(onItemRenamed(QVariantMap)));
    connect(ubuntu(), SIGNAL(publishedChanged(QVariantMap)), fileManager(), SLOT(onItemPublishedChanged(QVariantMap)));
    connect(ubuntu(), SIGNAL(folderCreated(QVariantMap)), fileManager(), SLOT(onFolderCreated(QVariantMap)));
    connect(ubuntu(), SIGNAL(filesError()), fileManager(), SLOT(onFilesError()));
    connect(transferWorker(), SIGNAL(fileCreated(QVariantMap)), fileManager(), SLOT(onFileCreated(QVariantMap)));
    connect(fileManager(), SIGNAL(getFiles(QString)), ubuntu(), SLOT(getFiles(QString)));
    connect(fileManager(), SIGNAL(deleteItem(QString)), ubuntu(), SLOT(deleteItem(QString)));
    connect(fileManager(), SIGNAL(createFolder(QString)), ubuntu(), SLOT(createFolder(QString)));
    connect(fileManager(), SIGNAL(moveItem(QString,QString)), ubuntu(), SLOT(moveItem(QString,QString)));
    connect(fileManager(), SIGNAL(renameItem(QString,QString)), ubuntu(), SLOT(renameItem(QString,QString)));
    connect(fileManager(), SIGNAL(setItemPublished(QString,bool)), ubuntu(), SLOT(setPublished(QString,bool)));
    connect(database(), SIGNAL(credentialsChanged(QVariantMap)), ubuntu(), SLOT(setCredentials(QVariantMap)));
    connect(database(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(database(), SIGNAL(warning(QString)), this, SLOT(showNoteBanner(QString)));
    connect(fileManager(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(fileManager(), SIGNAL(error(QString)), this, SLOT(showNoteBanner(QString)));
    connect(fileManager(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    connect(settings(), SIGNAL(fileHandlersChanged(QVariantMap)), ubuntu(), SLOT(setFileHandlers(QVariantMap)));
    connect(settings(), SIGNAL(defaultTransferStatusChanged(TransferItem::Status)), transferManager(), SLOT(setDefaultTransferStatus(TransferItem::Status)));
    connect(transferManager(), SIGNAL(transferAdded(QSharedPointer<TransferItem>)), database(), SLOT(storeDownload(QSharedPointer<TransferItem>)));
    connect(transferManager(), SIGNAL(transferCompleted(QSharedPointer<TransferItem>)), database(), SLOT(removeStoredDownload(QSharedPointer<TransferItem>)));
    connect(transferManager(), SIGNAL(transferCancelled(QSharedPointer<TransferItem>)), database(), SLOT(removeStoredDownload(QSharedPointer<TransferItem>)));
    connect(transferManager(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(database(), SIGNAL(gotStoredDownloads(QList< QSharedPointer<TransferItem> >)), transferManager(), SLOT(addTransfers(QList< QSharedPointer<TransferItem> >)));
    connect(database(), SIGNAL(storedDownloadRemoved(QSharedPointer<TransferItem>)), transferManager(), SLOT(onTransferCleared(QSharedPointer<TransferItem>)));

    settings()->restoreSettings();
    database()->restoreAccounts();
    database()->getStoredDownloads();
}

Session::~Session() {}

void Session::showInfoBanner(const QString &message) {
    QMaemo5InformationBox::information(0, message);
}

void Session::showNoteBanner(const QString &message) {
    QMaemo5InformationBox::information(0, message, QMaemo5InformationBox::NoTimeout);
}

void Session::showProgressDialog(const QString &message, int numberOfOperations) {
    if (!m_progressDialog) {
        m_progressDialog = new QProgressDialog();
        m_progressDialog->setWindowTitle(tr("Please wait"));
        m_progressDialog->setCancelButtonText(QString());
        m_progressDialog->setMinimumDuration(0);
        m_progressDialog->setMinimum(0);

        connect(ubuntu(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
        connect(ubuntu(), SIGNAL(alert(QString)), m_progressDialog, SLOT(cancel()));
        connect(ubuntu(), SIGNAL(warning(QString)), m_progressDialog, SLOT(cancel()));
        connect(fileManager(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
        connect(fileManager(), SIGNAL(alert(QString)), m_progressDialog, SLOT(cancel()));
        connect(fileManager(), SIGNAL(error(QString)), m_progressDialog, SLOT(cancel()));
    }

    switch (settings()->screenOrientation()) {
    case ScreenOrientation::LockLandscape:
        m_progressDialog->setAttribute(Qt::WA_Maemo5LandscapeOrientation);
        break;
    case ScreenOrientation::LockPortrait:
        m_progressDialog->setAttribute(Qt::WA_Maemo5PortraitOrientation);
        break;
    default:
        m_progressDialog->setAttribute(Qt::WA_Maemo5AutoOrientation);
    }

    m_progressDialog->setLabelText(message);
    m_progressDialog->setMaximum(numberOfOperations);
    m_progressDialog->setValue(0);
    m_progressDialog->show();
}

void Session::toggleBusy(bool busy) const {
    if (QWidget *window = currentWindow()) {
        window->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
    }
}

QWidget* Session::currentWindow() const {
    if (!m_windowStack.isEmpty()) {
        return m_windowStack.last();
    }

    return 0;
}

void Session::addWindowToStack(QWidget *window) {
    m_windowStack.append(window);
    connect(window, SIGNAL(destroyed(QObject*)), this, SLOT(removeWindowFromStack(QObject*)));
}

void Session::removeWindowFromStack(QObject *window) {
    m_windowStack.removeOne(qobject_cast<QWidget*>(window));
}

void Session::unwindWindowStack() {
    for (int i = m_windowStack.size() - 1; i > 0; i--) {
        m_windowStack.at(i)->close();
    }
}
