#include "filedetailsdialog.h"
#include "utils.h"
#include <QDesktopServices>
#include <QUrl>

FileDetailsDialog::FileDetailsDialog(Session *session, const QModelIndex &file, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_file(file),
    m_nameEdit(new QLineEdit(this)),
    m_createdLabel(new QLabel(this)),
    m_modifiedLabel(new QLabel(this)),
    m_publicUrlLabel(new QLabel(this)),
    m_sizeLabel(new QLabel(this)),
    m_publishButton(new QPushButton(this)),
    m_openButton(new QPushButton(tr("Open"), this)),
    m_renameButton(new QPushButton(tr("Rename"), this)),
    m_deleteButton(new QPushButton(tr("Delete"), this)),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("File details"));

    m_renameButton->setMinimumWidth(150);
    m_publishButton->setMinimumWidth(150);
    m_openButton->setMinimumWidth(150);
    m_deleteButton->setMinimumWidth(150);

    connect(m_nameEdit, SIGNAL(textChanged(QString)), this, SLOT(onFileNameChanged(QString)));
    connect(m_openButton, SIGNAL(clicked()), this, SLOT(openFile()));
    connect(m_renameButton, SIGNAL(clicked()), this, SLOT(renameFile()));
    connect(m_publishButton, SIGNAL(clicked()), this, SLOT(toggleFilePublished()));
    connect(m_deleteButton, SIGNAL(clicked()), this, SLOT(deleteFile()));
    connect(m_publicUrlLabel, SIGNAL(linkActivated(QString)), this, SLOT(openLink(QString)));

    loadFileData();
    onOrientationChanged();
}

void FileDetailsDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_openButton->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_nameEdit, 0, 0);
        m_grid->addWidget(m_renameButton, 0, 1);
        m_grid->addWidget(m_publicUrlLabel, 1, 0);
        m_grid->addWidget(m_publishButton, 1, 1);
        m_grid->addWidget(m_createdLabel, 2, 0);
        m_grid->addWidget(m_modifiedLabel, 3, 0);
        m_grid->addWidget(m_sizeLabel, 4, 0);
        m_grid->addWidget(m_openButton, 3, 2);
        m_grid->addWidget(m_deleteButton, 4, 2);
    }
    else {
        m_grid->removeWidget(m_openButton);
        m_grid->removeWidget(m_deleteButton);
        m_grid->addWidget(m_openButton, 3, 2);
        m_grid->addWidget(m_deleteButton, 4, 2);
    }
}

void FileDetailsDialog::setPortraitLayout() {
    setFixedHeight(450);
    m_openButton->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_nameEdit, 0, 0);
        m_grid->addWidget(m_renameButton, 0, 1);
        m_grid->addWidget(m_publicUrlLabel, 1, 0);
        m_grid->addWidget(m_publishButton, 1, 1);
        m_grid->addWidget(m_createdLabel, 2, 0);
        m_grid->addWidget(m_modifiedLabel, 3, 0);
        m_grid->addWidget(m_sizeLabel, 4, 0);
        m_grid->addWidget(m_openButton, 5, 0);
        m_grid->addWidget(m_deleteButton, 6, 0);
    }
    else {
        m_grid->removeWidget(m_openButton);
        m_grid->removeWidget(m_deleteButton);
        m_grid->addWidget(m_openButton, 5, 0, 1, 2);
        m_grid->addWidget(m_deleteButton, 6, 0, 1, 2);
    }
}

void FileDetailsDialog::onFileNameChanged(const QString &name) {
    m_renameButton->setEnabled(!name.isEmpty());
}

void FileDetailsDialog::openLink(const QString &link) {
    if (!QDesktopServices::openUrl(QUrl(link))) {
        m_session->showNoteBanner(tr("Unable to open url"));
    }
}

void FileDetailsDialog::openFile() {
    m_session->ubuntu()->openFile(m_file.data(FileModel::ContentPathRole).toString());

    accept();
}

void FileDetailsDialog::renameFile() {
    m_session->fileManager()->renameItem(m_file, m_nameEdit->text());
    connect(m_session->fileManager(), SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SLOT(loadFileData()));
}

void FileDetailsDialog::deleteFile() {
    m_session->fileManager()->deleteItems(QModelIndexList() << m_file);

    accept();
}

void FileDetailsDialog::toggleFilePublished() {
    m_session->fileManager()->setItemPublished(m_file, !m_file.data(FileModel::PublicRole).toBool());
    connect(m_session->fileManager(), SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SLOT(loadFileData()));
}

void FileDetailsDialog::loadFileData() {
    m_nameEdit->setText(m_file.data(FileModel::FileNameRole).toString());
    m_publicUrlLabel->setText(QString("%1: %2").arg("Public url").arg(m_file.data(FileModel::PublicRole).toBool() ? QString("<a href='%1'>%1</a>").arg(m_file.data(FileModel::UrlRole).toString()) : tr("Not published")));
    m_createdLabel->setText(QString("%1: %2").arg(tr("Created")).arg(m_file.data(FileModel::CreatedRole).toString()));
    m_modifiedLabel->setText(QString("%1: %2").arg(tr("Modified")).arg(m_file.data(FileModel::ModifiedRole).toString()));
    m_sizeLabel->setText(QString("%1: %2").arg(tr("Size")).arg(Utils::fileSizeFromBytes(m_file.data(FileModel::SizeRole).toLongLong())));
    m_publishButton->setText(m_file.data(FileModel::PublicRole).toBool() ? tr("Unpublish") : tr("Publish"));
    m_renameButton->setEnabled(false);

    disconnect(m_session->fileManager(), SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SLOT(loadFileData()));
}
