#include "database.h"
#include <QSqlQuery>
#include <QSqlRecord>
#include <QDir>

Database::Database(QObject *parent) :
    QObject(parent) {

    QDir dbPath("/home/user/toBuntu/");
    if (!dbPath.exists()) {
        dbPath.mkdir(dbPath.path());
    }

    m_database = QSqlDatabase::addDatabase("QSQLITE");
    m_database.setDatabaseName("/home/user/toBuntu/toBuntu.sqlite");
    initialize();
}

void Database::initialize() {
    m_database.open();
    m_database.exec("CREATE TABLE IF NOT EXISTS ubuntuOneAccount (appName TEXT UNIQUE, consumerKey TEXT, consumerSecret TEXT, accessToken TEXT, tokenSecret TEXT)");
    m_database.exec("CREATE TABLE IF NOT EXISTS downloads (key TEXT UNIQUE, contentPath TEXT, downloadPath TEXT, size TEXT)");
    m_database.close();
}

void Database::restoreAccounts() {
    restoreUbunutOneAccount();
}

void Database::storeUbuntuOneAccount(const QVariantMap &account) {
    if ((m_database.isOpen()) || (m_database.open())) {

        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO ubuntuOneAccount VALUES (?, ?, ?, ?, ?)");
        query.addBindValue("tobuntu");
        query.addBindValue(account.value("consumerKey").toString());
        query.addBindValue(account.value("consumerSecret").toString());
        query.addBindValue(account.value("accessToken").toString());
        query.addBindValue(account.value("tokenSecret").toString());

        if (query.exec()) {
            emit credentialsChanged(account);
        }
        else {
            emit warning(tr("Database error. Unable to store account details"));
        }

        m_database.close();
    }
    else {
        emit warning(tr("Database error. Unable to store account details"));
    }
}

void Database::restoreUbunutOneAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {

        QVariantMap account;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT * FROM ubuntuOneAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                account.insert("appName", query.value(0));
                account.insert("consumerKey", query.value(1));
                account.insert("consumerSecret", query.value(2));
                account.insert("accessToken", query.value(3));
                account.insert("tokenSecret", query.value(4));
            }
        }

        emit credentialsChanged(account);
        m_database.close();

    }
    else {
        emit warning(tr("Database error. Unable to retrieve account details"));
    }
}

void Database::removeUbuntuOneAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM ubuntuOneAccount WHERE appName = ?");
        query.addBindValue("tobuntu");
        bool success = query.exec();
        m_database.close();

        if (!success) {
            emit warning(tr("Database error. Unable to remove Ubuntu One account"));
        }
    }
    else {
        emit warning(tr("Database error. Unable to remove Ubuntu One account"));
    }
}

bool Database::storeDownload(QSharedPointer<TransferItem> transfer) {
    if ((m_database.isOpen()) || (m_database.open())) {
        bool success = false;
        QSqlQuery query;
        query.prepare("INSERT OR IGNORE INTO downloads VALUES (?, ?, ?, ?)");
        query.addBindValue(transfer.data()->key());
        query.addBindValue(transfer.data()->contentPath());
        query.addBindValue(transfer.data()->filePath());
        query.addBindValue(transfer.data()->size());
        success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

void Database::removeStoredDownload(QSharedPointer<TransferItem> transfer) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM downloads WHERE key = ?");
        query.addBindValue(transfer.data()->key());
        bool success = query.exec();
        m_database.close();

        if (success) {
            emit storedDownloadRemoved(transfer);
        }
        else {
            emit warning(tr("Database error. Unable to remove stored download"));
        }
    }
    else {
        emit warning(tr("Database error. Unable to remove stored download"));
    }
}

void Database::getStoredDownloads() {
    if ((m_database.isOpen()) || (m_database.open())) {

        QSqlQuery query;
        QSqlRecord record;
        QList< QSharedPointer<TransferItem> > transfers;
        query.exec("SELECT * FROM downloads");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                QSharedPointer<TransferItem> transfer(new TransferItem);
                transfer.data()->setKey(query.value(0).toString());
                transfer.data()->setContentPath(query.value(1).toString());
                transfer.data()->setFileName(query.value(1).toString().section('/', -1));
                transfer.data()->setDownloadPath(query.value(2).toString());
                transfer.data()->setSize(query.value(3).toLongLong());
                transfers.append(transfer);
            }
        }

        m_database.close();

        if (!transfers.isEmpty()) {
            emit gotStoredDownloads(transfers);
        }
    }
    else {
        emit warning(tr("Database error. Unable to retrieve stored downloads"));
    }
}
