#include "settingsdialog.h"
#include "separatorlabel.h"
#include "filehandlersdialog.h"
#include <QLabel>
#include <QFileDialog>

SettingsDialog::SettingsDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_emailEdit(!m_session->ubuntu()->userSignedIn() ? new QLineEdit(this) : 0),
    m_passEdit(!m_session->ubuntu()->userSignedIn() ? new QLineEdit(this) : 0),
    m_accountButton(new QPushButton(m_session->ubuntu()->userSignedIn() ? tr("Remove") : tr("Add"), this)),
    m_transferStatusCheckbox(new QCheckBox(tr("Start transfers immediately"), this)),
    m_downloadCheckbox(new QCheckBox(tr("Use default download folder"), this)),
    m_downloadPathButton(new QMaemo5ValueButton(tr("Download folder"), this)),
    m_handlersButton(new QPushButton(tr("File handlers"), this)),
    m_orientationSelector(new ScreenOrientationSelector(tr("Screen orientation"), this)),
    m_languageSelector(new LanguageSelector(tr("Language"), true, this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Save"), QDialogButtonBox::ActionRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Settings"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    QLabel *accountLabel = new QLabel(m_session->ubuntu()->userSignedIn() ? tr("You have added your Ubuntu One account") : tr("Add your Ubuntu One account"), this);

    vbox->addWidget(new SeparatorLabel(tr("Ubuntu One account"), this));
    vbox->addWidget(accountLabel, 0, Qt::AlignHCenter);

    if (!m_session->ubuntu()->userSignedIn()) {
        vbox->addWidget(new QLabel(tr("Email address"), this));
        vbox->addWidget(m_emailEdit);
        vbox->addWidget(new QLabel(tr("Password"), this));
        vbox->addWidget(m_passEdit);
    }

    vbox->addWidget(m_accountButton, 0, Qt::AlignHCenter);
    vbox->addWidget(new SeparatorLabel(tr("Transfers"), this));
    vbox->addWidget(m_transferStatusCheckbox);
    vbox->addWidget(m_downloadCheckbox);
    vbox->addWidget(m_downloadPathButton);
    vbox->addWidget(new SeparatorLabel(tr("Other"), this));
    vbox->addWidget(m_handlersButton);
    vbox->addWidget(m_languageSelector);
    vbox->addWidget(m_orientationSelector);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

    connect(m_doneButton, SIGNAL(clicked()), this, SLOT(saveSettings()));
    connect(m_downloadCheckbox, SIGNAL(toggled(bool)), m_downloadPathButton, SLOT(setEnabled(bool)));
    connect(m_downloadPathButton, SIGNAL(clicked()), this, SLOT(showFileDialog()));
    connect(m_handlersButton, SIGNAL(clicked()), this, SLOT(showFileHandlersDialog()));

    if (m_session->ubuntu()->userSignedIn()) {
        connect(m_accountButton, SIGNAL(clicked()), m_session->ubuntu(), SLOT(signOut()));
        connect(m_accountButton, SIGNAL(clicked()), this, SLOT(accept()));
    }
    else {
        connect(m_emailEdit, SIGNAL(textChanged(QString)), this, SLOT(onEmailChanged(QString)));
        connect(m_passEdit, SIGNAL(textChanged(QString)), this, SLOT(onPasswordChanged(QString)));
        connect(m_accountButton, SIGNAL(clicked()), this, SLOT(signIn()));
    }

    loadSettings();
    onOrientationChanged();
}

void SettingsDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void SettingsDialog::setPortraitLayout() {
    setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
}

void SettingsDialog::onEmailChanged(const QString &email) {
    m_accountButton->setEnabled((!email.isEmpty()) && (!m_passEdit->text().isEmpty()));
}

void SettingsDialog::onPasswordChanged(const QString &pass) {
    m_accountButton->setEnabled((!pass.isEmpty()) && (!m_emailEdit->text().isEmpty()));
}

void SettingsDialog::signIn() {
    m_session->ubuntu()->signIn(m_emailEdit->text(), m_passEdit->text());

    accept();
}

void SettingsDialog::showFileDialog() {
    QString path = QFileDialog::getExistingDirectory(this, tr("Download folder"), m_downloadPath);

    if (!path.isEmpty()) {
        m_downloadPath = path;
        m_downloadPathButton->setValueText(m_downloadPath);
    }
}

void SettingsDialog::showFileHandlersDialog() {
    FileHandlersDialog *dialog = new FileHandlersDialog(m_session, this);
    dialog->open();
}

void SettingsDialog::loadSettings() {
    m_accountButton->setEnabled(m_session->ubuntu()->userSignedIn());
    m_transferStatusCheckbox->setChecked(m_session->settings()->defaultTransferStatus() == TransferItem::Queued);
    m_downloadCheckbox->setChecked(m_session->settings()->useDefaultDownloadPath());
    m_downloadPath = m_session->settings()->downloadPath();
    m_downloadPathButton->setValueText(m_downloadPath);
    m_downloadPathButton->setEnabled(m_downloadCheckbox->isChecked());
    m_languageSelector->setValue(m_session->settings()->language());
    m_orientationSelector->setValue(m_session->settings()->screenOrientation());
}

void SettingsDialog::saveSettings() {
    m_session->settings()->setDefaultTransferStatus(m_transferStatusCheckbox->isChecked() ? TransferItem::Queued : TransferItem::Paused);
    m_session->settings()->setUseDefaultDownloadPath(m_downloadCheckbox->isChecked());
    m_session->settings()->setDownloadPath(m_downloadPath);
    m_session->settings()->setLanguage(m_languageSelector->currentValue());
    m_session->settings()->setScreenOrientation(m_orientationSelector->currentValue());
    m_session->settings()->saveSettings();

    accept();
}
