#include "fileswindow.h"
#include "filelistdelegate.h"
#include "aboutdialog.h"
#include "settingsdialog.h"
#include "filenamedialog.h"
#include "querydialog.h"
#include "filedetailsdialog.h"
#include "fileselectionwindow.h"
#include "folderselectiondialog.h"
#include "transferlistwindow.h"
#include "uploaddialog.h"
#include "folderlistdialog.h"
#include <QMenuBar>
#include <QFileDialog>

FilesWindow::FilesWindow(QWidget *parent) :
    StackedWindow(parent),
    m_listView(new ListView(this)),
    m_uploadAction(menuBar()->addAction(tr("Upload file"), this, SLOT(showUploadDialog()))),
    m_refreshAction(menuBar()->addAction(tr("Refresh folder"), this, SLOT(refreshFolder()))),
    m_newFolderAction(menuBar()->addAction(tr("New folder"), this, SLOT(showNewFolderDialog()))),
    m_moveAction(menuBar()->addAction(tr("Move files"), this, SLOT(moveFiles()))),
    m_deleteAction(menuBar()->addAction(tr("Delete files"), this, SLOT(deleteFiles()))),
    m_downloadAction(menuBar()->addAction(tr("Download files"), this, SLOT(downloadFiles()))),
    m_transfersAction(menuBar()->addAction(QString("%1 (%2)").arg(tr("Transfers")).arg(m_session->transferManager()->rowCount()), this, SLOT(showTransfers()))),
    m_settingsAction(menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()))),
    m_aboutAction(menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()))),
    m_contextMenu(new QMenu(this)),
    m_openContextAction(m_contextMenu->addAction(tr("Open"), this, SLOT(openFile()))),
    m_moveContextAction(m_contextMenu->addAction(tr("Move"), this, SLOT(showMoveFileDialog()))),
    m_renameContextAction(m_contextMenu->addAction(tr("Rename"), this, SLOT(showRenameDialog()))),
    m_deleteContextAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(showConfirmDeleteDialog()))),
    m_downloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(downloadFile()))),
    m_publishContextAction(m_contextMenu->addAction(tr("Publish"), this, SLOT(toggleFilePublished()))),
    m_upButton(new QPushButton(QIcon::fromTheme("filemanager_folder_up"), "", this)),
    m_homeButton(new QPushButton(QIcon::fromTheme("general_presence_home"), "", this)),
    m_folderButton(new QMaemo5ValueButton(this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("Folder empty")), m_listView)),
    m_grid(0)
{
    setWindowTitle(tr("toBuntu"));
    setAttribute(Qt::WA_DeleteOnClose, false);
    setCentralWidget(new QWidget(this));

    m_listView->setModel(m_session->fileManager());
    m_listView->setItemDelegate(new FileListDelegate(m_listView));

    m_upButton->setFixedWidth(80);
    m_homeButton->setFixedWidth(80);

    m_grid = new QGridLayout(centralWidget());
    m_grid->setContentsMargins(11, 11, 11, 11);
    m_grid->addWidget(m_upButton, 0, 0);
    m_grid->addWidget(m_homeButton, 0, 1);
    m_grid->addWidget(m_folderButton, 0, 2);
    m_grid->addWidget(m_listView, 1, 0, 1, 3);

    m_noResultsLabel->hide();

    connect(m_session->fileManager(), SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    connect(m_session->fileManager(), SIGNAL(loadingChanged(bool)), this, SLOT(toggleBusy(bool)));
    connect(m_session->fileManager(), SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    connect(m_session->transferManager(), SIGNAL(countChanged(int)), this, SLOT(onTransferCountChanged(int)));
    connect(m_listView, SIGNAL(rootIndexChanged(QModelIndex)), this, SLOT(onRootIndexChanged(QModelIndex)));
    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_upButton, SIGNAL(clicked()), this, SLOT(moveUp()));
    connect(m_homeButton, SIGNAL(clicked()), this, SLOT(goHome()));
    connect(m_folderButton, SIGNAL(clicked()), this, SLOT(showFolderList()));

    onOrientationChanged();
    onRootIndexChanged(QModelIndex());

    if (m_session->ubuntu()->userSignedIn()) {
        m_session->fileManager()->getFiles();
    }
}

void FilesWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(780, 300);
}

void FilesWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(460, 600);
}

void FilesWindow::keyPressEvent(QKeyEvent *event) {
    if (event->modifiers() == Qt::ControlModifier) {
        switch (event->key()) {
        case Qt::Key_N:
            event->accept();
            showNewFolderDialog();
            break;
        case Qt::Key_R:
            event->accept();
            refreshFolder();
            break;
        default:
            event->ignore();
            break;
        }
    }
    else {
        event->ignore();
    }
}

void FilesWindow::onLoadingChanged(bool loading) {
    m_noResultsLabel->setVisible((!loading) && (m_session->fileManager()->rowCount(m_session->fileManager()->activeIndex()) == 0));
    m_upButton->setEnabled((!loading) && (m_listView->rootIndex().isValid()));
    m_homeButton->setEnabled(!loading);
    m_folderButton->setEnabled(!loading);
}

void FilesWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}

void FilesWindow::onRootIndexChanged(const QModelIndex &index) {
    if (index.isValid()) {
        m_upButton->setEnabled(true);
        m_folderButton->setText(index.data(FileModel::PathRole).toString().section('/', -1));
        m_folderButton->setValueText(index.data(FileModel::ResourcePathRole).toString().section('/', 0, -2));
    }
    else {
        m_upButton->setEnabled(false);
        m_folderButton->setText(m_session->ubuntu()->rootNodePath());
        m_folderButton->setValueText("");
    }

    m_noResultsLabel->setVisible((!m_session->fileManager()->loading()) && (m_session->fileManager()->rowCount(m_listView->rootIndex()) == 0));
}

void FilesWindow::onItemClicked(const QModelIndex &index) {
    if (index.data(FileModel::FolderRole).toBool()) {
        m_listView->setRootIndex(index);

        if ((index.data(FileModel::ChildrenRole).toBool()) && (m_session->fileManager()->rowCount(index) == 0)) {
            m_session->fileManager()->getFiles(index);
        }
    }
    else {
        FileDetailsDialog *dialog = new FileDetailsDialog(m_session, index, this);
        dialog->open();
    }
}

void FilesWindow::showContextMenu(const QPoint &pos) {
    if (m_listView->currentIndex().isValid()) {
        bool isFolder = m_listView->currentIndex().data(FileModel::FolderRole).toBool();
        m_publishContextAction->setVisible(!isFolder);
        m_downloadContextAction->setVisible(!isFolder);

        if (!isFolder) {
            bool published = m_listView->currentIndex().data(FileModel::PublicRole).toBool();
            m_publishContextAction->setText(published ? tr("Unpublish") : tr("Publish"));
        }

        m_contextMenu->popup(pos, m_openContextAction);
    }
}

void FilesWindow::moveUp() {
    m_listView->setRootIndex(m_listView->rootIndex().parent());
}

void FilesWindow::goHome() {
    m_listView->setRootIndex(QModelIndex());
}

void FilesWindow::showFolderList() {
    FolderListDialog *dialog = new FolderListDialog(m_session, m_listView->rootIndex(), this);
    dialog->open();

    connect(dialog, SIGNAL(folderChosen(QString)), this, SLOT(setRootIndexFromResourcePath(QString)));
}

void FilesWindow::setRootIndexFromResourcePath(const QString &resourcePath) {
    QModelIndex index = m_listView->rootIndex();

    while ((index.isValid()) && (index.data(FileModel::ResourcePathRole).toString() != resourcePath)) {
        index = index.parent();
    }

    m_listView->setRootIndex(index);
}

void FilesWindow::moveFiles() {
    FileSelectionWindow *window = new FileSelectionWindow(m_session, tr("Select files to move"), tr("Move"), SLOT(showMoveItemsDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void FilesWindow::deleteFiles() {
    FileSelectionWindow *window = new FileSelectionWindow(m_session, tr("Select files to delete"), tr("Delete"), SLOT(showConfirmDeleteDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void FilesWindow::downloadFiles() {
    FileSelectionWindow *window = new FileSelectionWindow(m_session, tr("Select files to download"), tr("Download"), SLOT(downloadItems()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void FilesWindow::refreshFolder() {
    m_session->fileManager()->reload(m_listView->rootIndex());
}

void FilesWindow::showNewFolderDialog() {
    FileNameDialog *dialog = new FileNameDialog(tr("Folder name"), "", this);
    dialog->setWindowTitle(tr("New folder"));
    dialog->open();

    connect(dialog, SIGNAL(fileNameChosen(QString)), this, SLOT(createNewFolder(QString)));
}

void FilesWindow::createNewFolder(const QString &name) {
    m_session->fileManager()->createFolder(m_listView->rootIndex(), name);
}

void FilesWindow::showUploadDialog() {
    UploadDialog *dialog = new UploadDialog(m_session, this);
    dialog->open();
}

void FilesWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(m_session, this);
    dialog->open();
}

void FilesWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}

void FilesWindow::openFile() {
    if (m_listView->currentIndex().data(FileModel::FolderRole).toBool()) {
        onItemClicked(m_listView->currentIndex());
    }
    else {
        m_session->ubuntu()->openFile(m_listView->currentIndex().data(FileModel::ContentPathRole).toString());
    }
}

void FilesWindow::showMoveFileDialog() {
    FolderSelectionDialog *dialog = new FolderSelectionDialog(m_session, this);
    dialog->setRootIndex(m_listView->rootIndex());
    dialog->open();

    connect(dialog, SIGNAL(folderChosen(QModelIndex)), this, SLOT(moveFile(QModelIndex)));
}

void FilesWindow::moveFile(const QModelIndex &destination) {
    m_session->fileManager()->moveItems(QModelIndexList() << m_listView->currentIndex(), destination);
}

void FilesWindow::showRenameDialog() {
    FileNameDialog *dialog = new FileNameDialog(tr("New name"), m_listView->currentIndex().data(FileModel::FileNameRole).toString(), this);
    dialog->setWindowTitle(tr("Rename"));
    dialog->open();

    connect(dialog, SIGNAL(fileNameChosen(QString)), this, SLOT(renameFile(QString)));
}

void FilesWindow::renameFile(const QString &name) {
    m_session->fileManager()->renameItem(m_listView->currentIndex(), name);
}

void FilesWindow::showConfirmDeleteDialog() {
    QString filename(m_listView->currentIndex().data(FileModel::FileNameRole).toString());
    QueryDialog *dialog = new QueryDialog(QString("%1 '%2'?").arg(tr("Do you wish to delete")).arg(filename), this);
    dialog->setWindowTitle(tr("Delete?"));
    dialog->open();

    connect(dialog, SIGNAL(accepted()), this, SLOT(deleteFile()));
}

void FilesWindow::deleteFile() {
    m_session->fileManager()->deleteItems(QModelIndexList() << m_listView->currentIndex());
}

void FilesWindow::downloadFile() {
    QString downloadPath;

    if (m_session->settings()->useDefaultDownloadPath()) {
        downloadPath = m_session->settings()->downloadPath();
    }
    else {
        downloadPath = QFileDialog::getExistingDirectory(this, tr("Download folder"), m_session->settings()->downloadPath());
    }

    if (!downloadPath.isEmpty()) {
        m_session->transferManager()->addDownloadTransfer(m_session->fileManager()->get(m_listView->currentIndex()), downloadPath);
    }
}

void FilesWindow::toggleFilePublished() {
    bool published = m_listView->currentIndex().data(FileModel::PublicRole).toBool();
    m_session->fileManager()->setItemPublished(m_listView->currentIndex(), !published);
}

void FilesWindow::showTransfers() {
    TransferListWindow *window = new TransferListWindow(m_session, this);
    window->show();
}

void FilesWindow::onTransferCountChanged(int count) {
    m_transfersAction->setText(QString("%1 (%2)").arg(tr("Transfers")).arg(count));
}
