#include "fileselectionwindow.h"
#include "filelistdelegate.h"
#include "querydialog.h"
#include "folderselectiondialog.h"
#include <QFileDialog>

FileSelectionWindow::FileSelectionWindow(Session *session, const QString &title, const QString &buttonText, const char* slot, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_editBar(new QMaemo5EditBar(title, this)),
    m_acceptButton(new QPushButton(buttonText, this)),
    m_vbox(0)
{
    setWindowState(Qt::WindowFullScreen);
    setCentralWidget(new QWidget(this));

    m_listView->setModel(m_session->fileManager());
    m_listView->setItemDelegate(new FileListDelegate(m_listView));

    m_editBar->addButton(m_acceptButton);

    m_vbox = new QVBoxLayout(centralWidget());
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->setSpacing(0);
    m_vbox->addWidget(m_editBar);
    m_vbox->addWidget(m_listView);

    connect(m_listView, SIGNAL(clicked(QModelIndex)), m_session->fileManager(), SLOT(toggleChecked(QModelIndex)));
    connect(m_acceptButton, SIGNAL(clicked()), this, SLOT(onButtonClicked()));
    connect(this, SIGNAL(accepted()), this, slot);
}

FileSelectionWindow::~FileSelectionWindow() {
    m_session->fileManager()->checkNone();
}

void FileSelectionWindow::positionAtIndex(const QModelIndex &index) {
    m_listView->setRootIndex(index.parent());
    hideFolders(index.parent());
    m_listView->scrollTo(index, QListView::PositionAtTop);
}

void FileSelectionWindow::onButtonClicked() {
    if (m_session->fileManager()->indexesChecked()) {
        emit accepted();
    }
}

void FileSelectionWindow::showMoveItemsDialog() {
    FolderSelectionDialog *dialog = new FolderSelectionDialog(m_session, this);
    dialog->setRootIndex(m_listView->rootIndex());
    dialog->open();

    connect(dialog, SIGNAL(folderChosen(QModelIndex)), this, SLOT(moveItems(QModelIndex)));
}

void FileSelectionWindow::moveItems(const QModelIndex &destination) {
    m_session->fileManager()->moveItems(m_session->fileManager()->checkedIndexes(), destination);
    m_session->fileManager()->checkNone();

    close();
}

void FileSelectionWindow::deleteItems() {
    m_session->fileManager()->deleteItems(m_session->fileManager()->checkedIndexes());
    m_session->fileManager()->checkNone();

    close();
}

void FileSelectionWindow::downloadItems() {
    QString downloadPath;

    if (m_session->settings()->useDefaultDownloadPath()) {
        downloadPath = m_session->settings()->downloadPath();
    }
    else {
        downloadPath = QFileDialog::getExistingDirectory(this, tr("Download folder"), m_session->settings()->downloadPath());
    }

    if (!downloadPath.isEmpty()) {
        QList<FileItem*> files;
        QModelIndexList indexes = m_session->fileManager()->checkedIndexes();

        while (!indexes.isEmpty()) {
            files.append(m_session->fileManager()->get(indexes.takeFirst()));
        }

        m_session->transferManager()->addDownloadTransfers(files, downloadPath);
    }

    m_session->fileManager()->checkNone();

    close();
}

void FileSelectionWindow::showConfirmDeleteDialog() {
    QueryDialog *dialog = new QueryDialog(tr("Do you wish to delete the selected files?"), this);
    dialog->setWindowTitle(tr("Delete?"));
    dialog->open();

    connect(dialog, SIGNAL(accepted()), this, SLOT(deleteItems()));
}

void FileSelectionWindow::hideFolders(const QModelIndex &parent) {
    for (int i = 0; i < m_session->fileManager()->rowCount(parent); i++) {
        if (m_session->fileManager()->index(i, 0, parent).data(FileModel::FolderRole).toBool()) {
            m_listView->setRowHidden(i, true);
        }
    }
}
