// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//	Theremin server

#include <ting/Ref.hpp>
#include <ting/Array.hpp>
#include <ting/Thread.hpp>
#include <ting/types.hpp>
#include <ting/utils.hpp>

#include "Therver.hpp"
#include "Cast.hpp"
#include "Client.hpp"
#include "DatabaseThread.hpp"


using namespace ting;



//override
ting::Ref<cliser::Client> Therver::CreateClientObject(){
	return ting::Ref<cliser::Client>(new ::Client());
}



//override
void Therver::OnClientConnected(ting::Ref<cliser::Client>& client){
	ASSERT(client.IsValid())
	ASSERT(client.DynamicCast<Client>().IsValid())
	Ref<Client> c = client.StaticCast<Client>();

//	TRACE(<< "!!!! CLIENT CONNECTED !!!!" << std::endl)
}



//override
void Therver::OnClientDisconnected(ting::Ref<cliser::Client>& client){
	ASSERT(client.IsValid())
	ASSERT(client.DynamicCast<Client>().IsValid())
	Ref<Client> c = client.StaticCast<Client>();
	
//	TRACE(<< "!!!! CLIENT DISCONNECTED !!!!" << std::endl)

	this->PushMessage(Ptr<Message>(
			new ClientDisconnectedMessage(this, c)
		));
}



//override
void Therver::OnDataReceivedFromClient(ting::Ref<cliser::Client>& client, ting::Array<ting::byte> d){
	ASSERT(client.IsValid())
	ASSERT(client.DynamicCast<Client>().IsValid())
	Ref<Client> c = client.StaticCast<Client>();
	
	ASSERT(d.Size() > 0)
	
//	TRACE(<< "!!!! DATA RECEIVED !!!! " << d.SizeInBytes() << " bytes, d[0] = " << u32(d[0])  << std::endl)

	this->PushMessage(Ptr<Message>(
			new NetworkDataFromClientReceivedMessage(this, c, d)
		));
}



//override
void ClientDisconnectedMessage::Handle(){
	this->c->SetNetworkState(Client::DISCONNECTED);

	this->t->MoveClientToIdleState(this->c);
}



//override
void NetworkDataFromClientReceivedMessage::Handle(){
	ASSERT(this->data.Size() > 0)
	switch(this->data[0]){
		case 0x00: //log in
			this->thr->HandleLoginFromClient(this->client, this->data);
			break;
		case 0x01: //listen next player
			this->thr->HandleListenNextCommandFromClient(this->client);
			break;
		case 0x02:
			this->thr->HandleStopListeningCommandFromClient(this->client);
			break;
		case 0x03: //playing
			this->thr->HandlePlayingDataFromClient(this->client, this->data);
			break;
		case 0x04: //register request
			this->thr->HandleRegisterFromClient(this->client, this->data);
			break;
		default:
			TRACE_AND_LOG(<< "[!WARN] NetworkDataFromClientReceivedMessage::Handle(): unknown packet ID = " << u32(this->data[0]) << std::endl)
			break;
	}
}



void Therver::HandleRegisterFromClient(ting::Ref<Client>& c, ting::Array<ting::byte> d){
	ASSERT(c)
	ASSERT(d.Size() > 0)
	ASSERT(d[0] == 0x04) //4 = register command

	//TODO: ignore if client is logged in

	ting::uint i = 1;

	if(i + 2 > d.Size()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandleRegisterFromClient(): bad formed packet 1" << std::endl)
		return;
	}
	ting::uint unLen = ting::FromNetworkFormat16(&d[i]);
	i += 2;

	if(unLen == 0){
		//empty username is not allowed
		c->SendRegisterResult(Client::REGISTERRES_SOME_ERROR);
		return;
	}

	ASSERT(unLen != 0)
	if(i + unLen > d.Size()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandleRegisterFromClient(): bad formed packet 2" << std::endl)
		return;
	}
	std::string un(reinterpret_cast<char*>(&d[i]), unLen);
	i += unLen;

	if(i + 2 > d.Size()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandleRegisterFromClient(): bad formed packet 3" << std::endl)
		return;
	}
	ting::uint pwLen = ting::FromNetworkFormat16(&d[i]);
	i += 2;

	if(pwLen == 0){
		//empty password is not allowed
		c->SendRegisterResult(Client::REGISTERRES_SOME_ERROR);
		return;
	}

	ASSERT(pwLen != 0)
	if(i + pwLen > d.Size()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandleRegisterFromClient(): bad formed packet 4" << std::endl)
		return;
	}
	std::string pw(reinterpret_cast<char*>(&d[i]), pwLen);

	ASSERT(i + pwLen == d.Size())

	//send request to DatabaseThread
	this->dbThread.PushMessage(Ptr<Message>(
			new RegisterNewUserMessage(
					&this->dbThread,
					c,
					un,
					pw
				)
		));
}



void Therver::HandleLoginFromClient(ting::Ref< ::Client>& c, ting::Array<ting::byte> p){
	ASSERT(c)
	ASSERT(p.Size() > 0)
	ASSERT(p[0] == 0)// 0 = login command

	//TODO: ignore if already logged in

	//vars to be filled
	std::string un;
	std::string pw;

	ting::uint i = 1;

	//parse username
	{
		if(i + 2 > p.Size()){
			TRACE_AND_LOG(<< "Therver::HandleLoginFromClient(): bad formed login packet !!! IGNORING !!!" << std::endl)
			return;
		}
//		TRACE(<< "unLen = " << u32(d[i]) << " " << u32(d[i + 1]) << std::endl)
		ting::uint usernameLength = ting::FromNetworkFormat16(&p[i]);
		i += 2;

		if(usernameLength == 0){
			TRACE(<< "Therver::HandleLoginFromClient(): empty login is not allowed" << std::endl)
			c->SendLoginResult(false);
			return;
		}

		ASSERT(usernameLength != 0)
		if(i + usernameLength > p.Size()){
			TRACE_AND_LOG(<< "Therver::HandleLoginFromClient(): bad formed login packet !!! IGNORING !!!" << std::endl)
			return;
		}

		ASSERT(i < p.Size())
		un = std::string(reinterpret_cast<char*>(&p[i]), usernameLength);
		i += usernameLength;
	}

	//parse password
	{
		if(i + 2 > p.Size()){
			TRACE_AND_LOG(<< "Therver::HandleLoginFromClient(): bad formed login packet !!! IGNORING !!!" << std::endl)
			return;
		}
//		TRACE(<< "pwLen = " << u32(d[i]) << " " << u32(d[i + 1]) << std::endl)
		ting::uint passwordLength = ting::FromNetworkFormat16(&p[i]);
		i += 2;

		if(passwordLength == 0){
			TRACE(<< "Therver::HandleLoginFromClient(): empty password is not allowed" << std::endl)
			c->SendLoginResult(false);
			return;
		}

		ASSERT(passwordLength != 0)
		if(i + passwordLength > p.Size()){
			TRACE_AND_LOG(<< "Therver::HandleLoginFromClient(): bad formed login packet !!! IGNORING !!!" << std::endl)
			return;
		}

		ASSERT(i < p.Size())
		pw = std::string(reinterpret_cast<char*>(&p[i]), passwordLength);
		i += passwordLength;
	}

	ASSERT(i == p.Size())

	c->username = un;
	c->password = pw;

	c->SetNetworkState(Client::LOGGING_IN);

	//send query request to DB thread
	this->dbThread.PushMessage(Ptr<Message>(
			new QueryUserPasswordMessage(
					&this->dbThread,
					c,
					c->username
				)
		));
}



void Therver::HandlePlayingDataFromClient(ting::Ref< ::Client>& c, ting::Array<ting::byte> d){
	//if client is not logged in then ignore that
	if(c->IsNotLoggedIn()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandlePlayingDataFromClient(): client is not logged in" << std::endl)
		return;
	}

	//if the client does not play and does not have its own cast
	//then create one, because we received playing data from player
	//which means it started playing
	if(!c->IsPlaying()){
		this->MoveClientToIdleState(c);

		//create new cast
		this->casts.push_back(Cast(c));

		ASSERT(this->casts.size() > 0)
		T_CastsIter castIter = (--this->casts.end());

		c->SetState(Client::PLAYING);
		c->cast = castIter;
	}

	ASSERT(c->IsPlaying())

	c->cast->PushNewMusicDataPacket(d);

	//if there are waiting listeners, move them to the cast
	this->MoveWaitingListenersToCast();
}



void Therver::HandleStopListeningCommandFromClient(ting::Ref<Client>& c){
	//if client is not logged in then ignore that
	if(c->IsNotLoggedIn()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandleStopListeningCommandFromClient(): client is not logged in" << std::endl)
		return;
	}

	if(!c->IsListening() && !c->IsWaitingForListening()){
		//already not listening
		return;
	}

	this->MoveClientToIdleState(c);
}



void Therver::HandleListenNextCommandFromClient(ting::Ref<Client>& c){
	//if client is not logged in then ignore that
	if(c->IsNotLoggedIn()){
		TRACE_AND_LOG(<< "[!WARN] Therver::HandleListenNextCommandFromClient(): client is not logged in" << std::endl)
		return;
	}

	if(!c->IsListening()){
		this->MoveClientToIdleState(c);

		c->cast = this->casts.begin();
		c->SetState(Client::LISTENING);
	}else{
		ASSERT(c->cast != this->casts.end())
		(*c->cast).RemoveListener(c);
		++c->cast;
	}

	ASSERT(c->IsListening())

	//if we reached end try to jump to the beginning
	if(c->cast == this->casts.end()){
		if(this->casts.size() > 0){
			c->cast = this->casts.begin();
			ASSERT(c->cast != this->casts.end())
		}else{
			//There is no one to listen to.
			//Move to wait for listening state.
			c->SetState(Client::WAITING_FOR_LISTENING);
			this->waitingForListeningClients.push_back(c);
			
			c->SendStopListening();//send 'stop listening' to client
			return;
		}
	}

	ASSERT(c->cast != this->casts.end())
	(*c->cast).AddListenerAndSendDataToClient(c);
}



T_CastsIter Therver::RemoveCast(T_CastsIter castIter){
	ASSERT(castIter != this->casts.end())
	Cast& cast = *castIter;

	T_CastsIter nextCastIter = castIter;
	++nextCastIter;

	if(nextCastIter == this->casts.end()){
		if(this->casts.size() > 1){
			nextCastIter = this->casts.begin();
		}
	}

	ASSERT(nextCastIter != castIter)

	if(nextCastIter != this->casts.end()){
		Cast& nextCast = *nextCastIter;

		while(Ref<Client> c = cast.PopListener()){
			ASSERT(c->IsListening())
			ASSERT(c->cast == castIter)

			c->cast = nextCastIter;
			nextCast.AddListenerAndSendDataToClient(c);
		}//~while
	}else{//there is no one to listen to
		while(Ref<Client> c = cast.PopListener()){
			ASSERT(c->IsListening())
			ASSERT(c->cast == castIter)

			//move to wait for listening state
			c->SetState(Client::WAITING_FOR_LISTENING);
			this->waitingForListeningClients.push_back(c);
			
			c->SendStopListening();//send 'stop listening' to client
		}//~while
	}

	Ref<Client> plr = cast.GetPlayer();
	ASSERT(plr)
	plr->SetState(Client::IDLE);
	plr->cast = this->casts.end();

	return this->casts.erase(castIter);
}



//override
void UserPasswordQueriedMessage::Handle(){
	ASSERT(this->c)
	
	if(this->res != SUCCESS){
		this->c->SetNetworkState(Client::CONNECTED);
		this->c->SendLoginResult(false);//send login failure
		return;
	}

	//compare passwords
	if(this->c->password.compare(this->pw) != 0){
		this->c->SetNetworkState(Client::CONNECTED);
		this->c->SendLoginResult(false);//send login failure
		return;
	}

	this->c->SetNetworkState(Client::LOGGED_IN);
	this->c->SendLoginResult(true);//send login success
}



void Therver::PushRegisterResultMessage(Client::ERegisterRes result, ting::Ref<Client> c){
	this->PushMessage(Ptr<Message>(
			new NewUserRegisteredMessage(this, c, result)
		));
}



//override
void NewUserRegisteredMessage::Handle(){
	this->c->SendRegisterResult(this->res);
}



//override
u32 Therver::KillIdleCastsTimer::OnExpire(){
	this->t->PushMessage(Ptr<Message>(
			new KillIdleCastsMessage(this->t)
		));
	return 0;//do not reschedule timer
}



void Therver::KillIdleCasts(){
//	TRACE(<< "Therver::KillIdleCasts(): invoked" << std::endl)
	u32 curTicks = ting::GetTicks();

	for(T_CastsIter i = this->casts.begin(); i != this->casts.end(); ){
		if(curTicks - (*i).GetLastChunkArrivalTimestamp() >= Therver::DIdleCastDuration()){
			i = this->RemoveCast(i);
			TRACE(<< "cast killed" << std::endl)
		}else{
			++i;
		}
	}

	this->killIdleCastsTimer.StartTimer();//schedule timer again
}



void Therver::MoveClientToIdleState(ting::Ref<Client>& c){
	switch(c->GetState()){
		case Client::IDLE:
			//do nothing
			return;
			break;
		case Client::PLAYING:
			//if client has a playing cast then notify listeners that the cast has finished
			//and remove the cast
			this->RemoveCast(c->cast);
			break;
		case Client::LISTENING:
			//remove the client from cast it listening to
			ASSERT(c->cast != this->casts.end())
			(*c->cast).RemoveListener(c);
			c->cast = this->casts.end();
			break;
		case Client::WAITING_FOR_LISTENING:
			this->RemoveClientFromWaitingListenersList(c);
			break;
		default:
			ASSERT(false)
			break;
	}

	c->SetState(Client::IDLE);
}



void Therver::MoveWaitingListenersToCast(){
	ASSERT(this->casts.size() > 0)
	while(this->waitingForListeningClients.size() > 0){
		Ref<Client> c = this->waitingForListeningClients.back();
		this->waitingForListeningClients.pop_back();
		ASSERT(c->IsWaitingForListening())

		//add to cast
		ASSERT(this->casts.size() > 0)
		c->cast = this->casts.begin();
		c->SetState(Client::LISTENING);
		(*c->cast).AddListenerAndSendDataToClient(c);
	}
}



void Therver::RemoveClientFromWaitingListenersList(ting::Ref<Client>& c){
	for(T_ClientsIter i = this->waitingForListeningClients.begin();
			i != this->waitingForListeningClients.end();
			++i
		)
	{
		if(*i == c){
			this->waitingForListeningClients.erase(i);
			return;
		}
	}
	ASSERT(false)
}

