// (c) Ivan Gagis
// e-mail: igagis@gmail.com
// Version: 1

// Description:
//	SQLite wrapper

#pragma once

#include <string>

#include <sqlite3.h>

#include <ting/debug.hpp>
#include <ting/types.hpp>
#include <ting/Exc.hpp>
#include <ting/Array.hpp>



namespace sqlite{



//forward declarations
class Database;



class Privates{
	friend class Database;
	friend class Cursor;
	
	class Statement{
	public:
		sqlite3* db;
		sqlite3_stmt *stmt;

		Statement() :
				db(0),
				stmt(0)
		{}

		Statement(sqlite3* database, const std::string& sqlStatement);
		~Statement();

		//true if done
		bool Step();

		void Destroy();
	};

	Privates(){}
};



class SQLExc : public ting::Exc{
public:
	SQLExc(const std::string& msg) :
			ting::Exc((std::string("SQLExc: ") + msg).c_str())
	{}
};



class Cursor{
	friend class Database;
	
	Database* db;

	Privates::Statement statement;

	ting::uint numColumns;
	
	Cursor(Database* database, const std::string& sqlQuery);

public:
	Cursor() :
			db(0)
	{}

	Cursor(const Cursor& s){
		this->operator=(s);
	}

	Cursor& operator=(const Cursor& s){
		this->Destroy();
		this->statement.Destroy();

		this->db = s.db;
		this->statement.db = s.statement.db;
		this->statement.stmt = s.statement.stmt;
		this->numColumns = s.numColumns;
		const_cast<Cursor&>(s).db = 0;
		const_cast<Cursor&>(s).statement.db = 0;
		const_cast<Cursor&>(s).statement.stmt = 0;
		
		return *this;
	}

	~Cursor();

	void Destroy();

	bool MoveToNext();


	
	int GetInt(ting::uint column);

	int GetInt(const std::string& column){
		return this->GetInt(
				this->FindColumnByName(column)
			);
	}



	ting::s64 GetInt64(ting::uint column);

	ting::s64 GetInt64(const std::string& column){
		return this->GetInt64(
				this->FindColumnByName(column)
			);
	}



	double GetDouble(ting::uint column);

	inline double GetDouble(const std::string& column){
		return this->GetDouble(
				this->FindColumnByName(column)
			);
	}



	inline float GetFloat(ting::uint column){
		return float(this->GetDouble(column));
	}

	inline float GetFloat(const std::string& column){
		return float(this->GetDouble(column));
	}



	std::string GetText(ting::uint column);

	inline std::string GetText(const std::string& column){
		return this->GetText(
				this->FindColumnByName(column)
			);
	}



	ting::Array<ting::byte> GetBlob(ting::uint column);

	ting::Array<ting::byte> GetBlob(const std::string& column){
		return this->GetBlob(
				this->FindColumnByName(column)
			);
	}

private:
	ting::uint FindColumnByName(const std::string& column);
};//~class Cursor



class Database{
	friend class Cursor;
	friend class Privates::Statement;
	
	sqlite3* db;

	ting::uint numCursors;

	//forbid copying
	Database(const Database& d){
		ASSERT(false)
	}

	Database& operator=(const Database& d){
		ASSERT(false)
		return *this;
	}
	
public:
	Database(const std::string& dbFilename);

	~Database();

	void CreateTable(const std::string& name, const std::string& columns);

	void Insert(
			const std::string& table,
			const std::string& columns,
			const std::string& values
		);

	void Update(
			const std::string& table,
			const std::string& what,
			const std::string& inRowsWhere
		);

	void Delete(
			const std::string& fromTable,
			const std::string& rowsWhere
		);

	Cursor Query(
			const std::string& table,
			const std::string& columns,
			const std::string& where
		);



private:
};//~class Database







}//~namespace
